"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerVolumeConsistency = exports.BundlingDockerImage = void 0;
const child_process_1 = require("child_process");
/**
 * A Docker image used for asset bundling
 */
class BundlingDockerImage {
    /** @param image The Docker image */
    constructor(image) {
        this.image = image;
    }
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name
     */
    static fromRegistry(image) {
        return new BundlingDockerImage(image);
    }
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file
     * @param options Docker build options
     */
    static fromAsset(path, options = {}) {
        const buildArgs = options.buildArgs || {};
        const dockerArgs = [
            'build', '-q',
            ...flatten(Object.entries(buildArgs).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            path,
        ];
        const docker = dockerExec(dockerArgs);
        const match = docker.stdout.toString().match(/sha256:([a-z0-9]+)/);
        if (!match) {
            throw new Error('Failed to extract image ID from Docker build output');
        }
        return new BundlingDockerImage(match[1]);
    }
    /**
     * Runs a Docker image
     *
     * @internal
     */
    _run(options = {}) {
        const volumes = options.volumes || [];
        const environment = options.environment || {};
        const command = options.command || [];
        const dockerArgs = [
            'run', '--rm',
            ...options.user
                ? ['-u', options.user]
                : [],
            ...flatten(volumes.map(v => { var _a; return ['-v', `${v.hostPath}:${v.containerPath}:${(_a = v.consistency) !== null && _a !== void 0 ? _a : DockerVolumeConsistency.DELEGATED}`]; })),
            ...flatten(Object.entries(environment).map(([k, v]) => ['--env', `${k}=${v}`])),
            ...options.workingDirectory
                ? ['-w', options.workingDirectory]
                : [],
            this.image,
            ...command,
        ];
        dockerExec(dockerArgs);
    }
}
exports.BundlingDockerImage = BundlingDockerImage;
/**
 * Supported Docker volume consistency types. Only valid on macOS due to the way file storage works on Mac
 */
var DockerVolumeConsistency;
(function (DockerVolumeConsistency) {
    /**
     * Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes
     */
    DockerVolumeConsistency["CONSISTENT"] = "consistent";
    /**
     * Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine
     */
    DockerVolumeConsistency["DELEGATED"] = "delegated";
    /**
     * Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container
     */
    DockerVolumeConsistency["CACHED"] = "cached";
})(DockerVolumeConsistency = exports.DockerVolumeConsistency || (exports.DockerVolumeConsistency = {}));
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
function dockerExec(args) {
    var _a, _b, _c;
    const prog = (_a = process.env.CDK_DOCKER) !== null && _a !== void 0 ? _a : 'docker';
    const proc = child_process_1.spawnSync(prog, args);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        throw new Error(`[Status ${proc.status}] stdout: ${(_b = proc.stdout) === null || _b === void 0 ? void 0 : _b.toString().trim()}\n\n\nstderr: ${(_c = proc.stderr) === null || _c === void 0 ? void 0 : _c.toString().trim()}`);
    }
    return proc;
}
//# sourceMappingURL=data:application/json;base64,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