"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.outputs = {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        // ToDo implement preserveLogicalIds=false
        this.preserveLogicalIds = true;
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.createCondition(conditionName);
        }
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            // render all sections of the template unchanged,
            // except Conditions, Resources, Parameters, and Outputs which will be taken care of by the created L1s
            if (section !== 'Conditions' && section !== 'Resources' && section !== 'Parameters' && section !== 'Outputs') {
                ret[section] = this.template[section];
            }
        }
        return ret;
    }
    createParameter(logicalId) {
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
            },
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        cfnParameter.overrideLogicalId(logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition() {
                    return undefined;
                },
            },
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name\
 '${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        cfnOutput.overrideLogicalId(logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    createCondition(conditionName) {
        // ToDo condition expressions can refer to other conditions -
        // will be important when implementing preserveLogicalIds=false
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Condition definitions is not allowed'); },
                // ToDo handle one Condition referencing another using the { Condition: "ConditionName" } syntax
                findCondition() { return undefined; },
            },
        }).parseValue(this.template.Conditions[conditionName]);
        const cfnCondition = new core.CfnCondition(this, conditionName, {
            expression,
        });
        // ToDo handle renaming of the logical IDs of the conditions
        cfnCondition.overrideLogicalId(conditionName);
        this.conditions[conditionName] = cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
        if (!l1ClassFqn) {
            // currently, we only handle types we know the L1 for -
            // in the future, we might construct an instance of CfnResource instead
            throw new Error(`Unrecognized CloudFormation resource type: '${resourceAttributes.Type}'`);
        }
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The ${attribute} resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const [moduleName, ...className] = l1ClassFqn.split('.');
        const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
        const jsClassFromModule = module[className.join('.')];
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const options = {
            finder,
        };
        const l1Instance = jsClassFromModule.fromCloudFormation(this, logicalId, resourceAttributes, options);
        if (this.preserveLogicalIds) {
            // override the logical ID to match the original template
            l1Instance.overrideLogicalId(logicalId);
        }
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluY2x1ZGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5jbHVkZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxtQ0FBbUMsQ0FBQyxnREFBZ0Q7QUFDcEYsc0RBQXNELENBQUMsOERBQThEO0FBQ3JILG1FQUFtRTtBQUNuRSx1Q0FBdUM7QUFZdkM7Ozs7R0FJRztBQUNILE1BQWEsVUFBVyxTQUFRLElBQUksQ0FBQyxVQUFVO0lBZTNDLFlBQVksS0FBcUIsRUFBRSxFQUFVLEVBQUUsS0FBc0I7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQWZKLGVBQVUsR0FFdkIsRUFBRSxDQUFDO1FBQ1UsY0FBUyxHQUV0QixFQUFFLENBQUM7UUFDVSxlQUFVLEdBRXZCLEVBQUUsQ0FBQztRQUNVLFlBQU8sR0FFcEIsRUFBRSxDQUFDO1FBS0gscUNBQXFDO1FBQ3JDLElBQUksQ0FBQyxRQUFRLEdBQUcsTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDeEQsMENBQTBDO1FBQzFDLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7UUFDL0IsNkJBQTZCO1FBQzdCLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNqRSxJQUFJLENBQUMsZUFBZSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ25DO1FBQ0QsNkJBQTZCO1FBQzdCLEtBQUssTUFBTSxhQUFhLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNyRSxJQUFJLENBQUMsZUFBZSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQ3ZDO1FBQ0QsOENBQThDO1FBQzlDLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNoRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDdkM7UUFDRCxNQUFNLFdBQVcsR0FBRyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3hELEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUMsRUFBRTtZQUM5RCxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQztTQUM3QztJQUNMLENBQUM7SUFDRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxXQUFXLENBQUMsU0FBaUI7UUFDaEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsU0FBUyxpQ0FBaUMsQ0FBQyxDQUFDO1NBQzVGO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ0Q7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWSxDQUFDLGFBQXFCO1FBQ3JDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLGFBQWEsaUNBQWlDLENBQUMsQ0FBQztTQUMzRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztJQUNEOzs7Ozs7Ozs7T0FTRztJQUNJLFlBQVksQ0FBQyxhQUFxQjtRQUNyQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDTixNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixhQUFhLGlDQUFpQyxDQUFDLENBQUM7U0FDM0Y7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSSxTQUFTLENBQUMsU0FBaUI7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsU0FBUyxpQ0FBaUMsQ0FBQyxDQUFDO1NBQzFGO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ0QsZ0JBQWdCO0lBQ1QsaUJBQWlCO1FBQ3BCLE1BQU0sR0FBRyxHQUVMLEVBQUUsQ0FBQztRQUNQLEtBQUssTUFBTSxPQUFPLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDOUMsaURBQWlEO1lBQ2pELHVHQUF1RztZQUN2RyxJQUFJLE9BQU8sS0FBSyxZQUFZLElBQUksT0FBTyxLQUFLLFdBQVcsSUFBSSxPQUFPLEtBQUssWUFBWSxJQUFJLE9BQU8sS0FBSyxTQUFTLEVBQUU7Z0JBQzFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQ3pDO1NBQ0o7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDTyxlQUFlLENBQUMsU0FBaUI7UUFDckMsTUFBTSxVQUFVLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLE1BQU0sRUFBRTtnQkFDSixZQUFZLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdkcsYUFBYSxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3JHLGFBQWEsS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQzFHO1NBQ0osQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ25ELE1BQU0sWUFBWSxHQUFHLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3hELElBQUksRUFBRSxVQUFVLENBQUMsSUFBSTtZQUNyQixPQUFPLEVBQUUsVUFBVSxDQUFDLE9BQU87WUFDM0IsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjO1lBQ3pDLHFCQUFxQixFQUFFLFVBQVUsQ0FBQyxxQkFBcUI7WUFDdkQsV0FBVyxFQUFFLFVBQVUsQ0FBQyxXQUFXO1lBQ25DLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUztZQUMvQixRQUFRLEVBQUUsVUFBVSxDQUFDLFFBQVE7WUFDN0IsU0FBUyxFQUFFLFVBQVUsQ0FBQyxTQUFTO1lBQy9CLFFBQVEsRUFBRSxVQUFVLENBQUMsUUFBUTtZQUM3QixNQUFNLEVBQUUsVUFBVSxDQUFDLE1BQU07U0FDNUIsQ0FBQyxDQUFDO1FBQ0gsWUFBWSxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLEdBQUcsWUFBWSxDQUFDO0lBQzlDLENBQUM7SUFDTyxZQUFZLENBQUMsU0FBaUIsRUFBRSxLQUFxQjtRQUN6RCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUM7WUFDN0MsTUFBTSxFQUFFO2dCQUNKLFlBQVksQ0FBQyxHQUFHO29CQUNaLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDL0IsQ0FBQztnQkFDRCxhQUFhLENBQUMsV0FBbUI7O29CQUM3QixhQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLG1DQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3ZFLENBQUM7Z0JBQ0QsYUFBYTtvQkFDVCxPQUFPLFNBQVMsQ0FBQztnQkFDckIsQ0FBQzthQUNKO1NBQ0osQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ25ELEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLO1lBQzdCLFdBQVcsRUFBRSxnQkFBZ0IsQ0FBQyxXQUFXO1lBQ3pDLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDOUUsU0FBUyxFQUFFLENBQUMsR0FBRyxFQUFFO2dCQUNiLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUU7b0JBQzdCLE9BQU8sU0FBUyxDQUFDO2lCQUNwQjtxQkFDSSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLEVBQUU7b0JBQ2xELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztpQkFDeEQ7Z0JBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsU0FBUztJQUMxRCxnQkFBZ0IsQ0FBQyxTQUFTLHdDQUF3QyxDQUFDLENBQUM7WUFDNUQsQ0FBQyxDQUFDLEVBQUU7U0FDUCxDQUFDLENBQUM7UUFDSCxTQUFTLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLENBQUM7SUFDeEMsQ0FBQztJQUNPLGVBQWUsQ0FBQyxhQUFxQjtRQUN6Qyw2REFBNkQ7UUFDN0QsK0RBQStEO1FBQy9ELE1BQU0sVUFBVSxHQUFHLElBQUksU0FBUyxDQUFDLFNBQVMsQ0FBQztZQUN2QyxNQUFNLEVBQUU7Z0JBQ0osWUFBWSxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQzNGLGFBQWEsS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLCtEQUErRCxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNyRyxnR0FBZ0c7Z0JBQ2hHLGFBQWEsS0FBSyxPQUFPLFNBQVMsQ0FBQyxDQUFDLENBQUM7YUFDeEM7U0FDSixDQUFDLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7UUFDdkQsTUFBTSxZQUFZLEdBQUcsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDNUQsVUFBVTtTQUNiLENBQUMsQ0FBQztRQUNILDREQUE0RDtRQUM1RCxZQUFZLENBQUMsaUJBQWlCLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsR0FBRyxZQUFZLENBQUM7SUFDbEQsQ0FBQztJQUNPLG1CQUFtQixDQUFDLFNBQWlCO1FBQ3pDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEMsSUFBSSxHQUFHLEVBQUU7WUFDTCxPQUFPLEdBQUcsQ0FBQztTQUNkO1FBQ0QsTUFBTSxrQkFBa0IsR0FBUSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNuRSxNQUFNLFVBQVUsR0FBRyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUUsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNiLHVEQUF1RDtZQUN2RCx1RUFBdUU7WUFDdkUsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0Msa0JBQWtCLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQztTQUM5RjtRQUNELDBEQUEwRDtRQUMxRCxNQUFNLGVBQWUsR0FBRztZQUNwQixNQUFNLEVBQUUsWUFBWSxFQUFFLFdBQVcsRUFBRSxXQUFXLEVBQUUsVUFBVTtZQUMxRCxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCO1NBQzVFLENBQUM7UUFDRixLQUFLLE1BQU0sU0FBUyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsRUFBRTtZQUNyRCxJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsRUFBRTtnQkFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLFNBQVMsc0VBQXNFO29CQUNsRyxnR0FBZ0csQ0FBQyxDQUFDO2FBQ3pHO1NBQ0o7UUFDRCxNQUFNLENBQUMsVUFBVSxFQUFFLEdBQUcsU0FBUyxDQUFDLEdBQUcsVUFBVSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN6RCxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyw0REFBNEQ7UUFDaEcsTUFBTSxpQkFBaUIsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixNQUFNLE1BQU0sR0FBb0I7WUFDNUIsYUFBYSxDQUFDLGFBQXFCO2dCQUMvQixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDMUMsQ0FBQztZQUNELFlBQVksQ0FBQyxHQUFXO2dCQUNwQixJQUFJLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFO29CQUMzQyxPQUFPLFNBQVMsQ0FBQztpQkFDcEI7Z0JBQ0QsT0FBTyxJQUFJLENBQUMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDekMsQ0FBQztZQUNELGFBQWEsQ0FBQyxXQUFtQjtnQkFDN0IsSUFBSSxXQUFXLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtvQkFDaEMsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2lCQUN2QztnQkFDRCxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDMUMsQ0FBQztTQUNKLENBQUM7UUFDRixNQUFNLE9BQU8sR0FBbUM7WUFDNUMsTUFBTTtTQUNULENBQUM7UUFDRixNQUFNLFVBQVUsR0FBRyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3RHLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQ3pCLHlEQUF5RDtZQUN6RCxVQUFVLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDM0M7UUFDRCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxHQUFHLFVBQVUsQ0FBQztRQUN2QyxPQUFPLFVBQVUsQ0FBQztJQUN0QixDQUFDO0NBQ0o7QUEvUEQsZ0NBK1BDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY29yZSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCAqIGFzIGNmbl9wYXJzZSBmcm9tIFwiLi4vLi4vY29yZS9saWIvY2ZuLXBhcnNlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlL2xpYi9jZm4tcGFyc2UnXG5pbXBvcnQgKiBhcyBjZm5fdHlwZV90b19sMV9tYXBwaW5nIGZyb20gJy4vY2ZuLXR5cGUtdG8tbDEtbWFwcGluZyc7XG5pbXBvcnQgKiBhcyBmdXRpbHMgZnJvbSAnLi9maWxlLXV0aWxzJztcbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgb2Yge0BsaW5rIENmbkluY2x1ZGV9LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENmbkluY2x1ZGVQcm9wcyB7XG4gICAgLyoqXG4gICAgICogUGF0aCB0byB0aGUgdGVtcGxhdGUgZmlsZS5cbiAgICAgKlxuICAgICAqIEJvdGggSlNPTiBhbmQgWUFNTCB0ZW1wbGF0ZSBmb3JtYXRzIGFyZSBzdXBwb3J0ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGVtcGxhdGVGaWxlOiBzdHJpbmc7XG59XG4vKipcbiAqIENvbnN0cnVjdCB0byBpbXBvcnQgYW4gZXhpc3RpbmcgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgZmlsZSBpbnRvIGEgQ0RLIGFwcGxpY2F0aW9uLlxuICogQWxsIHJlc291cmNlcyBkZWZpbmVkIGluIHRoZSB0ZW1wbGF0ZSBmaWxlIGNhbiBiZSByZXRyaWV2ZWQgYnkgY2FsbGluZyB0aGUge0BsaW5rIGdldFJlc291cmNlfSBtZXRob2QuXG4gKiBBbnkgbW9kaWZpY2F0aW9ucyBtYWRlIG9uIHRoZSByZXR1cm5lZCByZXNvdXJjZSBvYmplY3RzIHdpbGwgYmUgcmVmbGVjdGVkIGluIHRoZSByZXN1bHRpbmcgQ0RLIHRlbXBsYXRlLlxuICovXG5leHBvcnQgY2xhc3MgQ2ZuSW5jbHVkZSBleHRlbmRzIGNvcmUuQ2ZuRWxlbWVudCB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBjb25kaXRpb25zOiB7XG4gICAgICAgIFtjb25kaXRpb25OYW1lOiBzdHJpbmddOiBjb3JlLkNmbkNvbmRpdGlvbjtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZXM6IHtcbiAgICAgICAgW2xvZ2ljYWxJZDogc3RyaW5nXTogY29yZS5DZm5SZXNvdXJjZTtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIFtsb2dpY2FsSWQ6IHN0cmluZ106IGNvcmUuQ2ZuUGFyYW1ldGVyO1xuICAgIH0gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IG91dHB1dHM6IHtcbiAgICAgICAgW2xvZ2ljYWxJZDogc3RyaW5nXTogY29yZS5DZm5PdXRwdXQ7XG4gICAgfSA9IHt9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgdGVtcGxhdGU6IGFueTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHByZXNlcnZlTG9naWNhbElkczogYm9vbGVhbjtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY29yZS5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZm5JbmNsdWRlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgLy8gcmVhZCB0aGUgdGVtcGxhdGUgaW50byBhIEpTIG9iamVjdFxuICAgICAgICB0aGlzLnRlbXBsYXRlID0gZnV0aWxzLnJlYWRZYW1sU3luYyhwcm9wcy50ZW1wbGF0ZUZpbGUpO1xuICAgICAgICAvLyBUb0RvIGltcGxlbWVudCBwcmVzZXJ2ZUxvZ2ljYWxJZHM9ZmFsc2VcbiAgICAgICAgdGhpcy5wcmVzZXJ2ZUxvZ2ljYWxJZHMgPSB0cnVlO1xuICAgICAgICAvLyBpbnN0YW50aWF0ZSBhbGwgcGFyYW1ldGVyc1xuICAgICAgICBmb3IgKGNvbnN0IGxvZ2ljYWxJZCBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLlBhcmFtZXRlcnMgfHwge30pKSB7XG4gICAgICAgICAgICB0aGlzLmNyZWF0ZVBhcmFtZXRlcihsb2dpY2FsSWQpO1xuICAgICAgICB9XG4gICAgICAgIC8vIGluc3RhbnRpYXRlIHRoZSBjb25kaXRpb25zXG4gICAgICAgIGZvciAoY29uc3QgY29uZGl0aW9uTmFtZSBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLkNvbmRpdGlvbnMgfHwge30pKSB7XG4gICAgICAgICAgICB0aGlzLmNyZWF0ZUNvbmRpdGlvbihjb25kaXRpb25OYW1lKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBpbnN0YW50aWF0ZSBhbGwgcmVzb3VyY2VzIGFzIENESyBMMSBvYmplY3RzXG4gICAgICAgIGZvciAoY29uc3QgbG9naWNhbElkIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuUmVzb3VyY2VzIHx8IHt9KSkge1xuICAgICAgICAgICAgdGhpcy5nZXRPckNyZWF0ZVJlc291cmNlKGxvZ2ljYWxJZCk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3Qgb3V0cHV0U2NvcGUgPSBuZXcgY29yZS5Db25zdHJ1Y3QodGhpcywgJyRPdXB1dHMnKTtcbiAgICAgICAgZm9yIChjb25zdCBsb2dpY2FsSWQgb2YgT2JqZWN0LmtleXModGhpcy50ZW1wbGF0ZS5PdXRwdXRzIHx8IHt9KSkge1xuICAgICAgICAgICAgdGhpcy5jcmVhdGVPdXRwdXQobG9naWNhbElkLCBvdXRwdXRTY29wZSk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyB0aGUgbG93LWxldmVsIENmblJlc291cmNlIGZyb20gdGhlIHRlbXBsYXRlIHdpdGggdGhlIGdpdmVuIGxvZ2ljYWwgSUQuXG4gICAgICogQW55IG1vZGlmaWNhdGlvbnMgcGVyZm9ybWVkIG9uIHRoYXQgcmVzb3VyY2Ugd2lsbCBiZSByZWZsZWN0ZWQgaW4gdGhlIHJlc3VsdGluZyBDREsgdGVtcGxhdGUuXG4gICAgICpcbiAgICAgKiBUaGUgcmV0dXJuZWQgb2JqZWN0IHdpbGwgYmUgb2YgdGhlIHByb3BlciB1bmRlcmx5aW5nIGNsYXNzO1xuICAgICAqIHlvdSBjYW4gYWx3YXlzIGNhc3QgaXQgdG8gdGhlIGNvcnJlY3QgdHlwZSBpbiB5b3VyIGNvZGU6XG4gICAgICpcbiAgICAgKiAgICAgLy8gYXNzdW1lIHRoZSB0ZW1wbGF0ZSBjb250YWlucyBhbiBBV1M6OlMzOjpCdWNrZXQgd2l0aCBsb2dpY2FsIElEICdCdWNrZXQnXG4gICAgICogICAgIGNvbnN0IGNmbkJ1Y2tldCA9IGNmblRlbXBsYXRlLmdldFJlc291cmNlKCdCdWNrZXQnKSBhcyBzMy5DZm5CdWNrZXQ7XG4gICAgICogICAgIC8vIGNmbkJ1Y2tldCBpcyBvZiB0eXBlIHMzLkNmbkJ1Y2tldFxuICAgICAqXG4gICAgICogSWYgdGhlIHRlbXBsYXRlIGRvZXMgbm90IGNvbnRhaW4gYSByZXNvdXJjZSB3aXRoIHRoZSBnaXZlbiBsb2dpY2FsIElELFxuICAgICAqIGFuIGV4Y2VwdGlvbiB3aWxsIGJlIHRocm93bi5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBsb2dpY2FsSWQgdGhlIGxvZ2ljYWwgSUQgb2YgdGhlIHJlc291cmNlIGluIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBmaWxlXG4gICAgICovXG4gICAgcHVibGljIGdldFJlc291cmNlKGxvZ2ljYWxJZDogc3RyaW5nKTogY29yZS5DZm5SZXNvdXJjZSB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMucmVzb3VyY2VzW2xvZ2ljYWxJZF07XG4gICAgICAgIGlmICghcmV0KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFJlc291cmNlIHdpdGggbG9naWNhbCBJRCAnJHtsb2dpY2FsSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybnMgdGhlIENmbkNvbmRpdGlvbiBvYmplY3QgZnJvbSB0aGUgJ0NvbmRpdGlvbnMnXG4gICAgICogc2VjdGlvbiBvZiB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgd2l0aCB0aGUgZ2l2ZW4gbmFtZS5cbiAgICAgKiBBbnkgbW9kaWZpY2F0aW9ucyBwZXJmb3JtZWQgb24gdGhhdCBvYmplY3Qgd2lsbCBiZSByZWZsZWN0ZWQgaW4gdGhlIHJlc3VsdGluZyBDREsgdGVtcGxhdGUuXG4gICAgICpcbiAgICAgKiBJZiBhIENvbmRpdGlvbiB3aXRoIHRoZSBnaXZlbiBuYW1lIGlzIG5vdCBwcmVzZW50IGluIHRoZSB0ZW1wbGF0ZSxcbiAgICAgKiB0aHJvd3MgYW4gZXhjZXB0aW9uLlxuICAgICAqXG4gICAgICogQHBhcmFtIGNvbmRpdGlvbk5hbWUgdGhlIG5hbWUgb2YgdGhlIENvbmRpdGlvbiBpbiB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgZmlsZVxuICAgICAqL1xuICAgIHB1YmxpYyBnZXRDb25kaXRpb24oY29uZGl0aW9uTmFtZTogc3RyaW5nKTogY29yZS5DZm5Db25kaXRpb24ge1xuICAgICAgICBjb25zdCByZXQgPSB0aGlzLmNvbmRpdGlvbnNbY29uZGl0aW9uTmFtZV07XG4gICAgICAgIGlmICghcmV0KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENvbmRpdGlvbiB3aXRoIG5hbWUgJyR7Y29uZGl0aW9uTmFtZX0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyB0aGUgQ2ZuUGFyYW1ldGVyIG9iamVjdCBmcm9tIHRoZSAnUGFyYW1ldGVycydcbiAgICAgKiBzZWN0aW9uIG9mIHRoZSBpbmNsdWRlZCB0ZW1wbGF0ZVxuICAgICAqIEFueSBtb2RpZmljYXRpb25zIHBlcmZvcm1lZCBvbiB0aGF0IG9iamVjdCB3aWxsIGJlIHJlZmxlY3RlZCBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZS5cbiAgICAgKlxuICAgICAqIElmIGEgUGFyYW1ldGVyIHdpdGggdGhlIGdpdmVuIG5hbWUgaXMgbm90IHByZXNlbnQgaW4gdGhlIHRlbXBsYXRlLFxuICAgICAqIHRocm93cyBhbiBleGNlcHRpb24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gcGFyYW1ldGVyTmFtZSB0aGUgbmFtZSBvZiB0aGUgcGFyYW1ldGVyIHRvIHJldHJpZXZlXG4gICAgICovXG4gICAgcHVibGljIGdldFBhcmFtZXRlcihwYXJhbWV0ZXJOYW1lOiBzdHJpbmcpOiBjb3JlLkNmblBhcmFtZXRlciB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMucGFyYW1ldGVyc1twYXJhbWV0ZXJOYW1lXTtcbiAgICAgICAgaWYgKCFyZXQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgUGFyYW1ldGVyIHdpdGggbmFtZSAnJHtwYXJhbWV0ZXJOYW1lfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHRoZSBDZm5PdXRwdXQgb2JqZWN0IGZyb20gdGhlICdPdXRwdXRzJ1xuICAgICAqIHNlY3Rpb24gb2YgdGhlIGluY2x1ZGVkIHRlbXBsYXRlXG4gICAgICogQW55IG1vZGlmaWNhdGlvbnMgcGVyZm9ybWVkIG9uIHRoYXQgb2JqZWN0IHdpbGwgYmUgcmVmbGVjdGVkIGluIHRoZSByZXN1bHRpbmcgQ0RLIHRlbXBsYXRlLlxuICAgICAqXG4gICAgICogSWYgYW4gT3V0cHV0IHdpdGggdGhlIGdpdmVuIG5hbWUgaXMgbm90IHByZXNlbnQgaW4gdGhlIHRlbXBsYXRlLFxuICAgICAqIHRocm93cyBhbiBleGNlcHRpb24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gbG9naWNhbElkIHRoZSBuYW1lIG9mIHRoZSBvdXRwdXQgdG8gcmV0cmlldmVcbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0T3V0cHV0KGxvZ2ljYWxJZDogc3RyaW5nKTogY29yZS5DZm5PdXRwdXQge1xuICAgICAgICBjb25zdCByZXQgPSB0aGlzLm91dHB1dHNbbG9naWNhbElkXTtcbiAgICAgICAgaWYgKCFyZXQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgT3V0cHV0IHdpdGggbG9naWNhbCBJRCAnJHtsb2dpY2FsSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgfVxuICAgIC8qKiBAaW50ZXJuYWwgKi9cbiAgICBwdWJsaWMgX3RvQ2xvdWRGb3JtYXRpb24oKTogb2JqZWN0IHtcbiAgICAgICAgY29uc3QgcmV0OiB7XG4gICAgICAgICAgICBbc2VjdGlvbjogc3RyaW5nXTogYW55O1xuICAgICAgICB9ID0ge307XG4gICAgICAgIGZvciAoY29uc3Qgc2VjdGlvbiBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlKSkge1xuICAgICAgICAgICAgLy8gcmVuZGVyIGFsbCBzZWN0aW9ucyBvZiB0aGUgdGVtcGxhdGUgdW5jaGFuZ2VkLFxuICAgICAgICAgICAgLy8gZXhjZXB0IENvbmRpdGlvbnMsIFJlc291cmNlcywgUGFyYW1ldGVycywgYW5kIE91dHB1dHMgd2hpY2ggd2lsbCBiZSB0YWtlbiBjYXJlIG9mIGJ5IHRoZSBjcmVhdGVkIEwxc1xuICAgICAgICAgICAgaWYgKHNlY3Rpb24gIT09ICdDb25kaXRpb25zJyAmJiBzZWN0aW9uICE9PSAnUmVzb3VyY2VzJyAmJiBzZWN0aW9uICE9PSAnUGFyYW1ldGVycycgJiYgc2VjdGlvbiAhPT0gJ091dHB1dHMnKSB7XG4gICAgICAgICAgICAgICAgcmV0W3NlY3Rpb25dID0gdGhpcy50ZW1wbGF0ZVtzZWN0aW9uXTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZVBhcmFtZXRlcihsb2dpY2FsSWQ6IHN0cmluZyk6IHZvaWQge1xuICAgICAgICBjb25zdCBleHByZXNzaW9uID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgICAgICAgZmluZGVyOiB7XG4gICAgICAgICAgICAgICAgZmluZFJlc291cmNlKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1VzaW5nIEdldEF0dCBleHByZXNzaW9ucyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kUmVmVGFyZ2V0KCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1VzaW5nIFJlZiBleHByZXNzaW9ucyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1JlZmVycmluZyB0byBDb25kaXRpb25zIGluIFBhcmFtZXRlciBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSkucGFyc2VWYWx1ZSh0aGlzLnRlbXBsYXRlLlBhcmFtZXRlcnNbbG9naWNhbElkXSk7XG4gICAgICAgIGNvbnN0IGNmblBhcmFtZXRlciA9IG5ldyBjb3JlLkNmblBhcmFtZXRlcih0aGlzLCBsb2dpY2FsSWQsIHtcbiAgICAgICAgICAgIHR5cGU6IGV4cHJlc3Npb24uVHlwZSxcbiAgICAgICAgICAgIGRlZmF1bHQ6IGV4cHJlc3Npb24uRGVmYXVsdCxcbiAgICAgICAgICAgIGFsbG93ZWRQYXR0ZXJuOiBleHByZXNzaW9uLkFsbG93ZWRQYXR0ZXJuLFxuICAgICAgICAgICAgY29uc3RyYWludERlc2NyaXB0aW9uOiBleHByZXNzaW9uLkNvbnN0cmFpbnREZXNjcmlwdGlvbixcbiAgICAgICAgICAgIGRlc2NyaXB0aW9uOiBleHByZXNzaW9uLkRlc2NyaXB0aW9uLFxuICAgICAgICAgICAgbWF4TGVuZ3RoOiBleHByZXNzaW9uLk1heExlbmd0aCxcbiAgICAgICAgICAgIG1heFZhbHVlOiBleHByZXNzaW9uLk1heFZhbHVlLFxuICAgICAgICAgICAgbWluTGVuZ3RoOiBleHByZXNzaW9uLk1pbkxlbmd0aCxcbiAgICAgICAgICAgIG1pblZhbHVlOiBleHByZXNzaW9uLk1pblZhbHVlLFxuICAgICAgICAgICAgbm9FY2hvOiBleHByZXNzaW9uLk5vRWNobyxcbiAgICAgICAgfSk7XG4gICAgICAgIGNmblBhcmFtZXRlci5vdmVycmlkZUxvZ2ljYWxJZChsb2dpY2FsSWQpO1xuICAgICAgICB0aGlzLnBhcmFtZXRlcnNbbG9naWNhbElkXSA9IGNmblBhcmFtZXRlcjtcbiAgICB9XG4gICAgcHJpdmF0ZSBjcmVhdGVPdXRwdXQobG9naWNhbElkOiBzdHJpbmcsIHNjb3BlOiBjb3JlLkNvbnN0cnVjdCk6IHZvaWQge1xuICAgICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgICAgY29uc3Qgb3V0cHV0QXR0cmlidXRlcyA9IG5ldyBjZm5fcGFyc2UuQ2ZuUGFyc2VyKHtcbiAgICAgICAgICAgIGZpbmRlcjoge1xuICAgICAgICAgICAgICAgIGZpbmRSZXNvdXJjZShsSWQpOiBjb3JlLkNmblJlc291cmNlIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYucmVzb3VyY2VzW2xJZF07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kUmVmVGFyZ2V0KGVsZW1lbnROYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkVsZW1lbnQgfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5yZXNvdXJjZXNbZWxlbWVudE5hbWVdID8/IHNlbGYucGFyYW1ldGVyc1tlbGVtZW50TmFtZV07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKCk6IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pLnBhcnNlVmFsdWUodGhpcy50ZW1wbGF0ZS5PdXRwdXRzW2xvZ2ljYWxJZF0pO1xuICAgICAgICBjb25zdCBjZm5PdXRwdXQgPSBuZXcgY29yZS5DZm5PdXRwdXQoc2NvcGUsIGxvZ2ljYWxJZCwge1xuICAgICAgICAgICAgdmFsdWU6IG91dHB1dEF0dHJpYnV0ZXMuVmFsdWUsXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogb3V0cHV0QXR0cmlidXRlcy5EZXNjcmlwdGlvbixcbiAgICAgICAgICAgIGV4cG9ydE5hbWU6IG91dHB1dEF0dHJpYnV0ZXMuRXhwb3J0ID8gb3V0cHV0QXR0cmlidXRlcy5FeHBvcnQuTmFtZSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIGNvbmRpdGlvbjogKCgpID0+IHtcbiAgICAgICAgICAgICAgICBpZiAoIW91dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9uKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIGVsc2UgaWYgKHRoaXMuY29uZGl0aW9uc1tvdXRwdXRBdHRyaWJ1dGVzLkNvbmRpdGlvbl0pIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYuZ2V0Q29uZGl0aW9uKG91dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9uKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBPdXRwdXQgd2l0aCBuYW1lICcke2xvZ2ljYWxJZH0nIHJlZmVycyB0byBhIENvbmRpdGlvbiB3aXRoIG5hbWVcXFxuICcke291dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9ufScgd2hpY2ggd2FzIG5vdCBmb3VuZCBpbiB0aGlzIHRlbXBsYXRlYCk7XG4gICAgICAgICAgICB9KSgpLFxuICAgICAgICB9KTtcbiAgICAgICAgY2ZuT3V0cHV0Lm92ZXJyaWRlTG9naWNhbElkKGxvZ2ljYWxJZCk7XG4gICAgICAgIHRoaXMub3V0cHV0c1tsb2dpY2FsSWRdID0gY2ZuT3V0cHV0O1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZUNvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiB2b2lkIHtcbiAgICAgICAgLy8gVG9EbyBjb25kaXRpb24gZXhwcmVzc2lvbnMgY2FuIHJlZmVyIHRvIG90aGVyIGNvbmRpdGlvbnMgLVxuICAgICAgICAvLyB3aWxsIGJlIGltcG9ydGFudCB3aGVuIGltcGxlbWVudGluZyBwcmVzZXJ2ZUxvZ2ljYWxJZHM9ZmFsc2VcbiAgICAgICAgY29uc3QgZXhwcmVzc2lvbiA9IG5ldyBjZm5fcGFyc2UuQ2ZuUGFyc2VyKHtcbiAgICAgICAgICAgIGZpbmRlcjoge1xuICAgICAgICAgICAgICAgIGZpbmRSZXNvdXJjZSgpIHsgdGhyb3cgbmV3IEVycm9yKCdVc2luZyBHZXRBdHQgaW4gQ29uZGl0aW9uIGRlZmluaXRpb25zIGlzIG5vdCBhbGxvd2VkJyk7IH0sXG4gICAgICAgICAgICAgICAgZmluZFJlZlRhcmdldCgpIHsgdGhyb3cgbmV3IEVycm9yKCdVc2luZyBSZWYgZXhwcmVzc2lvbnMgaW4gQ29uZGl0aW9uIGRlZmluaXRpb25zIGlzIG5vdCBhbGxvd2VkJyk7IH0sXG4gICAgICAgICAgICAgICAgLy8gVG9EbyBoYW5kbGUgb25lIENvbmRpdGlvbiByZWZlcmVuY2luZyBhbm90aGVyIHVzaW5nIHRoZSB7IENvbmRpdGlvbjogXCJDb25kaXRpb25OYW1lXCIgfSBzeW50YXhcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKCkgeyByZXR1cm4gdW5kZWZpbmVkOyB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSkucGFyc2VWYWx1ZSh0aGlzLnRlbXBsYXRlLkNvbmRpdGlvbnNbY29uZGl0aW9uTmFtZV0pO1xuICAgICAgICBjb25zdCBjZm5Db25kaXRpb24gPSBuZXcgY29yZS5DZm5Db25kaXRpb24odGhpcywgY29uZGl0aW9uTmFtZSwge1xuICAgICAgICAgICAgZXhwcmVzc2lvbixcbiAgICAgICAgfSk7XG4gICAgICAgIC8vIFRvRG8gaGFuZGxlIHJlbmFtaW5nIG9mIHRoZSBsb2dpY2FsIElEcyBvZiB0aGUgY29uZGl0aW9uc1xuICAgICAgICBjZm5Db25kaXRpb24ub3ZlcnJpZGVMb2dpY2FsSWQoY29uZGl0aW9uTmFtZSk7XG4gICAgICAgIHRoaXMuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXSA9IGNmbkNvbmRpdGlvbjtcbiAgICB9XG4gICAgcHJpdmF0ZSBnZXRPckNyZWF0ZVJlc291cmNlKGxvZ2ljYWxJZDogc3RyaW5nKTogY29yZS5DZm5SZXNvdXJjZSB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMucmVzb3VyY2VzW2xvZ2ljYWxJZF07XG4gICAgICAgIGlmIChyZXQpIHtcbiAgICAgICAgICAgIHJldHVybiByZXQ7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgcmVzb3VyY2VBdHRyaWJ1dGVzOiBhbnkgPSB0aGlzLnRlbXBsYXRlLlJlc291cmNlc1tsb2dpY2FsSWRdO1xuICAgICAgICBjb25zdCBsMUNsYXNzRnFuID0gY2ZuX3R5cGVfdG9fbDFfbWFwcGluZy5sb29rdXAocmVzb3VyY2VBdHRyaWJ1dGVzLlR5cGUpO1xuICAgICAgICBpZiAoIWwxQ2xhc3NGcW4pIHtcbiAgICAgICAgICAgIC8vIGN1cnJlbnRseSwgd2Ugb25seSBoYW5kbGUgdHlwZXMgd2Uga25vdyB0aGUgTDEgZm9yIC1cbiAgICAgICAgICAgIC8vIGluIHRoZSBmdXR1cmUsIHdlIG1pZ2h0IGNvbnN0cnVjdCBhbiBpbnN0YW5jZSBvZiBDZm5SZXNvdXJjZSBpbnN0ZWFkXG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVucmVjb2duaXplZCBDbG91ZEZvcm1hdGlvbiByZXNvdXJjZSB0eXBlOiAnJHtyZXNvdXJjZUF0dHJpYnV0ZXMuVHlwZX0nYCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gZmFpbCBlYXJseSBmb3IgcmVzb3VyY2UgYXR0cmlidXRlcyB3ZSBkb24ndCBzdXBwb3J0IHlldFxuICAgICAgICBjb25zdCBrbm93bkF0dHJpYnV0ZXMgPSBbXG4gICAgICAgICAgICAnVHlwZScsICdQcm9wZXJ0aWVzJywgJ0NvbmRpdGlvbicsICdEZXBlbmRzT24nLCAnTWV0YWRhdGEnLFxuICAgICAgICAgICAgJ0NyZWF0aW9uUG9saWN5JywgJ1VwZGF0ZVBvbGljeScsICdEZWxldGlvblBvbGljeScsICdVcGRhdGVSZXBsYWNlUG9saWN5JyxcbiAgICAgICAgXTtcbiAgICAgICAgZm9yIChjb25zdCBhdHRyaWJ1dGUgb2YgT2JqZWN0LmtleXMocmVzb3VyY2VBdHRyaWJ1dGVzKSkge1xuICAgICAgICAgICAgaWYgKCFrbm93bkF0dHJpYnV0ZXMuaW5jbHVkZXMoYXR0cmlidXRlKSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVGhlICR7YXR0cmlidXRlfSByZXNvdXJjZSBhdHRyaWJ1dGUgaXMgbm90IHN1cHBvcnRlZCBieSBjbG91ZGZvcm1hdGlvbi1pbmNsdWRlIHlldC4gYCArXG4gICAgICAgICAgICAgICAgICAgICdFaXRoZXIgcmVtb3ZlIGl0IGZyb20gdGhlIHRlbXBsYXRlLCBvciB1c2UgdGhlIENka0luY2x1ZGUgY2xhc3MgZnJvbSB0aGUgY29yZSBwYWNrYWdlIGluc3RlYWQuJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgW21vZHVsZU5hbWUsIC4uLmNsYXNzTmFtZV0gPSBsMUNsYXNzRnFuLnNwbGl0KCcuJyk7XG4gICAgICAgIGNvbnN0IG1vZHVsZSA9IHJlcXVpcmUobW9kdWxlTmFtZSk7IC8vIGVzbGludC1kaXNhYmxlLWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuICAgICAgICBjb25zdCBqc0NsYXNzRnJvbU1vZHVsZSA9IG1vZHVsZVtjbGFzc05hbWUuam9pbignLicpXTtcbiAgICAgICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgICAgIGNvbnN0IGZpbmRlcjogY29yZS5JQ2ZuRmluZGVyID0ge1xuICAgICAgICAgICAgZmluZENvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXTtcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBmaW5kUmVzb3VyY2UobElkOiBzdHJpbmcpOiBjb3JlLkNmblJlc291cmNlIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICBpZiAoIShsSWQgaW4gKHNlbGYudGVtcGxhdGUuUmVzb3VyY2VzIHx8IHt9KSkpIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYuZ2V0T3JDcmVhdGVSZXNvdXJjZShsSWQpO1xuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIGZpbmRSZWZUYXJnZXQoZWxlbWVudE5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuRWxlbWVudCB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgaWYgKGVsZW1lbnROYW1lIGluIHNlbGYucGFyYW1ldGVycykge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5wYXJhbWV0ZXJzW2VsZW1lbnROYW1lXTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgcmV0dXJuIHRoaXMuZmluZFJlc291cmNlKGVsZW1lbnROYW1lKTtcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgICAgIGNvbnN0IG9wdGlvbnM6IGNvcmUuRnJvbUNsb3VkRm9ybWF0aW9uT3B0aW9ucyA9IHtcbiAgICAgICAgICAgIGZpbmRlcixcbiAgICAgICAgfTtcbiAgICAgICAgY29uc3QgbDFJbnN0YW5jZSA9IGpzQ2xhc3NGcm9tTW9kdWxlLmZyb21DbG91ZEZvcm1hdGlvbih0aGlzLCBsb2dpY2FsSWQsIHJlc291cmNlQXR0cmlidXRlcywgb3B0aW9ucyk7XG4gICAgICAgIGlmICh0aGlzLnByZXNlcnZlTG9naWNhbElkcykge1xuICAgICAgICAgICAgLy8gb3ZlcnJpZGUgdGhlIGxvZ2ljYWwgSUQgdG8gbWF0Y2ggdGhlIG9yaWdpbmFsIHRlbXBsYXRlXG4gICAgICAgICAgICBsMUluc3RhbmNlLm92ZXJyaWRlTG9naWNhbElkKGxvZ2ljYWxJZCk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5yZXNvdXJjZXNbbG9naWNhbElkXSA9IGwxSW5zdGFuY2U7XG4gICAgICAgIHJldHVybiBsMUluc3RhbmNlO1xuICAgIH1cbn1cbiJdfQ==