"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketNotifications = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.LAMBDA:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.QUEUE:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.TOPIC:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined,
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: cdk.Lazy.anyValue({ produce: () => this.renderNotificationConfiguration() }),
                },
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    let hasPrefix = false;
    let hasSuffix = false;
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            if (hasSuffix) {
                throw new Error('Cannot specify more than one suffix rule in a filter.');
            }
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
            hasSuffix = true;
        }
        if (rule.prefix) {
            if (hasPrefix) {
                throw new Error('Cannot specify more than one prefix rule in a filter.');
            }
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
            hasPrefix = true;
        }
    }
    return {
        Key: {
            FilterRules: renderedRules,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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