"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Asset = void 0;
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs");
const path = require("path");
const compat_1 = require("./compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: compat_1.toSymlinkFollow(props.follow),
            assetHash: (_a = props.assetHash) !== null && _a !== void 0 ? _a : props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        this.assetPath = staging.stagedPath;
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === cdk.FileAssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        const stack = cdk.Stack.of(this);
        const location = stack.synthesizer.addFileAsset({
            packaging,
            sourceHash: this.sourceHash,
            fileName: staging.stagedPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        const kmsKey = location.kmsKeyArn ? kms.Key.fromKeyArn(this, 'Key', location.kmsKeyArn) : undefined;
        this.bucket = s3.Bucket.fromBucketAttributes(this, 'AssetBucket', {
            bucketName: this.s3BucketName,
            encryptionKey: kmsKey,
        });
        for (const reader of ((_b = props.readers) !== null && _b !== void 0 ? _b : [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the assets bucket.
     */
    grantRead(grantee) {
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return cdk.FileAssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return cdk.FileAssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
//# sourceMappingURL=data:application/json;base64,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