"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        if (props.retryAttempts && (props.retryAttempts < 0 || props.retryAttempts > 10000)) {
            throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${props.retryAttempts}`);
        }
        if ((props.parallelizationFactor || props.parallelizationFactor === 0) && (props.parallelizationFactor < 1 || props.parallelizationFactor > 10)) {
            throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${props.parallelizationFactor}`);
        }
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB or Kinesis stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,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