"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        if (baseProps.deletionProtection) {
            this.setAttribute('deletion_protection.enabled', 'true');
        }
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = ELBV2_ACCOUNTS[region];
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
// https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html#access-logging-bucket-permissions
const ELBV2_ACCOUNTS = {
    'us-east-1': '127311923021',
    'us-east-2': '033677994240',
    'us-west-1': '027434742980',
    'us-west-2': '797873946194',
    'af-south-1': '098369216593',
    'ca-central-1': '985666609251',
    'eu-central-1': '054676820928',
    'eu-west-1': '156460612806',
    'eu-west-2': '652711504416',
    'eu-west-3': '009996457667',
    'eu-south-1': '635631232127',
    'eu-north-1': '897822967062',
    'ap-east-1': '754344448648',
    'ap-northeast-1': '582318560864',
    'ap-northeast-2': '600734575887',
    'ap-northeast-3': '383597477331',
    'ap-southeast-1': '114774131450',
    'ap-southeast-2': '783225319266',
    'ap-south-1': '718504428378',
    'me-south-1': '076674570225',
    'sa-east-1': '507241528517',
    'us-gov-west-1': '048591011584',
    'us-gov-east-1': '190560391635',
    'cn-north-1': '638102146993',
    'cn-northwest-1': '037604701340',
};
//# sourceMappingURL=data:application/json;base64,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