"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.KubernetesVersion = void 0;
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
const YAML = require("yaml");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const eks_generated_1 = require("./eks.generated");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const k8s_patch_1 = require("./k8s-patch");
const k8s_resource_1 = require("./k8s-resource");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * Kubernetes cluster version
 */
class KubernetesVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version) { return new KubernetesVersion(version); }
}
exports.KubernetesVersion = KubernetesVersion;
/**
 * Kubernetes version 1.14
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * Kubernetes version 1.15
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * Kubernetes version 1.16
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // Get subnetIds for all selected subnets
        const placements = props.vpcSubnets || [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const subnetIds = [...new Set(Array().concat(...placements.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
        };
        let resource;
        this.kubectlEnabled = props.kubectlEnabled === undefined ? true : props.kubectlEnabled;
        if (this.kubectlEnabled) {
            resource = new cluster_resource_1.ClusterResource(this, 'Resource', clusterProps);
            this._clusterResource = resource;
            // we use an SSM parameter as a barrier because it's free and fast.
            this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
                type: 'AWS::SSM::Parameter',
                properties: {
                    Type: 'String',
                    Value: 'aws:cdk:eks:kubectl-ready',
                },
            });
            // add the cluster resource itself as a dependency of the barrier
            this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        }
        else {
            resource = new eks_generated_1.CfnCluster(this, 'Resource', clusterProps);
        }
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // map the IAM role to the `system:masters` group.
        if (props.mastersRole) {
            if (!this.kubectlEnabled) {
                throw new Error('Cannot specify a "masters" role if kubectl is disabled');
            }
            this.awsAuth.addMastersRole(props.mastersRole);
            if (props.outputMastersRoleArn) {
                new core_1.CfnOutput(this, 'MastersRoleArn', { value: props.mastersRole.roleArn });
            }
            commonCommandOptions.push(`--role-arn ${props.mastersRole.roleArn}`);
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroup('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        if (this.kubectlEnabled) {
            this.defineCoreDnsComputeType((_a = props.coreDnsComputeType) !== null && _a !== void 0 ? _a : CoreDnsComputeType.EC2);
        }
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new BottleRocketImage() :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tag.add(autoScalingGroup, `kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        if (options.mapRole === true && !this.kubectlEnabled) {
            throw new Error('Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster');
        }
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole && this.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && this.kubectlEnabled) {
            this.addSpotInterruptHandler();
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this.kubectlEnabled) {
            throw new Error('Cannot define aws-auth mappings if kubectl is disabled');
        }
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl() {
        if (!this._clusterResource) {
            throw new Error('unable to obtain OpenID Connect issuer URL. Cluster must be kubectl-enabled');
        }
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer() {
        if (!this._clusterResource) {
            throw new Error('unable to obtain OpenID Connect issuer. Cluster must be kubectl-enabled');
        }
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider() {
        if (!this.kubectlEnabled) {
            throw new Error('Cannot specify a OpenID Connect Provider if kubectl is disabled');
        }
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new iam.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
                clientIds: ['sts.amazonaws.com'],
                /**
                 * For some reason EKS isn't validating the root certificate but a intermediat certificate
                 * which is one level up in the tree. Because of the a constant thumbprint value has to be
                 * stated with this OpenID Connect provider. The certificate thumbprint is the same for all the regions.
                 */
                thumbprints: ['9e99a48a9960b14926bb7f3b02e22da2b0ab7280'],
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     * @throws If `kubectlEnabled` is `false`
     */
    addResource(id, ...manifest) {
        return new k8s_resource_1.KubernetesResource(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     * @throws If `kubectlEnabled` is `false`
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Adds a service account to this cluster.
     *
     * @param id the id of this service account
     * @param options service account options
     */
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Returns the role ARN for the cluster creation role for kubectl-enabled
     * clusters.
     * @param assumedBy The IAM that will assume this role. If omitted, the
     * creation role will be returned witout modification of its trust policy.
     *
     * @internal
     */
    get _kubectlCreationRole() {
        if (!this._clusterResource) {
            throw new Error('Unable to perform this operation since kubectl is not enabled for this cluster');
        }
        return this._clusterResource.creationRole;
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        if (this._kubectlReadyBarrier) {
            this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        }
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @property resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        if (!this._clusterResource) {
            throw new Error('Unable to perform this operation since kubectl is not enabled for this cluster');
        }
        // singleton
        let provider = this.stack.node.tryFindChild(uid);
        if (!provider) {
            // create the provider.
            provider = new kubectl_provider_1.KubectlProvider(this.stack, uid);
        }
        // allow the kubectl provider to assume the cluster creation role.
        this._clusterResource.addTrustedRole(provider.role);
        if (!this._kubectlReadyBarrier) {
            throw new Error('unexpected: kubectl enabled clusters should have a kubectl-ready barrier resource');
        }
        resourceScope.node.addDependency(this._kubectlReadyBarrier);
        return provider;
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.7.3',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    'nodeSelector.lifecycle': user_data_1.LifecycleLabel.SPOT,
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addResource('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    this.node.addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                subnet.node.applyAspect(new core_1.Tag(tag, '1'));
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        if (!this.kubectlEnabled) {
            throw new Error('kubectl must be enabled in order to define the compute type for CoreDNS');
        }
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, 'VPC', props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        this.clusterSecurityGroupId = props.clusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = props.clusterEncryptionConfigKeyArn;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props = {}) {
        var _a, _b;
        this.nodeType = (_a = props.nodeType) !== null && _a !== void 0 ? _a : NodeType.STANDARD;
        this.kubernetesVersion = (_b = props.kubernetesVersion) !== null && _b !== void 0 ? _b : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? 'amazon-linux-2/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM
 */
class BottleRocketImage {
    /**
     * Constructs a new instance of the BottleRocketImage class.
     */
    constructor() {
        // only 1.15 is currently available
        this.kubernetesVersion = '1.15';
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/aws-k8s-${this.kubernetesVersion}/x86_64/latest/image_id`;
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
    /**
     * Inferentia instances
     */
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
const INFERENTIA_INSTANCETYPES = ['inf1'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        INFERENTIA_INSTANCETYPES.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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