"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.allRouteTableIds = exports.range = exports.ImportSubnetGroup = exports.subnetId = exports.subnetGroupNameFromConstructId = exports.defaultSubnetName = exports.slugify = void 0;
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            // tslint:disable-next-line: max-line-length
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(subnets) {
    const ret = new Set();
    for (const subnet of subnets) {
        if (subnet.routeTable && subnet.routeTable.routeTableId) {
            ret.add(subnet.routeTable.routeTableId);
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
function flatten(xs) {
    return Array.prototype.concat.apply([], xs);
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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