"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReportGroup = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const codebuild_generated_1 = require("./codebuild.generated");
const report_group_utils_1 = require("./report-group-utils");
class ReportGroupBase extends cdk.Resource {
    grantWrite(identity) {
        const ret = iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resourceArns: [this.reportGroupArn],
        });
        if (this.exportBucket) {
            this.exportBucket.grantWrite(identity);
        }
        return ret;
    }
}
/**
 * The ReportGroup resource class.
 */
class ReportGroup extends ReportGroupBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.reportGroupName,
        });
        const resource = new codebuild_generated_1.CfnReportGroup(this, 'Resource', {
            type: 'TEST',
            exportConfig: {
                exportConfigType: props.exportBucket ? 'S3' : 'NO_EXPORT',
                s3Destination: props.exportBucket
                    ? {
                        bucket: props.exportBucket.bucketName,
                        encryptionDisabled: props.exportBucket.encryptionKey ? false : undefined,
                        encryptionKey: (_a = props.exportBucket.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
                        packaging: props.zipExport ? 'ZIP' : undefined,
                    }
                    : undefined,
            },
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: cdk.RemovalPolicy.RETAIN,
        });
        this.reportGroupArn = this.getResourceArnAttribute(resource.attrArn, report_group_utils_1.reportGroupArnComponents(this.physicalName));
        this.reportGroupName = this.getResourceNameAttribute(
        // there is no separate name attribute,
        // so use Fn::Select + Fn::Split to make one
        cdk.Fn.select(1, cdk.Fn.split('/', resource.ref)));
        this.exportBucket = props.exportBucket;
    }
    /**
     * Reference an existing ReportGroup,
     * defined outside of the CDK code,
     * by name.
     */
    static fromReportGroupName(scope, id, reportGroupName) {
        class Import extends ReportGroupBase {
            constructor() {
                super(...arguments);
                this.reportGroupName = reportGroupName;
                this.reportGroupArn = report_group_utils_1.renderReportGroupArn(scope, reportGroupName);
                this.exportBucket = undefined;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReportGroup = ReportGroup;
//# sourceMappingURL=data:application/json;base64,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