"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_ecr_assets_1 = require("../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
const S3_BUCKET_ENV = 'SCRIPT_S3_BUCKET';
const S3_KEY_ENV = 'SCRIPT_S3_KEY';
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', { target });
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.metric('Builds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.metric('Duration', {
            statistic: 'avg',
            ...props,
        });
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.metric('SucceededBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.metric('FailedBuilds', {
            statistic: 'sum',
            ...props,
        });
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props.environment, environmentVariables),
            fileSystemLocations: this.renderFileSystemLocations(),
            // lazy, because we have a setter for it in setEncryptionKey
            encryptionKey: core_1.Lazy.stringValue({ produce: () => this._encryptionKey && this._encryptionKey.keyArn }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            secondarySources: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.anyValue({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.anyValue({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
    }
    static fromProjectArn(scope, id, projectArn) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.projectArn = projectArn;
                this.projectName = core_1.Stack.of(scope).parseArn(projectArn).resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable},
     * which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty},
     * which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     */
    static serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            name,
            type: environmentVariables[name].type || BuildEnvironmentVariableType.PLAINTEXT,
            value: environmentVariables[name].value,
        }));
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in
     * @param options additional options for the binding
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * @override
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(env = {}, projectVars = {}) {
        var _a;
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? undefined
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_a = this.buildImage.secretsManagerCredentials) === null || _a === void 0 ? void 0 : _a.grantRead(this);
        }
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: this.buildImage.secretsManagerCredentials
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    credential: this.buildImage.secretsManagerCredentials.secretArn,
                }
                : undefined,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars ? Project.serializeEnvVariables(vars) : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:aws:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    /**
     * CODEBUILD specifies that CodeBuild uses its own identity when pulling the image.
     * This means the resource policy of the ECR repository that hosts the image will be modified to trust
     * CodeBuild's service principal.
     * This is the required principal type when using CodeBuild's pre-defined images.
     */
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    /**
     * SERVICE_ROLE specifies that AWS CodeBuild uses the project's role when pulling the image.
     * The role will be granted pull permissions on the ECR repository hosting the image.
     */
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(props) {
        this.type = 'LINUX_CONTAINER';
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Linux build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @returns A Docker image provided by CodeBuild.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     *
     * @param id The image identifier
     * @example 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    validate(_) {
        return [];
    }
    runScriptBuildspec(entrypoint) {
        return runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/** The `aws/codebuild/standard:4.0` build image. */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/** The Amazon Linux 2 x86_64 standard image, version `3.0`. */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
function runScriptLinuxBuildSpec(entrypoint) {
    return build_spec_1.BuildSpec.fromObject({
        version: '0.2',
        phases: {
            pre_build: {
                commands: [
                    // Better echo the location here; if this fails, the error message only contains
                    // the unexpanded variables by default. It might fail if you're running an old
                    // definition of the CodeBuild project--the permissions will have been changed
                    // to only allow downloading the very latest version.
                    `echo "Downloading scripts from s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}}"`,
                    `aws s3 cp s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}} /tmp`,
                    'mkdir -p /tmp/scriptdir',
                    `unzip /tmp/$(basename \$${S3_KEY_ENV}) -d /tmp/scriptdir`,
                ],
            },
            build: {
                commands: [
                    'export SCRIPT_DIR=/tmp/scriptdir',
                    `echo "Running ${entrypoint}"`,
                    `chmod +x /tmp/scriptdir/${entrypoint}`,
                    `/tmp/scriptdir/${entrypoint}`,
                ],
            },
        },
    });
}
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag])
 * - WindowsBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(props) {
        this.type = 'WINDOWS_CONTAINER';
        this.defaultComputeType = ComputeType.MEDIUM;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${S3_BUCKET_ENV}/$env:${S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
/**
 * Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is
 * based off Windows Server Core 2016.
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    /**
     * An environment variable in plaintext format.
     */
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    /**
     * An environment variable stored in AWS Secrets Manager.
     */
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
//# sourceMappingURL=data:application/json;base64,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