"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
class AutoScalingGroupBase extends core_1.Resource {
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.notifications = [];
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            this.node.addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Add command to the startup script of fleet instances.
     * The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launced
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custome group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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