"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi
 */
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
}
exports.RestApiBase = RestApiBase;
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, props, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing RestApi that can be configured with additional Methods and Resources.
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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