"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const cognito = require("../lib");
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new cognito.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool'
        }));
        test.done();
    },
    'lambda triggers are defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        // WHEN
        const pool = new cognito.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        pool.addCustomMessageTrigger(fn);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
                CustomMessage: stack.resolve(fn.functionArn)
            }
        }));
        test.done();
    },
    'on* API correctly appends triggers'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        // WHEN
        const pool = new cognito.UserPool(stack, 'Pool', {});
        pool.addCreateAuthChallengeTrigger(fn);
        pool.addCustomMessageTrigger(fn);
        pool.addDefineAuthChallengeTrigger(fn);
        pool.addPostAuthenticationTrigger(fn);
        pool.addPostConfirmationTrigger(fn);
        pool.addPreAuthenticationTrigger(fn);
        pool.addPreSignUpTrigger(fn);
        pool.addVerifyAuthChallengeResponseTrigger(fn);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(fn.functionArn),
                CustomMessage: stack.resolve(fn.functionArn),
                DefineAuthChallenge: stack.resolve(fn.functionArn),
                PostAuthentication: stack.resolve(fn.functionArn),
                PostConfirmation: stack.resolve(fn.functionArn),
                PreAuthentication: stack.resolve(fn.functionArn),
                PreSignUp: stack.resolve(fn.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(fn.functionArn)
            }
        }));
        test.done();
    },
    'lambdas are given cognito service grant'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        // WHEN
        new cognito.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Permission', {
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com'
        }));
        test.done();
    },
    'set sign in type'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new cognito.UserPool(stack, 'Pool', {
            signInType: cognito.SignInType.EMAIL,
            autoVerifiedAttributes: [cognito.UserPoolAttribute.EMAIL]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email'],
            AutoVerifiedAttributes: ['email']
        }));
        test.done();
    },
    'usernameAliasAttributes require signInType of USERNAME'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = () => {
            new cognito.UserPool(stack, 'Pool', {
                signInType: cognito.SignInType.EMAIL,
                usernameAliasAttributes: [cognito.UserPoolAttribute.PREFERRED_USERNAME]
            });
        };
        // THEN
        test.throws(() => toThrow(), /'usernameAliasAttributes' can only be set with a signInType of 'USERNAME'/);
        test.done();
    },
    'usernameAliasAttributes must be one or more of EMAIL, PHONE_NUMBER, or PREFERRED_USERNAME'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = () => {
            new cognito.UserPool(stack, 'Pool', {
                signInType: cognito.SignInType.USERNAME,
                usernameAliasAttributes: [cognito.UserPoolAttribute.GIVEN_NAME]
            });
        };
        // THEN
        test.throws(() => toThrow(), /'usernameAliasAttributes' can only include EMAIL, PHONE_NUMBER, or PREFERRED_USERNAME/);
        test.done();
    },
    'autoVerifiedAttributes must be one or more of EMAIL or PHONE_NUMBER'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const toThrow = () => {
            new cognito.UserPool(stack, 'Pool', {
                signInType: cognito.SignInType.EMAIL,
                autoVerifiedAttributes: [cognito.UserPoolAttribute.EMAIL, cognito.UserPoolAttribute.GENDER]
            });
        };
        // THEN
        test.throws(() => toThrow(), /'autoVerifiedAttributes' can only include EMAIL or PHONE_NUMBER/);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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