"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Standard attributes
 * Specified following the OpenID Connect spec
 * @see https://openid.net/specs/openid-connect-core-1_0.html#StandardClaims
 */
var UserPoolAttribute;
(function (UserPoolAttribute) {
    /**
     * End-User's preferred postal address.
     */
    UserPoolAttribute["ADDRESS"] = "address";
    /**
     * End-User's birthday, represented as an ISO 8601:2004 [ISO8601‑2004] YYYY-MM-DD format.
     * The year MAY be 0000, indicating that it is omitted.
     * To represent only the year, YYYY format is allowed.
     */
    UserPoolAttribute["BIRTHDATE"] = "birthdate";
    /**
     * End-User's preferred e-mail address.
     * Its value MUST conform to the RFC 5322 [RFC5322] addr-spec syntax.
     */
    UserPoolAttribute["EMAIL"] = "email";
    /**
     * Surname(s) or last name(s) of the End-User.
     * Note that in some cultures, people can have multiple family names or no family name;
     * all can be present, with the names being separated by space characters.
     */
    UserPoolAttribute["FAMILY_NAME"] = "family_name";
    /**
     * End-User's gender.
     */
    UserPoolAttribute["GENDER"] = "gender";
    /**
     * Given name(s) or first name(s) of the End-User.
     * Note that in some cultures, people can have multiple given names;
     * all can be present, with the names being separated by space characters.
     */
    UserPoolAttribute["GIVEN_NAME"] = "given_name";
    /**
     * End-User's locale, represented as a BCP47 [RFC5646] language tag.
     * This is typically an ISO 639-1 Alpha-2 [ISO639‑1] language code in lowercase
     * and an ISO 3166-1 Alpha-2 [ISO3166‑1] country code in uppercase, separated by a dash.
     * For example, en-US or fr-CA.
     */
    UserPoolAttribute["LOCALE"] = "locale";
    /**
     * Middle name(s) of the End-User.
     * Note that in some cultures, people can have multiple middle names;
     * all can be present, with the names being separated by space characters.
     * Also note that in some cultures, middle names are not used.
     */
    UserPoolAttribute["MIDDLE_NAME"] = "middle_name";
    /**
     * End-User's full name in displayable form including all name parts,
     * possibly including titles and suffixes, ordered according to the End-User's locale and preferences.
     */
    UserPoolAttribute["NAME"] = "name";
    /**
     * Casual name of the End-User that may or may not be the same as the given_name.
     * For instance, a nickname value of Mike might be returned alongside a given_name value of Michael.
     */
    UserPoolAttribute["NICKNAME"] = "nickname";
    /**
     * End-User's preferred telephone number.
     * E.164 [E.164] is RECOMMENDED as the format of this Claim, for example, +1 (425) 555-1212 or +56 (2) 687 2400.
     * If the phone number contains an extension, it is RECOMMENDED that the extension be represented using the
     * RFC 3966 [RFC3966] extension syntax, for example, +1 (604) 555-1234;ext=5678.
     */
    UserPoolAttribute["PHONE_NUMBER"] = "phone_number";
    /**
     * URL of the End-User's profile picture.
     * This URL MUST refer to an image file (for example, a PNG, JPEG, or GIF image file),
     * rather than to a Web page containing an image.
     * Note that this URL SHOULD specifically reference a profile photo of the End-User
     * suitable for displaying when describing the End-User, rather than an arbitrary photo taken by the End-User
     */
    UserPoolAttribute["PICTURE"] = "picture";
    /**
     * Shorthand name by which the End-User wishes to be referred to.
     */
    UserPoolAttribute["PREFERRED_USERNAME"] = "preferred_username";
    /**
     * URL of the End-User's profile page. The contents of this Web page SHOULD be about the End-User.
     */
    UserPoolAttribute["PROFILE"] = "profile";
    /**
     * The End-User's time zone
     */
    UserPoolAttribute["TIMEZONE"] = "timezone";
    /**
     * Time the End-User's information was last updated.
     * Its value is a JSON number representing the number of seconds from 1970-01-01T0:0:0Z
     * as measured in UTC until the date/time.
     */
    UserPoolAttribute["UPDATED_AT"] = "updated_at";
    /**
     * URL of the End-User's Web page or blog.
     * This Web page SHOULD contain information published by the End-User or an organization that the End-User is affiliated with.
     */
    UserPoolAttribute["WEBSITE"] = "website";
})(UserPoolAttribute = exports.UserPoolAttribute || (exports.UserPoolAttribute = {}));
/**
 * Methods of user sign-in
 */
var SignInType;
(function (SignInType) {
    /**
     * End-user will sign in with a username, with optional aliases
     */
    SignInType[SignInType["USERNAME"] = 0] = "USERNAME";
    /**
     * End-user will sign in using an email address
     */
    SignInType[SignInType["EMAIL"] = 1] = "EMAIL";
    /**
     * End-user will sign in using a phone number
     */
    SignInType[SignInType["PHONE"] = 2] = "PHONE";
    /**
     * End-user will sign in using either an email address or phone number
     */
    SignInType[SignInType["EMAIL_OR_PHONE"] = 3] = "EMAIL_OR_PHONE";
})(SignInType = exports.SignInType || (exports.SignInType = {}));
/**
 * Define a Cognito User Pool
 */
class UserPool extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.userPoolName,
        });
        this.triggers = {};
        let aliasAttributes;
        let usernameAttributes;
        if (props.usernameAliasAttributes != null && props.signInType !== SignInType.USERNAME) {
            throw new Error(`'usernameAliasAttributes' can only be set with a signInType of 'USERNAME'`);
        }
        if (props.usernameAliasAttributes
            && !props.usernameAliasAttributes.every(a => {
                return a === UserPoolAttribute.EMAIL || a === UserPoolAttribute.PHONE_NUMBER || a === UserPoolAttribute.PREFERRED_USERNAME;
            })) {
            throw new Error(`'usernameAliasAttributes' can only include EMAIL, PHONE_NUMBER, or PREFERRED_USERNAME`);
        }
        if (props.autoVerifiedAttributes
            && !props.autoVerifiedAttributes.every(a => a === UserPoolAttribute.EMAIL || a === UserPoolAttribute.PHONE_NUMBER)) {
            throw new Error(`'autoVerifiedAttributes' can only include EMAIL or PHONE_NUMBER`);
        }
        switch (props.signInType) {
            case SignInType.USERNAME:
                aliasAttributes = props.usernameAliasAttributes;
                break;
            case SignInType.EMAIL:
                usernameAttributes = [UserPoolAttribute.EMAIL];
                break;
            case SignInType.PHONE:
                usernameAttributes = [UserPoolAttribute.PHONE_NUMBER];
                break;
            case SignInType.EMAIL_OR_PHONE:
                usernameAttributes = [UserPoolAttribute.EMAIL, UserPoolAttribute.PHONE_NUMBER];
                break;
            default:
                aliasAttributes = props.usernameAliasAttributes;
                break;
        }
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: this.physicalName,
            usernameAttributes,
            aliasAttributes,
            autoVerifiedAttributes: props.autoVerifiedAttributes,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => this.triggers })
        });
        this.userPoolId = this.getResourceNameAttribute(userPool.ref);
        this.userPoolArn = this.getResourceArnAttribute(userPool.attrArn, {
            service: 'cognito',
            resource: 'userpool',
            resourceName: this.physicalName,
        });
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported user pool properties
     */
    static fromUserPoolAttributes(scope, id, attrs) {
        /**
         * Define a user pool which has been declared in another stack
         */
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolId = attrs.userPoolId;
                this.userPoolArn = attrs.userPoolArn;
                this.userPoolProviderName = attrs.userPoolProviderName;
                this.userPoolProviderUrl = attrs.userPoolProviderUrl;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Attach 'Create Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addCreateAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'CreateAuthChallenge');
        this.triggers = { ...this.triggers, createAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Custom Message' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
     * @param fn the lambda function to attach
     */
    addCustomMessageTrigger(fn) {
        this.addLambdaPermission(fn, 'CustomMessage');
        this.triggers = { ...this.triggers, customMessage: fn.functionArn };
    }
    /**
     * Attach 'Define Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addDefineAuthChallengeTrigger(fn) {
        this.addLambdaPermission(fn, 'DefineAuthChallenge');
        this.triggers = { ...this.triggers, defineAuthChallenge: fn.functionArn };
    }
    /**
     * Attach 'Post Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
     * @param fn the lambda function to attach
     */
    addPostAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostAuthentication');
        this.triggers = { ...this.triggers, postAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Post Confirmation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
     * @param fn the lambda function to attach
     */
    addPostConfirmationTrigger(fn) {
        this.addLambdaPermission(fn, 'PostConfirmation');
        this.triggers = { ...this.triggers, postConfirmation: fn.functionArn };
    }
    /**
     * Attach 'Pre Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
     * @param fn the lambda function to attach
     */
    addPreAuthenticationTrigger(fn) {
        this.addLambdaPermission(fn, 'PreAuthentication');
        this.triggers = { ...this.triggers, preAuthentication: fn.functionArn };
    }
    /**
     * Attach 'Pre Sign Up' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
     * @param fn the lambda function to attach
     */
    addPreSignUpTrigger(fn) {
        this.addLambdaPermission(fn, 'PreSignUp');
        this.triggers = { ...this.triggers, preSignUp: fn.functionArn };
    }
    /**
     * Attach 'Verify Auth Challenge Response' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
     * @param fn the lambda function to attach
     */
    addVerifyAuthChallengeResponseTrigger(fn) {
        this.addLambdaPermission(fn, 'VerifyAuthChallengeResponse');
        this.triggers = { ...this.triggers, verifyAuthChallengeResponse: fn.functionArn };
    }
    addLambdaPermission(fn, name) {
        const normalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${normalize}Cognito`, {
            principal: new iam.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn
        });
    }
}
exports.UserPool = UserPool;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidXNlci1wb29sLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXlDO0FBRXpDLHdDQUFxRTtBQUNyRSwyREFBa0Q7QUFFbEQ7Ozs7R0FJRztBQUNILElBQVksaUJBOEdYO0FBOUdELFdBQVksaUJBQWlCO0lBQzNCOztPQUVHO0lBQ0gsd0NBQW1CLENBQUE7SUFFbkI7Ozs7T0FJRztJQUNILDRDQUF1QixDQUFBO0lBRXZCOzs7T0FHRztJQUNILG9DQUFlLENBQUE7SUFFZjs7OztPQUlHO0lBQ0gsZ0RBQTJCLENBQUE7SUFFM0I7O09BRUc7SUFDSCxzQ0FBaUIsQ0FBQTtJQUVqQjs7OztPQUlHO0lBQ0gsOENBQXlCLENBQUE7SUFFekI7Ozs7O09BS0c7SUFDSCxzQ0FBaUIsQ0FBQTtJQUVqQjs7Ozs7T0FLRztJQUNILGdEQUEyQixDQUFBO0lBRTNCOzs7T0FHRztJQUNILGtDQUFhLENBQUE7SUFFYjs7O09BR0c7SUFDSCwwQ0FBcUIsQ0FBQTtJQUVyQjs7Ozs7T0FLRztJQUNILGtEQUE2QixDQUFBO0lBRTdCOzs7Ozs7T0FNRztJQUNILHdDQUFtQixDQUFBO0lBRW5COztPQUVHO0lBQ0gsOERBQXlDLENBQUE7SUFFekM7O09BRUc7SUFDSCx3Q0FBbUIsQ0FBQTtJQUVuQjs7T0FFRztJQUNILDBDQUFxQixDQUFBO0lBRXJCOzs7O09BSUc7SUFDSCw4Q0FBeUIsQ0FBQTtJQUV6Qjs7O09BR0c7SUFDSCx3Q0FBbUIsQ0FBQTtBQUNyQixDQUFDLEVBOUdXLGlCQUFpQixHQUFqQix5QkFBaUIsS0FBakIseUJBQWlCLFFBOEc1QjtBQUVEOztHQUVHO0FBQ0gsSUFBWSxVQW9CWDtBQXBCRCxXQUFZLFVBQVU7SUFDcEI7O09BRUc7SUFDSCxtREFBUSxDQUFBO0lBRVI7O09BRUc7SUFDSCw2Q0FBSyxDQUFBO0lBRUw7O09BRUc7SUFDSCw2Q0FBSyxDQUFBO0lBRUw7O09BRUc7SUFDSCwrREFBYyxDQUFBO0FBQ2hCLENBQUMsRUFwQlcsVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFvQnJCO0FBaUpEOztHQUVHO0FBQ0gsTUFBYSxRQUFTLFNBQVEsZUFBUTtJQTJDcEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUF1QixFQUFFO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1NBQ2pDLENBQUMsQ0FBQztRQUxHLGFBQVEsR0FBcUMsRUFBRyxDQUFDO1FBT3ZELElBQUksZUFBZ0QsQ0FBQztRQUNyRCxJQUFJLGtCQUFtRCxDQUFDO1FBRXhELElBQUksS0FBSyxDQUFDLHVCQUF1QixJQUFJLElBQUksSUFBSSxLQUFLLENBQUMsVUFBVSxLQUFLLFVBQVUsQ0FBQyxRQUFRLEVBQUU7WUFDckYsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQzlGO1FBRUQsSUFBSSxLQUFLLENBQUMsdUJBQXVCO2VBQzVCLENBQUMsS0FBSyxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDMUMsT0FBTyxDQUFDLEtBQUssaUJBQWlCLENBQUMsS0FBSyxJQUFJLENBQUMsS0FBSyxpQkFBaUIsQ0FBQyxZQUFZLElBQUksQ0FBQyxLQUFLLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDO1lBQzdILENBQUMsQ0FBQyxFQUFFO1lBQ0osTUFBTSxJQUFJLEtBQUssQ0FBQyx1RkFBdUYsQ0FBQyxDQUFDO1NBQzFHO1FBRUQsSUFBSSxLQUFLLENBQUMsc0JBQXNCO2VBQzNCLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxpQkFBaUIsQ0FBQyxLQUFLLElBQUksQ0FBQyxLQUFLLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQ3BILE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLENBQUMsQ0FBQztTQUNwRjtRQUVELFFBQVEsS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUN4QixLQUFLLFVBQVUsQ0FBQyxRQUFRO2dCQUN0QixlQUFlLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixDQUFDO2dCQUNoRCxNQUFNO1lBRVIsS0FBSyxVQUFVLENBQUMsS0FBSztnQkFDbkIsa0JBQWtCLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDL0MsTUFBTTtZQUVSLEtBQUssVUFBVSxDQUFDLEtBQUs7Z0JBQ25CLGtCQUFrQixHQUFHLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUM7Z0JBQ3RELE1BQU07WUFFUixLQUFLLFVBQVUsQ0FBQyxjQUFjO2dCQUM1QixrQkFBa0IsR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDL0UsTUFBTTtZQUVSO2dCQUNFLGVBQWUsR0FBRyxLQUFLLENBQUMsdUJBQXVCLENBQUM7Z0JBQ2hELE1BQU07U0FDVDtRQUVELElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUN4QixLQUFLLE1BQU0sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUNqRCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN4QyxJQUFJLE9BQU8sS0FBSyxTQUFTLEVBQUU7b0JBQ3pCLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxPQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDO29CQUN4RCxJQUFJLENBQUMsUUFBZ0IsQ0FBQyxDQUFDLENBQUMsR0FBSSxPQUE0QixDQUFDLFdBQVcsQ0FBQztpQkFDdkU7YUFDRjtTQUNGO1FBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSwrQkFBVyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDakQsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQy9CLGtCQUFrQjtZQUNsQixlQUFlO1lBQ2Ysc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxZQUFZLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7U0FDOUQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzlELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7WUFDaEUsT0FBTyxFQUFFLFNBQVM7WUFDbEIsUUFBUSxFQUFFLFVBQVU7WUFDcEIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQkFBb0IsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUM7UUFDdEQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLFFBQVEsQ0FBQyxlQUFlLENBQUM7SUFDdEQsQ0FBQztJQW5IRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF5QjtRQUMxRjs7V0FFRztRQUNILE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixlQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztnQkFDOUIsZ0JBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO2dCQUNoQyx5QkFBb0IsR0FBRyxLQUFLLENBQUMsb0JBQW9CLENBQUM7Z0JBQ2xELHdCQUFtQixHQUFHLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQztZQUNsRSxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBbUdEOzs7OztPQUtHO0lBQ0ksNkJBQTZCLENBQUMsRUFBb0I7UUFDdkQsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxRQUFRLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLEVBQUUsbUJBQW1CLEVBQUUsRUFBRSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQzVFLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLHVCQUF1QixDQUFDLEVBQW9CO1FBQ2pELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLFFBQVEsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLFFBQVEsRUFBRSxhQUFhLEVBQUUsRUFBRSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ3RFLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLDZCQUE2QixDQUFDLEVBQW9CO1FBQ3ZELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUNwRCxJQUFJLENBQUMsUUFBUSxHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLG1CQUFtQixFQUFFLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSw0QkFBNEIsQ0FBQyxFQUFvQjtRQUN0RCxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxFQUFFLG9CQUFvQixDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLFFBQVEsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLFFBQVEsRUFBRSxrQkFBa0IsRUFBRSxFQUFFLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksMEJBQTBCLENBQUMsRUFBb0I7UUFDcEQsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxRQUFRLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLDJCQUEyQixDQUFDLEVBQW9CO1FBQ3JELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsUUFBUSxHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUMxRSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxtQkFBbUIsQ0FBQyxFQUFvQjtRQUM3QyxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxRQUFRLEdBQUcsRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFLEVBQUUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxxQ0FBcUMsQ0FBQyxFQUFvQjtRQUMvRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxFQUFFLDZCQUE2QixDQUFDLENBQUM7UUFDNUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLFFBQVEsRUFBRSwyQkFBMkIsRUFBRSxFQUFFLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDcEYsQ0FBQztJQUVPLG1CQUFtQixDQUFDLEVBQW9CLEVBQUUsSUFBWTtRQUM1RCxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDL0QsRUFBRSxDQUFDLGFBQWEsQ0FBQyxHQUFHLFNBQVMsU0FBUyxFQUFFO1lBQ3RDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQywyQkFBMkIsQ0FBQztZQUNoRSxTQUFTLEVBQUUsSUFBSSxDQUFDLFdBQVc7U0FDNUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBck5ELDRCQXFOQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBpYW0gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtaWFtJyk7XG5pbXBvcnQgbGFtYmRhID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWxhbWJkYScpO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5Vc2VyUG9vbCB9IGZyb20gJy4vY29nbml0by5nZW5lcmF0ZWQnO1xuXG4vKipcbiAqIFN0YW5kYXJkIGF0dHJpYnV0ZXNcbiAqIFNwZWNpZmllZCBmb2xsb3dpbmcgdGhlIE9wZW5JRCBDb25uZWN0IHNwZWNcbiAqIEBzZWUgaHR0cHM6Ly9vcGVuaWQubmV0L3NwZWNzL29wZW5pZC1jb25uZWN0LWNvcmUtMV8wLmh0bWwjU3RhbmRhcmRDbGFpbXNcbiAqL1xuZXhwb3J0IGVudW0gVXNlclBvb2xBdHRyaWJ1dGUge1xuICAvKipcbiAgICogRW5kLVVzZXIncyBwcmVmZXJyZWQgcG9zdGFsIGFkZHJlc3MuXG4gICAqL1xuICBBRERSRVNTID0gJ2FkZHJlc3MnLFxuXG4gIC8qKlxuICAgKiBFbmQtVXNlcidzIGJpcnRoZGF5LCByZXByZXNlbnRlZCBhcyBhbiBJU08gODYwMToyMDA0IFtJU084NjAx4oCRMjAwNF0gWVlZWS1NTS1ERCBmb3JtYXQuXG4gICAqIFRoZSB5ZWFyIE1BWSBiZSAwMDAwLCBpbmRpY2F0aW5nIHRoYXQgaXQgaXMgb21pdHRlZC5cbiAgICogVG8gcmVwcmVzZW50IG9ubHkgdGhlIHllYXIsIFlZWVkgZm9ybWF0IGlzIGFsbG93ZWQuXG4gICAqL1xuICBCSVJUSERBVEUgPSAnYmlydGhkYXRlJyxcblxuICAvKipcbiAgICogRW5kLVVzZXIncyBwcmVmZXJyZWQgZS1tYWlsIGFkZHJlc3MuXG4gICAqIEl0cyB2YWx1ZSBNVVNUIGNvbmZvcm0gdG8gdGhlIFJGQyA1MzIyIFtSRkM1MzIyXSBhZGRyLXNwZWMgc3ludGF4LlxuICAgKi9cbiAgRU1BSUwgPSAnZW1haWwnLFxuXG4gIC8qKlxuICAgKiBTdXJuYW1lKHMpIG9yIGxhc3QgbmFtZShzKSBvZiB0aGUgRW5kLVVzZXIuXG4gICAqIE5vdGUgdGhhdCBpbiBzb21lIGN1bHR1cmVzLCBwZW9wbGUgY2FuIGhhdmUgbXVsdGlwbGUgZmFtaWx5IG5hbWVzIG9yIG5vIGZhbWlseSBuYW1lO1xuICAgKiBhbGwgY2FuIGJlIHByZXNlbnQsIHdpdGggdGhlIG5hbWVzIGJlaW5nIHNlcGFyYXRlZCBieSBzcGFjZSBjaGFyYWN0ZXJzLlxuICAgKi9cbiAgRkFNSUxZX05BTUUgPSAnZmFtaWx5X25hbWUnLFxuXG4gIC8qKlxuICAgKiBFbmQtVXNlcidzIGdlbmRlci5cbiAgICovXG4gIEdFTkRFUiA9ICdnZW5kZXInLFxuXG4gIC8qKlxuICAgKiBHaXZlbiBuYW1lKHMpIG9yIGZpcnN0IG5hbWUocykgb2YgdGhlIEVuZC1Vc2VyLlxuICAgKiBOb3RlIHRoYXQgaW4gc29tZSBjdWx0dXJlcywgcGVvcGxlIGNhbiBoYXZlIG11bHRpcGxlIGdpdmVuIG5hbWVzO1xuICAgKiBhbGwgY2FuIGJlIHByZXNlbnQsIHdpdGggdGhlIG5hbWVzIGJlaW5nIHNlcGFyYXRlZCBieSBzcGFjZSBjaGFyYWN0ZXJzLlxuICAgKi9cbiAgR0lWRU5fTkFNRSA9ICdnaXZlbl9uYW1lJyxcblxuICAvKipcbiAgICogRW5kLVVzZXIncyBsb2NhbGUsIHJlcHJlc2VudGVkIGFzIGEgQkNQNDcgW1JGQzU2NDZdIGxhbmd1YWdlIHRhZy5cbiAgICogVGhpcyBpcyB0eXBpY2FsbHkgYW4gSVNPIDYzOS0xIEFscGhhLTIgW0lTTzYzOeKAkTFdIGxhbmd1YWdlIGNvZGUgaW4gbG93ZXJjYXNlXG4gICAqIGFuZCBhbiBJU08gMzE2Ni0xIEFscGhhLTIgW0lTTzMxNjbigJExXSBjb3VudHJ5IGNvZGUgaW4gdXBwZXJjYXNlLCBzZXBhcmF0ZWQgYnkgYSBkYXNoLlxuICAgKiBGb3IgZXhhbXBsZSwgZW4tVVMgb3IgZnItQ0EuXG4gICAqL1xuICBMT0NBTEUgPSAnbG9jYWxlJyxcblxuICAvKipcbiAgICogTWlkZGxlIG5hbWUocykgb2YgdGhlIEVuZC1Vc2VyLlxuICAgKiBOb3RlIHRoYXQgaW4gc29tZSBjdWx0dXJlcywgcGVvcGxlIGNhbiBoYXZlIG11bHRpcGxlIG1pZGRsZSBuYW1lcztcbiAgICogYWxsIGNhbiBiZSBwcmVzZW50LCB3aXRoIHRoZSBuYW1lcyBiZWluZyBzZXBhcmF0ZWQgYnkgc3BhY2UgY2hhcmFjdGVycy5cbiAgICogQWxzbyBub3RlIHRoYXQgaW4gc29tZSBjdWx0dXJlcywgbWlkZGxlIG5hbWVzIGFyZSBub3QgdXNlZC5cbiAgICovXG4gIE1JRERMRV9OQU1FID0gJ21pZGRsZV9uYW1lJyxcblxuICAvKipcbiAgICogRW5kLVVzZXIncyBmdWxsIG5hbWUgaW4gZGlzcGxheWFibGUgZm9ybSBpbmNsdWRpbmcgYWxsIG5hbWUgcGFydHMsXG4gICAqIHBvc3NpYmx5IGluY2x1ZGluZyB0aXRsZXMgYW5kIHN1ZmZpeGVzLCBvcmRlcmVkIGFjY29yZGluZyB0byB0aGUgRW5kLVVzZXIncyBsb2NhbGUgYW5kIHByZWZlcmVuY2VzLlxuICAgKi9cbiAgTkFNRSA9ICduYW1lJyxcblxuICAvKipcbiAgICogQ2FzdWFsIG5hbWUgb2YgdGhlIEVuZC1Vc2VyIHRoYXQgbWF5IG9yIG1heSBub3QgYmUgdGhlIHNhbWUgYXMgdGhlIGdpdmVuX25hbWUuXG4gICAqIEZvciBpbnN0YW5jZSwgYSBuaWNrbmFtZSB2YWx1ZSBvZiBNaWtlIG1pZ2h0IGJlIHJldHVybmVkIGFsb25nc2lkZSBhIGdpdmVuX25hbWUgdmFsdWUgb2YgTWljaGFlbC5cbiAgICovXG4gIE5JQ0tOQU1FID0gJ25pY2tuYW1lJyxcblxuICAvKipcbiAgICogRW5kLVVzZXIncyBwcmVmZXJyZWQgdGVsZXBob25lIG51bWJlci5cbiAgICogRS4xNjQgW0UuMTY0XSBpcyBSRUNPTU1FTkRFRCBhcyB0aGUgZm9ybWF0IG9mIHRoaXMgQ2xhaW0sIGZvciBleGFtcGxlLCArMSAoNDI1KSA1NTUtMTIxMiBvciArNTYgKDIpIDY4NyAyNDAwLlxuICAgKiBJZiB0aGUgcGhvbmUgbnVtYmVyIGNvbnRhaW5zIGFuIGV4dGVuc2lvbiwgaXQgaXMgUkVDT01NRU5ERUQgdGhhdCB0aGUgZXh0ZW5zaW9uIGJlIHJlcHJlc2VudGVkIHVzaW5nIHRoZVxuICAgKiBSRkMgMzk2NiBbUkZDMzk2Nl0gZXh0ZW5zaW9uIHN5bnRheCwgZm9yIGV4YW1wbGUsICsxICg2MDQpIDU1NS0xMjM0O2V4dD01Njc4LlxuICAgKi9cbiAgUEhPTkVfTlVNQkVSID0gJ3Bob25lX251bWJlcicsXG5cbiAgLyoqXG4gICAqIFVSTCBvZiB0aGUgRW5kLVVzZXIncyBwcm9maWxlIHBpY3R1cmUuXG4gICAqIFRoaXMgVVJMIE1VU1QgcmVmZXIgdG8gYW4gaW1hZ2UgZmlsZSAoZm9yIGV4YW1wbGUsIGEgUE5HLCBKUEVHLCBvciBHSUYgaW1hZ2UgZmlsZSksXG4gICAqIHJhdGhlciB0aGFuIHRvIGEgV2ViIHBhZ2UgY29udGFpbmluZyBhbiBpbWFnZS5cbiAgICogTm90ZSB0aGF0IHRoaXMgVVJMIFNIT1VMRCBzcGVjaWZpY2FsbHkgcmVmZXJlbmNlIGEgcHJvZmlsZSBwaG90byBvZiB0aGUgRW5kLVVzZXJcbiAgICogc3VpdGFibGUgZm9yIGRpc3BsYXlpbmcgd2hlbiBkZXNjcmliaW5nIHRoZSBFbmQtVXNlciwgcmF0aGVyIHRoYW4gYW4gYXJiaXRyYXJ5IHBob3RvIHRha2VuIGJ5IHRoZSBFbmQtVXNlclxuICAgKi9cbiAgUElDVFVSRSA9ICdwaWN0dXJlJyxcblxuICAvKipcbiAgICogU2hvcnRoYW5kIG5hbWUgYnkgd2hpY2ggdGhlIEVuZC1Vc2VyIHdpc2hlcyB0byBiZSByZWZlcnJlZCB0by5cbiAgICovXG4gIFBSRUZFUlJFRF9VU0VSTkFNRSA9ICdwcmVmZXJyZWRfdXNlcm5hbWUnLFxuXG4gIC8qKlxuICAgKiBVUkwgb2YgdGhlIEVuZC1Vc2VyJ3MgcHJvZmlsZSBwYWdlLiBUaGUgY29udGVudHMgb2YgdGhpcyBXZWIgcGFnZSBTSE9VTEQgYmUgYWJvdXQgdGhlIEVuZC1Vc2VyLlxuICAgKi9cbiAgUFJPRklMRSA9ICdwcm9maWxlJyxcblxuICAvKipcbiAgICogVGhlIEVuZC1Vc2VyJ3MgdGltZSB6b25lXG4gICAqL1xuICBUSU1FWk9ORSA9ICd0aW1lem9uZScsXG5cbiAgLyoqXG4gICAqIFRpbWUgdGhlIEVuZC1Vc2VyJ3MgaW5mb3JtYXRpb24gd2FzIGxhc3QgdXBkYXRlZC5cbiAgICogSXRzIHZhbHVlIGlzIGEgSlNPTiBudW1iZXIgcmVwcmVzZW50aW5nIHRoZSBudW1iZXIgb2Ygc2Vjb25kcyBmcm9tIDE5NzAtMDEtMDFUMDowOjBaXG4gICAqIGFzIG1lYXN1cmVkIGluIFVUQyB1bnRpbCB0aGUgZGF0ZS90aW1lLlxuICAgKi9cbiAgVVBEQVRFRF9BVCA9ICd1cGRhdGVkX2F0JyxcblxuICAvKipcbiAgICogVVJMIG9mIHRoZSBFbmQtVXNlcidzIFdlYiBwYWdlIG9yIGJsb2cuXG4gICAqIFRoaXMgV2ViIHBhZ2UgU0hPVUxEIGNvbnRhaW4gaW5mb3JtYXRpb24gcHVibGlzaGVkIGJ5IHRoZSBFbmQtVXNlciBvciBhbiBvcmdhbml6YXRpb24gdGhhdCB0aGUgRW5kLVVzZXIgaXMgYWZmaWxpYXRlZCB3aXRoLlxuICAgKi9cbiAgV0VCU0lURSA9ICd3ZWJzaXRlJ1xufVxuXG4vKipcbiAqIE1ldGhvZHMgb2YgdXNlciBzaWduLWluXG4gKi9cbmV4cG9ydCBlbnVtIFNpZ25JblR5cGUge1xuICAvKipcbiAgICogRW5kLXVzZXIgd2lsbCBzaWduIGluIHdpdGggYSB1c2VybmFtZSwgd2l0aCBvcHRpb25hbCBhbGlhc2VzXG4gICAqL1xuICBVU0VSTkFNRSxcblxuICAvKipcbiAgICogRW5kLXVzZXIgd2lsbCBzaWduIGluIHVzaW5nIGFuIGVtYWlsIGFkZHJlc3NcbiAgICovXG4gIEVNQUlMLFxuXG4gIC8qKlxuICAgKiBFbmQtdXNlciB3aWxsIHNpZ24gaW4gdXNpbmcgYSBwaG9uZSBudW1iZXJcbiAgICovXG4gIFBIT05FLFxuXG4gIC8qKlxuICAgKiBFbmQtdXNlciB3aWxsIHNpZ24gaW4gdXNpbmcgZWl0aGVyIGFuIGVtYWlsIGFkZHJlc3Mgb3IgcGhvbmUgbnVtYmVyXG4gICAqL1xuICBFTUFJTF9PUl9QSE9ORVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sVHJpZ2dlcnMge1xuICAvKipcbiAgICogQ3JlYXRlcyBhbiBhdXRoZW50aWNhdGlvbiBjaGFsbGVuZ2UuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtY3JlYXRlLWF1dGgtY2hhbGxlbmdlLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IGNyZWF0ZUF1dGhDaGFsbGVuZ2U/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuXG4gIC8qKlxuICAgKiBBIGN1c3RvbSBNZXNzYWdlIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1jdXN0b20tbWVzc2FnZS5odG1sXG4gICAqL1xuICByZWFkb25seSBjdXN0b21NZXNzYWdlPzogbGFtYmRhLklGdW5jdGlvbjtcblxuICAvKipcbiAgICogRGVmaW5lcyB0aGUgYXV0aGVudGljYXRpb24gY2hhbGxlbmdlLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLWRlZmluZS1hdXRoLWNoYWxsZW5nZS5odG1sXG4gICAqL1xuICByZWFkb25seSBkZWZpbmVBdXRoQ2hhbGxlbmdlPzogbGFtYmRhLklGdW5jdGlvbjtcblxuICAvKipcbiAgICogQSBwb3N0LWF1dGhlbnRpY2F0aW9uIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wb3N0LWF1dGhlbnRpY2F0aW9uLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHBvc3RBdXRoZW50aWNhdGlvbj86IGxhbWJkYS5JRnVuY3Rpb247XG5cbiAgLyoqXG4gICAqIEEgcG9zdC1jb25maXJtYXRpb24gQVdTIExhbWJkYSB0cmlnZ2VyLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXBvc3QtY29uZmlybWF0aW9uLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHBvc3RDb25maXJtYXRpb24/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuXG4gIC8qKlxuICAgKiBBIHByZS1hdXRoZW50aWNhdGlvbiBBV1MgTGFtYmRhIHRyaWdnZXIuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtcHJlLWF1dGhlbnRpY2F0aW9uLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHByZUF1dGhlbnRpY2F0aW9uPzogbGFtYmRhLklGdW5jdGlvbjtcblxuICAvKipcbiAgICogQSBwcmUtcmVnaXN0cmF0aW9uIEFXUyBMYW1iZGEgdHJpZ2dlci5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wcmUtc2lnbi11cC5odG1sXG4gICAqL1xuICByZWFkb25seSBwcmVTaWduVXA/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuXG4gIC8qKlxuICAgKiBWZXJpZmllcyB0aGUgYXV0aGVudGljYXRpb24gY2hhbGxlbmdlIHJlc3BvbnNlLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXZlcmlmeS1hdXRoLWNoYWxsZW5nZS1yZXNwb25zZS5odG1sXG4gICAqL1xuICByZWFkb25seSB2ZXJpZnlBdXRoQ2hhbGxlbmdlUmVzcG9uc2U/OiBsYW1iZGEuSUZ1bmN0aW9uO1xuXG4gIC8qKlxuICAgKiBJbmRleCBzaWduYXR1cmVcbiAgICovXG4gIFt0cmlnZ2VyOiBzdHJpbmddOiBsYW1iZGEuSUZ1bmN0aW9uIHwgdW5kZWZpbmVkO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sUHJvcHMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgdXNlciBwb29sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYXV0b21hdGljYWxseSBnZW5lcmF0ZWQgbmFtZSBieSBDbG91ZEZvcm1hdGlvbiBhdCBkZXBsb3kgdGltZVxuICAgKi9cbiAgcmVhZG9ubHkgdXNlclBvb2xOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBNZXRob2QgdXNlZCBmb3IgdXNlciByZWdpc3RyYXRpb24gJiBzaWduIGluLlxuICAgKiBBbGxvd3MgZWl0aGVyIHVzZXJuYW1lIHdpdGggYWxpYXNlcyBPUiBzaWduIGluIHdpdGggZW1haWwsIHBob25lLCBvciBib3RoLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTaWduSW5UeXBlLlVzZXJuYW1lXG4gICAqL1xuICByZWFkb25seSBzaWduSW5UeXBlPzogU2lnbkluVHlwZTtcblxuICAvKipcbiAgICogQXR0cmlidXRlcyB0byBhbGxvdyBhcyB1c2VybmFtZSBhbGlhcy5cbiAgICogT25seSB2YWxpZCBpZiBzaWduSW5UeXBlIGlzIFVTRVJOQU1FXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWxpYXMuXG4gICAqL1xuICByZWFkb25seSB1c2VybmFtZUFsaWFzQXR0cmlidXRlcz86IFVzZXJQb29sQXR0cmlidXRlW107XG5cbiAgLyoqXG4gICAqIEF0dHJpYnV0ZXMgd2hpY2ggQ29nbml0byB3aWxsIGF1dG9tYXRpY2FsbHkgc2VuZCBhIHZlcmlmaWNhdGlvbiBtZXNzYWdlIHRvLlxuICAgKiBNdXN0IGJlIGVpdGhlciBFTUFJTCwgUEhPTkUsIG9yIGJvdGguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYXV0byB2ZXJpZmljYXRpb24uXG4gICAqL1xuICByZWFkb25seSBhdXRvVmVyaWZpZWRBdHRyaWJ1dGVzPzogVXNlclBvb2xBdHRyaWJ1dGVbXTtcblxuICAvKipcbiAgICogTGFtYmRhIGZ1bmN0aW9ucyB0byB1c2UgZm9yIHN1cHBvcnRlZCBDb2duaXRvIHRyaWdnZXJzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIExhbWJkYSB0cmlnZ2Vycy5cbiAgICovXG4gIHJlYWRvbmx5IGxhbWJkYVRyaWdnZXJzPzogVXNlclBvb2xUcmlnZ2Vycztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbEF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIElEIG9mIGFuIGV4aXN0aW5nIHVzZXIgcG9vbFxuICAgKi9cbiAgcmVhZG9ubHkgdXNlclBvb2xJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBpbXBvcnRlZCB1c2VyIHBvb2xcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwcm92aWRlciBuYW1lIG9mIHRoZSBpbXBvcnRlZCB1c2VyIHBvb2xcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sUHJvdmlkZXJOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBVUkwgb2YgdGhlIGltcG9ydGVkIHVzZXIgcG9vbFxuICAgKi9cbiAgcmVhZG9ubHkgdXNlclBvb2xQcm92aWRlclVybDogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIElVc2VyUG9vbCBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgcGh5c2ljYWwgSUQgb2YgdGhpcyB1c2VyIHBvb2wgcmVzb3VyY2VcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdXNlclBvb2xJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgdXNlciBwb29sIHJlc291cmNlXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwcm92aWRlciBuYW1lIG9mIHRoaXMgdXNlciBwb29sIHJlc291cmNlXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sUHJvdmlkZXJOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwcm92aWRlciBVUkwgb2YgdGhpcyB1c2VyIHBvb2wgcmVzb3VyY2VcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdXNlclBvb2xQcm92aWRlclVybDogc3RyaW5nO1xufVxuXG4vKipcbiAqIERlZmluZSBhIENvZ25pdG8gVXNlciBQb29sXG4gKi9cbmV4cG9ydCBjbGFzcyBVc2VyUG9vbCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVVzZXJQb29sIHtcbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyB1c2VyIHBvb2wgcmVzb3VyY2VcbiAgICogQHBhcmFtIHNjb3BlIFBhcmVudCBjb25zdHJ1Y3RcbiAgICogQHBhcmFtIGlkIENvbnN0cnVjdCBJRFxuICAgKiBAcGFyYW0gYXR0cnMgSW1wb3J0ZWQgdXNlciBwb29sIHByb3BlcnRpZXNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVVzZXJQb29sQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogVXNlclBvb2xBdHRyaWJ1dGVzKTogSVVzZXJQb29sIHtcbiAgICAvKipcbiAgICAgKiBEZWZpbmUgYSB1c2VyIHBvb2wgd2hpY2ggaGFzIGJlZW4gZGVjbGFyZWQgaW4gYW5vdGhlciBzdGFja1xuICAgICAqL1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVVzZXJQb29sIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbElkID0gYXR0cnMudXNlclBvb2xJZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbEFybiA9IGF0dHJzLnVzZXJQb29sQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHVzZXJQb29sUHJvdmlkZXJOYW1lID0gYXR0cnMudXNlclBvb2xQcm92aWRlck5hbWU7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgdXNlclBvb2xQcm92aWRlclVybCA9IGF0dHJzLnVzZXJQb29sUHJvdmlkZXJVcmw7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgcGh5c2ljYWwgSUQgb2YgdGhpcyB1c2VyIHBvb2wgcmVzb3VyY2VcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbElkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIHVzZXIgcG9vbFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHVzZXJQb29sQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFVzZXIgcG9vbCBwcm92aWRlciBuYW1lXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdXNlclBvb2xQcm92aWRlck5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVXNlciBwb29sIHByb3ZpZGVyIFVSTFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHVzZXJQb29sUHJvdmlkZXJVcmw6IHN0cmluZztcblxuICBwcml2YXRlIHRyaWdnZXJzOiBDZm5Vc2VyUG9vbC5MYW1iZGFDb25maWdQcm9wZXJ0eSA9IHsgfTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVXNlclBvb2xQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnVzZXJQb29sTmFtZSxcbiAgICB9KTtcblxuICAgIGxldCBhbGlhc0F0dHJpYnV0ZXM6IFVzZXJQb29sQXR0cmlidXRlW10gfCB1bmRlZmluZWQ7XG4gICAgbGV0IHVzZXJuYW1lQXR0cmlidXRlczogVXNlclBvb2xBdHRyaWJ1dGVbXSB8IHVuZGVmaW5lZDtcblxuICAgIGlmIChwcm9wcy51c2VybmFtZUFsaWFzQXR0cmlidXRlcyAhPSBudWxsICYmIHByb3BzLnNpZ25JblR5cGUgIT09IFNpZ25JblR5cGUuVVNFUk5BTUUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJ3VzZXJuYW1lQWxpYXNBdHRyaWJ1dGVzJyBjYW4gb25seSBiZSBzZXQgd2l0aCBhIHNpZ25JblR5cGUgb2YgJ1VTRVJOQU1FJ2ApO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy51c2VybmFtZUFsaWFzQXR0cmlidXRlc1xuICAgICAgJiYgIXByb3BzLnVzZXJuYW1lQWxpYXNBdHRyaWJ1dGVzLmV2ZXJ5KGEgPT4ge1xuICAgICAgICByZXR1cm4gYSA9PT0gVXNlclBvb2xBdHRyaWJ1dGUuRU1BSUwgfHwgYSA9PT0gVXNlclBvb2xBdHRyaWJ1dGUuUEhPTkVfTlVNQkVSIHx8IGEgPT09IFVzZXJQb29sQXR0cmlidXRlLlBSRUZFUlJFRF9VU0VSTkFNRTtcbiAgICAgIH0pKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCd1c2VybmFtZUFsaWFzQXR0cmlidXRlcycgY2FuIG9ubHkgaW5jbHVkZSBFTUFJTCwgUEhPTkVfTlVNQkVSLCBvciBQUkVGRVJSRURfVVNFUk5BTUVgKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuYXV0b1ZlcmlmaWVkQXR0cmlidXRlc1xuICAgICAgJiYgIXByb3BzLmF1dG9WZXJpZmllZEF0dHJpYnV0ZXMuZXZlcnkoYSA9PiBhID09PSBVc2VyUG9vbEF0dHJpYnV0ZS5FTUFJTCB8fCBhID09PSBVc2VyUG9vbEF0dHJpYnV0ZS5QSE9ORV9OVU1CRVIpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCdhdXRvVmVyaWZpZWRBdHRyaWJ1dGVzJyBjYW4gb25seSBpbmNsdWRlIEVNQUlMIG9yIFBIT05FX05VTUJFUmApO1xuICAgIH1cblxuICAgIHN3aXRjaCAocHJvcHMuc2lnbkluVHlwZSkge1xuICAgICAgY2FzZSBTaWduSW5UeXBlLlVTRVJOQU1FOlxuICAgICAgICBhbGlhc0F0dHJpYnV0ZXMgPSBwcm9wcy51c2VybmFtZUFsaWFzQXR0cmlidXRlcztcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgU2lnbkluVHlwZS5FTUFJTDpcbiAgICAgICAgdXNlcm5hbWVBdHRyaWJ1dGVzID0gW1VzZXJQb29sQXR0cmlidXRlLkVNQUlMXTtcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgU2lnbkluVHlwZS5QSE9ORTpcbiAgICAgICAgdXNlcm5hbWVBdHRyaWJ1dGVzID0gW1VzZXJQb29sQXR0cmlidXRlLlBIT05FX05VTUJFUl07XG4gICAgICAgIGJyZWFrO1xuXG4gICAgICBjYXNlIFNpZ25JblR5cGUuRU1BSUxfT1JfUEhPTkU6XG4gICAgICAgIHVzZXJuYW1lQXR0cmlidXRlcyA9IFtVc2VyUG9vbEF0dHJpYnV0ZS5FTUFJTCwgVXNlclBvb2xBdHRyaWJ1dGUuUEhPTkVfTlVNQkVSXTtcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIGFsaWFzQXR0cmlidXRlcyA9IHByb3BzLnVzZXJuYW1lQWxpYXNBdHRyaWJ1dGVzO1xuICAgICAgICBicmVhaztcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubGFtYmRhVHJpZ2dlcnMpIHtcbiAgICAgIGZvciAoY29uc3QgdCBvZiBPYmplY3Qua2V5cyhwcm9wcy5sYW1iZGFUcmlnZ2VycykpIHtcbiAgICAgICAgY29uc3QgdHJpZ2dlciA9IHByb3BzLmxhbWJkYVRyaWdnZXJzW3RdO1xuICAgICAgICBpZiAodHJpZ2dlciAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKHRyaWdnZXIgYXMgbGFtYmRhLklGdW5jdGlvbiwgdCk7XG4gICAgICAgICAgKHRoaXMudHJpZ2dlcnMgYXMgYW55KVt0XSA9ICh0cmlnZ2VyIGFzIGxhbWJkYS5JRnVuY3Rpb24pLmZ1bmN0aW9uQXJuO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgdXNlclBvb2wgPSBuZXcgQ2ZuVXNlclBvb2wodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgdXNlclBvb2xOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIHVzZXJuYW1lQXR0cmlidXRlcyxcbiAgICAgIGFsaWFzQXR0cmlidXRlcyxcbiAgICAgIGF1dG9WZXJpZmllZEF0dHJpYnV0ZXM6IHByb3BzLmF1dG9WZXJpZmllZEF0dHJpYnV0ZXMsXG4gICAgICBsYW1iZGFDb25maWc6IExhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnRyaWdnZXJzIH0pXG4gICAgfSk7XG5cbiAgICB0aGlzLnVzZXJQb29sSWQgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZSh1c2VyUG9vbC5yZWYpO1xuICAgIHRoaXMudXNlclBvb2xBcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKHVzZXJQb29sLmF0dHJBcm4sIHtcbiAgICAgIHNlcnZpY2U6ICdjb2duaXRvJyxcbiAgICAgIHJlc291cmNlOiAndXNlcnBvb2wnLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMudXNlclBvb2xQcm92aWRlck5hbWUgPSB1c2VyUG9vbC5hdHRyUHJvdmlkZXJOYW1lO1xuICAgIHRoaXMudXNlclBvb2xQcm92aWRlclVybCA9IHVzZXJQb29sLmF0dHJQcm92aWRlclVybDtcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2ggJ0NyZWF0ZSBBdXRoIENoYWxsZW5nZScgdHJpZ2dlclxuICAgKiBHcmFudHMgYWNjZXNzIGZyb20gY29nbml0by1pZHAuYW1hem9uYXdzLmNvbSB0byB0aGUgbGFtYmRhXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtY3JlYXRlLWF1dGgtY2hhbGxlbmdlLmh0bWxcbiAgICogQHBhcmFtIGZuIHRoZSBsYW1iZGEgZnVuY3Rpb24gdG8gYXR0YWNoXG4gICAqL1xuICBwdWJsaWMgYWRkQ3JlYXRlQXV0aENoYWxsZW5nZVRyaWdnZXIoZm46IGxhbWJkYS5JRnVuY3Rpb24pOiB2b2lkIHtcbiAgICB0aGlzLmFkZExhbWJkYVBlcm1pc3Npb24oZm4sICdDcmVhdGVBdXRoQ2hhbGxlbmdlJyk7XG4gICAgdGhpcy50cmlnZ2VycyA9IHsgLi4udGhpcy50cmlnZ2VycywgY3JlYXRlQXV0aENoYWxsZW5nZTogZm4uZnVuY3Rpb25Bcm4gfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2ggJ0N1c3RvbSBNZXNzYWdlJyB0cmlnZ2VyXG4gICAqIEdyYW50cyBhY2Nlc3MgZnJvbSBjb2duaXRvLWlkcC5hbWF6b25hd3MuY29tIHRvIHRoZSBsYW1iZGFcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1jdXN0b20tbWVzc2FnZS5odG1sXG4gICAqIEBwYXJhbSBmbiB0aGUgbGFtYmRhIGZ1bmN0aW9uIHRvIGF0dGFjaFxuICAgKi9cbiAgcHVibGljIGFkZEN1c3RvbU1lc3NhZ2VUcmlnZ2VyKGZuOiBsYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XG4gICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKGZuLCAnQ3VzdG9tTWVzc2FnZScpO1xuICAgIHRoaXMudHJpZ2dlcnMgPSB7IC4uLnRoaXMudHJpZ2dlcnMsIGN1c3RvbU1lc3NhZ2U6IGZuLmZ1bmN0aW9uQXJuIH07XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoICdEZWZpbmUgQXV0aCBDaGFsbGVuZ2UnIHRyaWdnZXJcbiAgICogR3JhbnRzIGFjY2VzcyBmcm9tIGNvZ25pdG8taWRwLmFtYXpvbmF3cy5jb20gdG8gdGhlIGxhbWJkYVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLWRlZmluZS1hdXRoLWNoYWxsZW5nZS5odG1sXG4gICAqIEBwYXJhbSBmbiB0aGUgbGFtYmRhIGZ1bmN0aW9uIHRvIGF0dGFjaFxuICAgKi9cbiAgcHVibGljIGFkZERlZmluZUF1dGhDaGFsbGVuZ2VUcmlnZ2VyKGZuOiBsYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XG4gICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKGZuLCAnRGVmaW5lQXV0aENoYWxsZW5nZScpO1xuICAgIHRoaXMudHJpZ2dlcnMgPSB7IC4uLnRoaXMudHJpZ2dlcnMsIGRlZmluZUF1dGhDaGFsbGVuZ2U6IGZuLmZ1bmN0aW9uQXJuIH07XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoICdQb3N0IEF1dGhlbnRpY2F0aW9uJyB0cmlnZ2VyXG4gICAqIEdyYW50cyBhY2Nlc3MgZnJvbSBjb2duaXRvLWlkcC5hbWF6b25hd3MuY29tIHRvIHRoZSBsYW1iZGFcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wb3N0LWF1dGhlbnRpY2F0aW9uLmh0bWxcbiAgICogQHBhcmFtIGZuIHRoZSBsYW1iZGEgZnVuY3Rpb24gdG8gYXR0YWNoXG4gICAqL1xuICBwdWJsaWMgYWRkUG9zdEF1dGhlbnRpY2F0aW9uVHJpZ2dlcihmbjogbGFtYmRhLklGdW5jdGlvbik6IHZvaWQge1xuICAgIHRoaXMuYWRkTGFtYmRhUGVybWlzc2lvbihmbiwgJ1Bvc3RBdXRoZW50aWNhdGlvbicpO1xuICAgIHRoaXMudHJpZ2dlcnMgPSB7IC4uLnRoaXMudHJpZ2dlcnMsIHBvc3RBdXRoZW50aWNhdGlvbjogZm4uZnVuY3Rpb25Bcm4gfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2ggJ1Bvc3QgQ29uZmlybWF0aW9uJyB0cmlnZ2VyXG4gICAqIEdyYW50cyBhY2Nlc3MgZnJvbSBjb2duaXRvLWlkcC5hbWF6b25hd3MuY29tIHRvIHRoZSBsYW1iZGFcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS1wb3N0LWNvbmZpcm1hdGlvbi5odG1sXG4gICAqIEBwYXJhbSBmbiB0aGUgbGFtYmRhIGZ1bmN0aW9uIHRvIGF0dGFjaFxuICAgKi9cbiAgcHVibGljIGFkZFBvc3RDb25maXJtYXRpb25UcmlnZ2VyKGZuOiBsYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XG4gICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKGZuLCAnUG9zdENvbmZpcm1hdGlvbicpO1xuICAgIHRoaXMudHJpZ2dlcnMgPSB7IC4uLnRoaXMudHJpZ2dlcnMsIHBvc3RDb25maXJtYXRpb246IGZuLmZ1bmN0aW9uQXJuIH07XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoICdQcmUgQXV0aGVudGljYXRpb24nIHRyaWdnZXJcbiAgICogR3JhbnRzIGFjY2VzcyBmcm9tIGNvZ25pdG8taWRwLmFtYXpvbmF3cy5jb20gdG8gdGhlIGxhbWJkYVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS91c2VyLXBvb2wtbGFtYmRhLXByZS1hdXRoZW50aWNhdGlvbi5odG1sXG4gICAqIEBwYXJhbSBmbiB0aGUgbGFtYmRhIGZ1bmN0aW9uIHRvIGF0dGFjaFxuICAgKi9cbiAgcHVibGljIGFkZFByZUF1dGhlbnRpY2F0aW9uVHJpZ2dlcihmbjogbGFtYmRhLklGdW5jdGlvbik6IHZvaWQge1xuICAgIHRoaXMuYWRkTGFtYmRhUGVybWlzc2lvbihmbiwgJ1ByZUF1dGhlbnRpY2F0aW9uJyk7XG4gICAgdGhpcy50cmlnZ2VycyA9IHsgLi4udGhpcy50cmlnZ2VycywgcHJlQXV0aGVudGljYXRpb246IGZuLmZ1bmN0aW9uQXJuIH07XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoICdQcmUgU2lnbiBVcCcgdHJpZ2dlclxuICAgKiBHcmFudHMgYWNjZXNzIGZyb20gY29nbml0by1pZHAuYW1hem9uYXdzLmNvbSB0byB0aGUgbGFtYmRhXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VzZXItcG9vbC1sYW1iZGEtcHJlLXNpZ24tdXAuaHRtbFxuICAgKiBAcGFyYW0gZm4gdGhlIGxhbWJkYSBmdW5jdGlvbiB0byBhdHRhY2hcbiAgICovXG4gIHB1YmxpYyBhZGRQcmVTaWduVXBUcmlnZ2VyKGZuOiBsYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XG4gICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKGZuLCAnUHJlU2lnblVwJyk7XG4gICAgdGhpcy50cmlnZ2VycyA9IHsgLi4udGhpcy50cmlnZ2VycywgcHJlU2lnblVwOiBmbi5mdW5jdGlvbkFybiB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaCAnVmVyaWZ5IEF1dGggQ2hhbGxlbmdlIFJlc3BvbnNlJyB0cmlnZ2VyXG4gICAqIEdyYW50cyBhY2Nlc3MgZnJvbSBjb2duaXRvLWlkcC5hbWF6b25hd3MuY29tIHRvIHRoZSBsYW1iZGFcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdXNlci1wb29sLWxhbWJkYS12ZXJpZnktYXV0aC1jaGFsbGVuZ2UtcmVzcG9uc2UuaHRtbFxuICAgKiBAcGFyYW0gZm4gdGhlIGxhbWJkYSBmdW5jdGlvbiB0byBhdHRhY2hcbiAgICovXG4gIHB1YmxpYyBhZGRWZXJpZnlBdXRoQ2hhbGxlbmdlUmVzcG9uc2VUcmlnZ2VyKGZuOiBsYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XG4gICAgdGhpcy5hZGRMYW1iZGFQZXJtaXNzaW9uKGZuLCAnVmVyaWZ5QXV0aENoYWxsZW5nZVJlc3BvbnNlJyk7XG4gICAgdGhpcy50cmlnZ2VycyA9IHsgLi4udGhpcy50cmlnZ2VycywgdmVyaWZ5QXV0aENoYWxsZW5nZVJlc3BvbnNlOiBmbi5mdW5jdGlvbkFybiB9O1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRMYW1iZGFQZXJtaXNzaW9uKGZuOiBsYW1iZGEuSUZ1bmN0aW9uLCBuYW1lOiBzdHJpbmcpOiB2b2lkIHtcbiAgICBjb25zdCBub3JtYWxpemUgPSBuYW1lLmNoYXJBdCgwKS50b1VwcGVyQ2FzZSgpICsgbmFtZS5zbGljZSgxKTtcbiAgICBmbi5hZGRQZXJtaXNzaW9uKGAke25vcm1hbGl6ZX1Db2duaXRvYCwge1xuICAgICAgcHJpbmNpcGFsOiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2NvZ25pdG8taWRwLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIHNvdXJjZUFybjogdGhpcy51c2VyUG9vbEFyblxuICAgIH0pO1xuICB9XG59XG4iXX0=