"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClusterFromSnapshot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const perms_1 = require("aws-cdk-lib/aws-rds/lib/perms");
const util_1 = require("aws-cdk-lib/aws-rds/lib/private/util");
/**
 * A Serverless Cluster restored from a snapshot.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
class ServerlessClusterFromSnapshot extends aws_cdk_lib_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.enableDataApi = props.enableDataApi;
        const { subnetIds } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            aws_cdk_lib_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_b = props.subnetGroup) !== null && _b !== void 0 ? _b : new aws_cdk_lib_1.aws_rds.SubnetGroup(this, "Subnets", {
            description: `Subnets for ${id} database`,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            removalPolicy: props.removalPolicy === aws_cdk_lib_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        if (props.backupRetention) {
            const backupRetentionDays = props.backupRetention.toDays();
            if (backupRetentionDays < 1 || backupRetentionDays > 35) {
                throw new Error(`backup retention period must be between 1 and 35 days. received: ${backupRetentionDays}`);
            }
        }
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_c = props.parameterGroup) !== null && _c !== void 0 ? _c : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const securityGroups = (_d = props.securityGroups) !== null && _d !== void 0 ? _d : [
            new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, "SecurityGroup", {
                description: "RDS security group",
                vpc: this.vpc,
            }),
        ];
        const clusterIdentifier = aws_cdk_lib_1.FeatureFlags.of(this).isEnabled(aws_cdk_lib_1.cx_api.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_e = props.clusterIdentifier) === null || _e === void 0 ? void 0 : _e.toLowerCase() : props.clusterIdentifier;
        const cluster = new aws_cdk_lib_1.aws_rds.CfnDBCluster(this, "Resource", {
            backupRetentionPeriod: (_f = props.backupRetention) === null || _f === void 0 ? void 0 : _f.toDays(),
            databaseName: props.defaultDatabaseName,
            dbClusterIdentifier: clusterIdentifier,
            snapshotIdentifier: props.snapshotIdentifier,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            engine: props.engine.engineType,
            engineVersion: (_g = props.engine.engineVersion) === null || _g === void 0 ? void 0 : _g.fullVersion,
            engineMode: "serverless",
            enableHttpEndpoint: aws_cdk_lib_1.Lazy.any({ produce: () => this.enableDataApi }),
            kmsKeyId: (_h = props.storageEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            scalingConfiguration: props.scaling ? this.renderScalingConfiguration(props.scaling) : undefined,
            vpcSecurityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = aws_cdk_lib_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new aws_cdk_lib_1.aws_rds.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new aws_cdk_lib_1.aws_rds.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups,
            defaultPort: aws_cdk_lib_1.aws_ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_j = props.removalPolicy) !== null && _j !== void 0 ? _j : aws_cdk_lib_1.RemovalPolicy.SNAPSHOT);
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        if (secret) {
            this.secret = secret.attach(this);
        }
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @stability stable
     */
    addRotationSingleUser(options = {}) {
        var _b;
        if (!this.secret) {
            throw new Error("Cannot add single user rotation for a cluster without secret.");
        }
        const id = "RotationSingleUser";
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error("A single user rotation was already added to this cluster.");
        }
        return new aws_cdk_lib_1.aws_secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_b = options.excludeCharacters) !== null && _b !== void 0 ? _b : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     *
     * @stability stable
     */
    addRotationMultiUser(id, options) {
        var _b;
        if (!this.secret) {
            throw new Error("Cannot add multi user rotation for a cluster without secret.");
        }
        return new aws_cdk_lib_1.aws_secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_b = options.excludeCharacters) !== null && _b !== void 0 ? _b : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    renderScalingConfiguration(options) {
        var _b;
        const minCapacity = options.minCapacity;
        const maxCapacity = options.maxCapacity;
        if (minCapacity && maxCapacity && minCapacity > maxCapacity) {
            throw new Error("Maximum capacity must be greater than or equal to minimum capacity.");
        }
        const secondsToAutoPause = (_b = options.autoPause) === null || _b === void 0 ? void 0 : _b.toSeconds();
        if (secondsToAutoPause && (secondsToAutoPause < 300 || secondsToAutoPause > 86400)) {
            throw new Error("Auto pause time must be between 5 minutes and 1 day.");
        }
        return {
            autoPause: secondsToAutoPause !== 0,
            minCapacity: options.minCapacity,
            maxCapacity: options.maxCapacity,
            secondsUntilAutoPause: secondsToAutoPause === 0 ? undefined : secondsToAutoPause,
        };
    }
    /**
     * The ARN of the cluster.
     *
     * @stability stable
     */
    get clusterArn() {
        return aws_cdk_lib_1.Stack.of(this).formatArn({
            service: "rds",
            resource: "cluster",
            arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: this.clusterIdentifier,
        });
    }
    /**
     * Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantDataApiAccess(grantee) {
        var _b;
        if (this.enableDataApi === false) {
            throw new Error("Cannot grant Data API access when the Data API is disabled");
        }
        this.enableDataApi = true;
        const ret = aws_cdk_lib_1.aws_iam.Grant.addToPrincipal({
            grantee,
            actions: perms_1.DATA_API_ACTIONS,
            resourceArns: ["*"],
            scope: this,
        });
        (_b = this.secret) === null || _b === void 0 ? void 0 : _b.grantRead(grantee);
        return ret;
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: aws_cdk_lib_1.aws_secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
exports.ServerlessClusterFromSnapshot = ServerlessClusterFromSnapshot;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClusterFromSnapshot[_a] = { fqn: "@pepperize/cdk-serverless-cluster-from-snapshot.ServerlessClusterFromSnapshot", version: "0.0.48" };
//# sourceMappingURL=data:application/json;base64,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