"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketDeploymentExpirator = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const handlerCodeBundle = path.join(__dirname, '..', '.build');
const handlerSourceDirectory = path.join(__dirname, '..', 'lambda', 'src');
const defaultMetaKey = 'deployed';
/**
 * @stability stable
 */
class BucketDeploymentExpirator extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const assetHash = calcSourceHash(handlerSourceDirectory);
        const handler = new aws_lambda_1.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(),
            code: aws_lambda_1.Code.fromAsset(handlerCodeBundle, { assetHash }),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeploymentExpirator',
            timeout: core_1.Duration.minutes(15),
            role: props.role,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        props.sourceBucket.grantRead(handler);
        props.sourceBucket.grantDelete(handler);
        const customResource = new core_1.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeploymentExpirator',
            properties: {
                SourceBucketName: props.sourceBucket.bucketName,
                MetaLookupKey: (_a = props.metaLookupKey) !== null && _a !== void 0 ? _a : defaultMetaKey,
                DeploymentsToKeep: (_b = props.deploymentsToKeep) !== null && _b !== void 0 ? _b : 3,
                RemoveUnmarked: (_c = props.removeUnmarked) !== null && _c !== void 0 ? _c : false,
                Timestamp: new Date().getTime(),
            },
        });
        // Add the bucket deployment as a dependency so we deploy after
        customResource.node.addDependency(props.bucketDeployment);
    }
    renderSingletonUuid() {
        return '7EE36E34-1860-4B22-AC9E-149ABDEB886C';
    }
}
exports.BucketDeploymentExpirator = BucketDeploymentExpirator;
/**
 * We need a custom source hash calculation since the bundle.zip file
 * contains python dependencies installed during build and results in a
 * non-deterministic behavior.
 *
 * So we just take the `src/` directory of our custom resource code.
 */
function calcSourceHash(srcDir) {
    const sha = crypto.createHash('sha256');
    for (const file of fs.readdirSync(srcDir)) {
        const data = fs.readFileSync(path.join(srcDir, file));
        sha.update(`<file name=${file}>`);
        sha.update(data);
        sha.update('</file>');
    }
    return sha.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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