# python3
# Copyright 2018 DeepMind Technologies Limited. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""New version of the RL agent Builder interface (with LearnerCore)."""

import abc
import dataclasses
from typing import Callable, Generic, Iterator, List, Optional

from acme import adders
from acme import core
from acme import specs
from acme.agents.jax import builders
from acme.jax import learner_core
from acme.jax import learning
from acme.jax import networks as networks_lib
from acme.jax.types import Networks, PolicyNetwork, Sample, TrainingState  # pylint: disable=g-multiple-import
from acme.utils import counting
from acme.utils import loggers
import reverb

# Note: we cannot just use GenericMeta, because it doesn't exist in Python 3.7.
_LearnerBuilderMeta = type(builders.GenericActorLearnerBuilder)


# TODO(b/192806089): migrate all builders to use the learner core, remove
# all base Builder classes except for GenericActorLearnerCoreBuilder and
# remove this meta.
class _LearnerCoreBuilderMeta(_LearnerBuilderMeta):
  """Metaclass that prevents subclasses from overriding make_learner."""

  def __init__(cls, class_name, bases, dct, **kwargs):  # pylint: disable=no-self-argument
    if (class_name != 'GenericActorLearnerCoreBuilder' and
        'make_learner' in dct):
      raise AssertionError('`make_learner` is final and cannot be overridden. '
                           '`make_learner_core` should be used instead.')
    _LearnerBuilderMeta.__init__(cls, class_name, bases, dct, **kwargs)  # pytype: disable=invalid-typevar


@dataclasses.dataclass
class GenericActorLearnerCoreBuilder(
    builders.GenericActorLearnerBuilder[Networks, PolicyNetwork, Sample],
    Generic[Networks, PolicyNetwork, Sample, TrainingState],
    metaclass=_LearnerCoreBuilderMeta):
  """Defines an interface for defining the components of an RL agent.

  Implementations of this interface contain a complete specification of a
  concrete RL agent. An instance of this class can be used to build an
  RL agent which interacts with the environment either locally or in a
  distributed setup.
  """

  logger_fn: Callable[[], loggers.Logger] = lambda: None

  @abc.abstractmethod
  def make_replay_tables(
      self,
      environment_spec: specs.EnvironmentSpec,
  ) -> List[reverb.Table]:
    """Create tables to insert data into."""

  @abc.abstractmethod
  def make_dataset_iterator(
      self,
      replay_client: reverb.Client,
  ) -> Iterator[Sample]:
    """Create a dataset iterator to use for learning/updating the agent."""

  @abc.abstractmethod
  def make_adder(
      self,
      replay_client: reverb.Client,
  ) -> Optional[adders.Adder]:
    """Create an adder which records data generated by the actor/environment.

    Args:
      replay_client: Reverb Client which points to the replay server.
    """

  @abc.abstractmethod
  def make_actor(
      self,
      random_key: networks_lib.PRNGKey,
      policy_network: PolicyNetwork,
      adder: Optional[adders.Adder] = None,
      variable_source: Optional[core.VariableSource] = None,
  ) -> core.Actor:
    """Create an actor instance.

    Args:
      random_key: A key for random number generation.
      policy_network: Instance of a policy network; this should be a callable
        which takes as input observations and returns actions.
      adder: How data is recorded (e.g. added to replay).
      variable_source: A source providing the necessary actor parameters.
    """

  @abc.abstractmethod
  def make_learner_core(
      self,
      networks: Networks) -> learner_core.LearnerCore[Sample, TrainingState]:
    """Creates the learner core.

    Args:
      networks: struct describing the networks needed by the learner; this can
        be specific to the learner in question.
    """

  # @final
  def make_learner(
      self,
      random_key: networks_lib.PRNGKey,
      networks: Networks,
      dataset: Iterator[Sample],
      replay_client: Optional[reverb.Client] = None,
      counter: Optional[counting.Counter] = None,
      # TODO(mwhoffman): eliminate checkpoint and move it outside.
      checkpoint: bool = False,
  ) -> core.Learner:
    """Creates an instance of the learner.

    NOTE: make_learner is not part of this interface and is defined here only to
    ensure compatibility with the old builder interface to enable incremental
    transition.
    Subclasses cannot override this method.
    make_learner_core should be overridden instead.

    Args:
      random_key: A key for random number generation.
      networks: struct describing the networks needed by the learner; this can
        be specific to the learner in question.
      dataset: iterator over samples from replay.
      replay_client: client which allows communication with replay, e.g. in
        order to update priorities.
      counter: a Counter which allows for recording of counts (learner steps,
        actor steps, etc.) distributed throughout the agent.
      checkpoint: bool controlling whether the learner checkpoints itself.
    Returns:
      The learner instance.
    """
    return learning.DefaultJaxLearner(
        self.make_learner_core(networks), dataset, random_key, counter,
        self.logger_fn())
