from functools import wraps

from flask import redirect, request, session, url_for

from .persistence import read


def ensure_session_version(f, CURRENT_VERSION=1):
    """
    Increment CURRENT_VERSION to nuke all current sessions.
    Useful for when I try to change session structure or something. Maybe.
    """

    @wraps(f)
    def decorated_function(*args, **kwargs):
        if (
            not session.get("version")
            or not isinstance(session["version"], int)
            or session["version"] < CURRENT_VERSION
        ):
            session.clear()
            session["version"] = CURRENT_VERSION
            return redirect("/")
        return f(*args, **kwargs)

    return decorated_function


def require_login(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if session.get("user") is None:
            return redirect(url_for("auth_view", next=request.url))
        return f(*args, **kwargs)

    return decorated_function


def trigger_has_read(f):
    """
    Augments a view with the ability to mark a chapter as read if there's a
    `?has_read=<chapter_id>` url param.
    """

    @wraps(f)
    def decorated_function(*args, **kwargs):
        assert "site" in kwargs  # only use on site-specific views
        has_read_chapter_id = request.args.get("has_read")
        if has_read_chapter_id:
            if session.get("user"):
                read(session["user"]["id"], kwargs["site"], has_read_chapter_id)
                return redirect(request.url[: request.url.rfind("?")])
        return f(*args, **kwargs)

    return decorated_function
