"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const aws_apigatewayv2_1 = require("aws-cdk-lib/aws-apigatewayv2");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(method, path) {
        this.method = method !== null && method !== void 0 ? method : HttpMethod.ANY;
        this.path = path;
        this.key = path ? `${method} ${path}` : '$default';
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_HttpMethod(method);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.with);
            }
            throw error;
        }
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('A route path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(method, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
_a = JSII_RTTI_SYMBOL_1;
HttpRouteKey[_a] = { fqn: "@aws-cdk/aws-apigatewayv2-alpha.HttpRouteKey", version: "2.23.0-alpha.0" };
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey();
/**
 * Supported Route Authorizer types
 */
var HttpRouteAuthorizationType;
(function (HttpRouteAuthorizationType) {
    /** AWS IAM */
    HttpRouteAuthorizationType["AWS_IAM"] = "AWS_IAM";
    /** JSON Web Tokens */
    HttpRouteAuthorizationType["JWT"] = "JWT";
    /** Lambda Authorizer */
    HttpRouteAuthorizationType["CUSTOM"] = "CUSTOM";
    /** No authorizer */
    HttpRouteAuthorizationType["NONE"] = "NONE";
})(HttpRouteAuthorizationType || (HttpRouteAuthorizationType = {}));
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g;
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_HttpRouteProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        this.method = props.routeKey.method;
        this.routeArn = this.produceRouteArn(props.routeKey.method);
        const config = props.integration._bindToRoute({
            route: this,
            scope: this,
        });
        this.authBindResult = (_c = props.authorizer) === null || _c === void 0 ? void 0 : _c.bind({
            route: this,
            scope: this.httpApi instanceof constructs_1.Construct ? this.httpApi : this,
        });
        if (this.authBindResult && !(this.authBindResult.authorizationType in HttpRouteAuthorizationType)) {
            throw new Error(`authorizationType should either be AWS_IAM, JWT, CUSTOM, or NONE but was '${this.authBindResult.authorizationType}'`);
        }
        let authorizationScopes = (_d = this.authBindResult) === null || _d === void 0 ? void 0 : _d.authorizationScopes;
        if (this.authBindResult && props.authorizationScopes) {
            authorizationScopes = Array.from(new Set([
                ...authorizationScopes !== null && authorizationScopes !== void 0 ? authorizationScopes : [],
                ...props.authorizationScopes,
            ]));
        }
        if ((authorizationScopes === null || authorizationScopes === void 0 ? void 0 : authorizationScopes.length) === 0) {
            authorizationScopes = undefined;
        }
        const routeProps = {
            apiId: props.httpApi.apiId,
            routeKey: props.routeKey.key,
            target: `integrations/${config.integrationId}`,
            authorizerId: (_e = this.authBindResult) === null || _e === void 0 ? void 0 : _e.authorizerId,
            authorizationType: (_g = (_f = this.authBindResult) === null || _f === void 0 ? void 0 : _f.authorizationType) !== null && _g !== void 0 ? _g : 'NONE',
            authorizationScopes,
        };
        const route = new aws_apigatewayv2_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
    produceRouteArn(httpMethod) {
        var _c;
        const stage = '*';
        const iamHttpMethod = httpMethod === HttpMethod.ANY ? '*' : httpMethod;
        const path = (_c = this.path) !== null && _c !== void 0 ? _c : '/';
        // When the user has provided a path with path variables, we replace the
        // path variable and all that follows with a wildcard.
        const iamPath = path.replace(/\{.*?\}.*/, '*');
        return `arn:aws:execute-api:${this.stack.region}:${this.stack.account}:${this.httpApi.apiId}/${stage}/${iamHttpMethod}${iamPath}`;
    }
    grantInvoke(grantee, options = {}) {
        var _c;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_GrantInvokeOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantInvoke);
            }
            throw error;
        }
        if (!this.authBindResult || this.authBindResult.authorizationType !== HttpRouteAuthorizationType.AWS_IAM) {
            throw new Error('To use grantInvoke, you must use IAM authorization');
        }
        const httpMethods = Array.from(new Set((_c = options.httpMethods) !== null && _c !== void 0 ? _c : [this.method]));
        if (this.method !== HttpMethod.ANY && httpMethods.some(method => method !== this.method)) {
            throw new Error('This route does not support granting invoke for all requested http methods');
        }
        const resourceArns = httpMethods.map(httpMethod => {
            return this.produceRouteArn(httpMethod);
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['execute-api:Invoke'],
            resourceArns: resourceArns,
        });
    }
}
exports.HttpRoute = HttpRoute;
_b = JSII_RTTI_SYMBOL_1;
HttpRoute[_b] = { fqn: "@aws-cdk/aws-apigatewayv2-alpha.HttpRoute", version: "2.23.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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