"""
`build_log_parser_test`
=======================================================================
Unit tests for parsing of build logs generated by compilers
* Author(s): Jimmy Gomez
"""
import os
import json
import pytest
from embedops_cli.eotools.parse import (
    parse_storage_sizes,
    main as parse_main,
)
from unittest import mock

file_dir = os.path.dirname(os.path.realpath(__file__))


def test_parsing_of_iar_build_log():
    """Test parsing of build log generated by IAR"""
    input_filename = file_dir + "/iar_example_build.log"
    storage_sizes = parse_storage_sizes(input_filename, "IAR")
    print(storage_sizes)
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "library"
    assert storage_sizes[0]["dimensions"]["build_target_group"] == "Debug"
    assert storage_sizes[0]["dimensions"]["build_target"] == "library - Debug"
    assert storage_sizes[0]["flash_code_size"] == 7636
    assert storage_sizes[0]["flash_data_size"] == 136
    assert storage_sizes[0]["ram_size"] == 1036


def test_parsing_of_gcc_build_log():
    """Test parsing of build log generated by GCC"""
    input_filename = file_dir + "/gcc_example_build.log"
    storage_sizes = parse_storage_sizes(input_filename, "GCC")
    assert storage_sizes[0]["flash_code_size"] == 36080
    assert storage_sizes[0]["flash_data_size"] == 664
    assert storage_sizes[0]["ram_size"] == 50960
    assert storage_sizes[0]["dimensions"]["build_target_name"] == "application.elf"


def test_parsing_of_iar_multiple_targets_build_log():
    """Test parsing of build log generated by IAR"""
    input_filename = file_dir + "/iar_example_multiple_targets_build.log"
    storage_sizes = parse_storage_sizes(input_filename, "IAR")
    print(json.dumps(storage_sizes, indent=2))
    assert len(storage_sizes) == 10
    for result in storage_sizes:
        if "OttoGUIMicrocontroller" == result["dimensions"]["build_target_name"]:
            if "Debug" == result["dimensions"]["build_target_group"]:
                assert result["flash_code_size"] == 327700
                assert result["flash_data_size"] == 34579193
                assert result["ram_size"] == 7391295
            elif "Production" == result["dimensions"]["build_target_group"]:
                assert result["flash_code_size"] == 311032
                assert result["flash_data_size"] == 34571647
                assert result["ram_size"] == 7391297


@mock.patch.dict(os.environ, {"EMBEDOPS_COMPILER": "FOO"})
def test_exit_main_if_unsupported_compiler():
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_main()
    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1


def test_exit_parse_storage_sizes_if_unsupported_compiler():
    input_filename = file_dir + "/gcc_example_build.log"
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_storage_sizes(input_filename, "FOO")
    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1


def test_exit_parse_storage_sizes_if_logfile_dne():
    input_filename = file_dir + "/does_not_exist.log"
    with pytest.raises(SystemExit) as pytest_wrapped_e:
        parse_storage_sizes(input_filename, "IAR")
    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1
