"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cicd_spa_website_1 = require("../lib/cicd_spa_website");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const CONSTRUCT_ID = 'my-test-construct';
let minimumProps = {
    url: 'demo.matthewbonig.com',
    githubSource: {
        owner: 'mbonig',
        repo: 'fld_public_site',
        oauthToken: core_1.SecretValue.secretsManager('github-oauth-token')
    }
};
let withUrlProps = {
    url: 'demo.matthewbonig.com',
    hostedZone: {
        hostedZoneId: 'ABCDEFGHIJKLMN',
        zoneName: 'matthewbonig.com'
    },
    githubSource: {
        owner: 'mbonig',
        repo: 'fld_public_site',
        oauthToken: core_1.SecretValue.secretsManager('github-oauth-token')
    }
};
let withUrlAndCertProps = {
    url: 'demo.matthewbonig.com',
    hostedZone: {
        hostedZoneId: 'ABCDEFGHIJKLMN',
        zoneName: 'matthewbonig.com'
    },
    certificate: true,
    githubSource: {
        owner: 'mbonig',
        repo: 'fld_public_site',
        oauthToken: core_1.SecretValue.secretsManager('github-oauth-token')
    }
};
function createStack(props) {
    const app = new cdk.App();
    const stack = new core_1.Stack(app, 'testing-stack', { env: { account: '1234567', region: 'us-east-1' } });
    let propsToUse = props;
    if (props instanceof Function) {
        propsToUse = props(stack);
    }
    new cicd_spa_website_1.CicdSpaWebsite(stack, CONSTRUCT_ID, propsToUse);
    return stack;
}
describe('s3 bucket website bucket', () => {
    test('has url as bucket name', () => {
        const stack = createStack(minimumProps);
        expect(stack).toHaveResource('AWS::S3::Bucket', {
            BucketName: minimumProps.url
        });
    });
    test('website configuration is set if certificate is not provided', () => {
        const stack = createStack(minimumProps);
        expect(stack).toHaveResource('AWS::S3::Bucket', {
            WebsiteConfiguration: {
                ErrorDocument: 'index.html',
                IndexDocument: 'index.html'
            }
        });
    });
    test('Does not make bucket public if cloudfront is used with certificate', () => {
        const stack = createStack(withUrlAndCertProps);
        expect(stack).not.toHaveResource('AWS::S3::Bucket', {
            WebsiteConfiguration: {
                ErrorDocument: 'index.html',
                IndexDocument: 'index.html'
            }
        });
        expect(stack).toHaveResourceLike("AWS::S3::BucketPolicy", {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "s3:GetObject",
                        "Effect": "Allow",
                        "Principal": {
                            "CanonicalUser": {
                                "Fn::GetAtt": [
                                    "mytestconstructoai955F29D5",
                                    "S3CanonicalUserId"
                                ]
                            }
                        }
                    }, {}
                ]
            }
        });
    });
});
describe('CloudFront distribution', () => {
    test('Creates if certificate is provided', () => {
        const stack = createStack((stack) => ({
            ...withUrlAndCertProps,
            certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'somelongarn')
        }));
        expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
            "DistributionConfig": {
                "ViewerCertificate": {
                    "AcmCertificateArn": "somelongarn",
                    "SslSupportMethod": "sni-only"
                }
            }
        });
    });
    test('Creates if certificate is true', () => {
        const stack = createStack(withUrlAndCertProps);
        expect(stack).toHaveResource("AWS::CloudFront::Distribution");
        const construct = stack.node.findChild(CONSTRUCT_ID);
        const certificate = construct.node.tryFindChild('certificate');
        expect(certificate).toBeTruthy();
    });
    test('does not create if certificate is false or not provided', () => {
        const stack = createStack(minimumProps);
        const construct = stack.node.findChild(CONSTRUCT_ID);
        const certificate = construct.node.tryFindChild('certificate');
        expect(certificate).toBeUndefined();
    });
    test('throws error if certificate requested but no hostedZone', () => {
        expect(() => {
            createStack({
                ...minimumProps,
                certificate: true
            });
        }).toThrow(/If you'd like a certificate then you must provide a `hostedZone`./);
    });
    test('uses OAI', () => {
        const stack = createStack(withUrlAndCertProps);
        expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity');
    });
});
describe('artifacts buckets', () => {
    test('is not public', () => {
        const stack = createStack(minimumProps);
        expect(stack).toHaveResource('AWS::S3::Bucket', {
            "BucketEncryption": {
                "ServerSideEncryptionConfiguration": [
                    {
                        "ServerSideEncryptionByDefault": {
                            "SSEAlgorithm": "aws:kms"
                        }
                    }
                ]
            },
            "BucketName": `${minimumProps.url.replace(/\./gi, '-')}-artifacts`,
            "PublicAccessBlockConfiguration": {
                "BlockPublicAcls": true,
                "BlockPublicPolicy": true,
                "IgnorePublicAcls": true,
                "RestrictPublicBuckets": true
            }
        });
    });
});
describe('github source', () => {
    test('uses passed values', () => {
        const stack = createStack(minimumProps);
        // we gotta go deep
        let iConstruct = stack.node.findChild(CONSTRUCT_ID);
        // inception style
        const codePipeline = iConstruct.node.findChild(`build-pipeline`);
        // there's the good stuff.
        const action = codePipeline.stages[0].actions[0];
        // @ts-ignore
        expect(action.props.oauthToken).toBe(minimumProps.githubSource.oauthToken);
        // @ts-ignore
        expect(action.props.repo).toBe(minimumProps.githubSource.repo);
        // @ts-ignore
        expect(action.props.owner).toBe(minimumProps.githubSource.owner);
    });
});
describe('codebuild', () => {
    test('uses default buildspec', () => {
        const stack = createStack(minimumProps);
        expect(stack).toHaveResourceLike("AWS::CodeBuild::Project", {
            "Source": {
                "BuildSpec": JSON.stringify(cicd_spa_website_1.DEFAULT_BUILD_SPEC, null, 2)
            }
        });
    });
    test('uses passed string for buildspec', () => {
        const stack = createStack({ ...minimumProps, buildSpec: 'buildspec.yaml' });
        expect(stack).toHaveResourceLike("AWS::CodeBuild::Project", {
            "Source": {
                "BuildSpec": 'buildspec.yaml'
            }
        });
    });
    test('uses passed object for buildspec', () => {
        let buildSpec = { one: 'two', three: 'four' };
        const stack = createStack({ ...minimumProps, buildSpec });
        expect(stack).toHaveResourceLike("AWS::CodeBuild::Project", {
            "Source": {
                "BuildSpec": JSON.stringify(buildSpec, null, 2)
            }
        });
    });
});
describe('codepipeline', () => {
    test(`doesn't includes invalidate when certificate is not supplied`, () => {
        const stack = createStack(minimumProps);
        const construct = stack.node.findChild(CONSTRUCT_ID);
        const pipeline = construct.node.findChild('build-pipeline');
        const deployActions = pipeline.stages[2].actions;
        expect(deployActions).toHaveLength(1);
    });
    test('includes invalidate when cloudfront is used', () => {
        const stack = createStack(withUrlAndCertProps);
        const construct = stack.node.findChild(CONSTRUCT_ID);
        const pipeline = construct.node.findChild('build-pipeline');
        const deployActions = pipeline.stages[2].actions;
        expect(deployActions).toHaveLength(2);
    });
    test('deploys to s3 bucket with public when no cert set', () => {
        const stack = createStack(minimumProps);
        const construct = stack.node.findChild(CONSTRUCT_ID);
        const pipeline = construct.node.findChild('build-pipeline');
        const deployActions = pipeline.stages[2].actions;
        const deployAction = deployActions[0];
        // @ts-ignore
        expect(deployAction.props.accessControl).toBe("public-read");
    });
    test('deploys to s3 bucket with private when cert set', () => {
        const stack = createStack(withUrlAndCertProps);
        const construct = stack.node.findChild(CONSTRUCT_ID);
        const pipeline = construct.node.findChild('build-pipeline');
        const deployActions = pipeline.stages[2].actions;
        const deployAction = deployActions[0];
        // @ts-ignore
        expect(deployAction.props.accessControl).toBe("private");
    });
});
describe('dns', () => {
    test(`Does not create dns record if hostedZone is not provided`, () => {
        const stack = createStack(minimumProps);
        expect(stack).not.toHaveResourceLike("AWS::Route53::RecordSet");
    });
    test('creates DNS record if hosted-zone is provided', () => {
        const stack = createStack(withUrlAndCertProps);
        expect(stack).toHaveResourceLike("AWS::Route53::RecordSet", {
            "Name": `${withUrlAndCertProps.url}.`,
            "Type": "A",
            "AliasTarget": {
                "DNSName": {
                    "Fn::GetAtt": [
                        "mytestconstructsitedistributionCFDistributionAE8D5429",
                        "DomainName"
                    ]
                },
                "HostedZoneId": "Z2FDTNDATAQYW2"
            },
            "HostedZoneId": "ABCDEFGHIJKLMN"
        });
    });
    test('points to s3 bucket if certificate is not supplied', () => {
        const stack = createStack(withUrlProps);
        expect(stack).toHaveResourceLike("AWS::Route53::RecordSet", {
            "Name": `${withUrlProps.url}.`,
            "Type": "A",
            "AliasTarget": {
                "DNSName": "s3-website-us-east-1.amazonaws.com",
                "HostedZoneId": "Z3AQBSTGFYJSTF"
            },
            "HostedZoneId": "ABCDEFGHIJKLMN"
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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