"""
AI-powered authentication risk assessment node.

This module provides comprehensive risk assessment for authentication requests
including device trust analysis, location verification, behavioral patterns,
and ML-based anomaly detection.
"""

import hashlib
import json
import logging
from dataclasses import dataclass
from datetime import UTC, datetime, timedelta
from enum import Enum
from typing import Any, Dict, List, Optional, Set

from kailash.nodes.ai.llm_agent import LLMAgentNode
from kailash.nodes.base import Node, NodeParameter, register_node
from kailash.nodes.mixins import LoggingMixin, PerformanceMixin, SecurityMixin

logger = logging.getLogger(__name__)


class RiskLevel(Enum):
    """Risk assessment levels."""

    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"


@dataclass
class RiskContext:
    """Risk assessment context."""

    user_id: str
    ip_address: str
    device_info: Dict[str, Any]
    timestamp: str
    location: Optional[Dict[str, Any]] = None
    user_timezone: Optional[str] = None
    usual_hours: Optional[Dict[str, int]] = None
    usual_locations: Optional[List[str]] = None


@dataclass
class RiskAssessment:
    """Risk assessment result."""

    risk_score: float
    risk_level: RiskLevel
    risk_factors: List[str]
    trust_factors: List[str]
    mitigation_required: List[str]
    additional_checks: List[str]
    confidence: float
    assessment_time: datetime


@register_node()
class RiskAssessmentNode(SecurityMixin, PerformanceMixin, LoggingMixin, Node):
    """AI-powered authentication risk assessment.

    This node provides comprehensive risk assessment for authentication requests:
    - Device trust analysis
    - Location verification and geographic anomaly detection
    - Behavioral pattern analysis
    - Time-based access pattern evaluation
    - Velocity checking (impossible travel detection)
    - ML-enhanced anomaly detection
    - Adaptive risk scoring based on user history

    Example:
        >>> risk_node = RiskAssessmentNode(
        ...     risk_factors=["ip_reputation", "device_trust", "location", "behavior"],
        ...     threshold_low=0.3,
        ...     threshold_medium=0.6,
        ...     threshold_high=0.8,
        ...     ml_enabled=True
        ... )
        >>>
        >>> context = {
        ...     "user_id": "user123",
        ...     "ip_address": "203.0.113.100",
        ...     "device_info": {"device_id": "device_456", "recognized": False},
        ...     "timestamp": datetime.now(UTC).isoformat()
        ... }
        >>>
        >>> result = risk_node.execute(action="assess", context=context)
        >>> print(f"Risk level: {result['risk_level']}")
    """

    def __init__(
        self,
        name: str = "risk_assessment",
        risk_factors: Optional[List[str]] = None,
        threshold_low: float = 0.3,
        threshold_medium: float = 0.6,
        threshold_high: float = 0.8,
        ml_enabled: bool = True,
        geoip_enabled: bool = True,
        velocity_check_enabled: bool = True,
        behavioral_analysis: bool = True,
        **kwargs,
    ):
        """Initialize risk assessment node.

        Args:
            name: Node name
            risk_factors: List of risk factors to evaluate
            threshold_low: Threshold for low risk classification
            threshold_medium: Threshold for medium risk classification
            threshold_high: Threshold for high risk classification
            ml_enabled: Enable machine learning-based analysis
            geoip_enabled: Enable GeoIP location analysis
            velocity_check_enabled: Enable velocity/travel time checking
            behavioral_analysis: Enable behavioral pattern analysis
            **kwargs: Additional node parameters
        """
        # Set attributes before calling super().__init__()
        self.risk_factors = risk_factors or [
            "ip_reputation",
            "device_trust",
            "location",
            "behavior",
            "time_pattern",
        ]
        self.threshold_low = threshold_low
        self.threshold_medium = threshold_medium
        self.threshold_high = threshold_high
        self.ml_enabled = ml_enabled
        self.geoip_enabled = geoip_enabled
        self.velocity_check_enabled = velocity_check_enabled
        self.behavioral_analysis = behavioral_analysis

        # Initialize parent classes
        super().__init__(name=name, **kwargs)

        # User history storage
        self.user_history: Dict[str, List[Dict[str, Any]]] = {}
        self.successful_auths: Dict[str, List[Dict[str, Any]]] = {}

        # Risk assessment statistics
        self.assessment_stats = {
            "total_assessments": 0,
            "high_risk_count": 0,
            "low_risk_count": 0,
            "blocked_attempts": 0,
            "avg_assessment_time_ms": 0,
        }

    def get_parameters(self) -> Dict[str, NodeParameter]:
        """Get node parameters for validation and documentation.

        Returns:
            Dictionary mapping parameter names to NodeParameter objects
        """
        return {
            "action": NodeParameter(
                name="action",
                type=str,
                description="Risk assessment action to perform",
                required=True,
            ),
            "context": NodeParameter(
                name="context",
                type=dict,
                description="Risk context for assessment",
                required=True,
            ),
            "include_mitigation": NodeParameter(
                name="include_mitigation",
                type=bool,
                description="Include mitigation recommendations",
                required=False,
                default=False,
            ),
        }

    def run(
        self,
        action: str,
        context: Dict[str, Any],
        include_mitigation: bool = False,
        **kwargs,
    ) -> Dict[str, Any]:
        """Run risk assessment.

        Args:
            action: Assessment action to perform
            context: Risk context
            include_mitigation: Include mitigation recommendations
            **kwargs: Additional parameters

        Returns:
            Dictionary containing risk assessment results
        """
        start_time = datetime.now(UTC)

        try:
            # Basic validation without deep sanitization to preserve context structure
            if not isinstance(action, str):
                raise ValueError("Action must be a string")
            if not isinstance(context, dict):
                raise ValueError("Context must be a dictionary")
            if not isinstance(include_mitigation, bool):
                include_mitigation = bool(include_mitigation)

            self.log_node_execution("risk_assessment_start", action=action)

            # Route to appropriate action handler
            if action == "assess":
                result = self._assess_risk(context, include_mitigation)
            elif action == "record_successful_auth":
                result = self._record_successful_auth(context)
            else:
                result = {"success": False, "error": f"Unknown action: {action}"}

            # Update statistics
            processing_time = (datetime.now(UTC) - start_time).total_seconds() * 1000
            self._update_stats(processing_time, result.get("risk_level", "unknown"))

            # Add timing information
            result["processing_time_ms"] = processing_time
            result["timestamp"] = start_time.isoformat()

            self.log_node_execution(
                "risk_assessment_complete",
                action=action,
                risk_level=result.get("risk_level", "unknown"),
                processing_time_ms=processing_time,
            )

            return result

        except Exception as e:
            self.log_error_with_traceback(e, "risk_assessment")
            raise

    async def execute_async(self, **inputs) -> Dict[str, Any]:
        """Async execution method for enterprise integration."""
        return self.execute(**inputs)

    def _assess_risk(
        self, context: Dict[str, Any], include_mitigation: bool = False
    ) -> Dict[str, Any]:
        """Assess authentication risk based on context.

        Args:
            context: Risk context
            include_mitigation: Include mitigation recommendations

        Returns:
            Risk assessment results
        """
        risk_context = self._parse_risk_context(context)

        # Record this assessment attempt in history for velocity checks
        self._record_assessment_attempt(risk_context)

        # Initialize risk factors
        risk_factors = []
        trust_factors = []
        risk_score = 0.0

        # Evaluate each risk factor
        for factor in self.risk_factors:
            factor_result = self._evaluate_risk_factor(factor, risk_context)
            risk_score += factor_result["score"]

            if factor_result["risk_indicators"]:
                risk_factors.extend(factor_result["risk_indicators"])

            if factor_result["trust_indicators"]:
                trust_factors.extend(factor_result["trust_indicators"])

        # Normalize risk score - don't divide by number of factors
        risk_score = min(1.0, risk_score)

        # Determine risk level
        risk_level = self._determine_risk_level(risk_score)

        # Check for adaptive adjustments based on history
        if self.behavioral_analysis:
            adjusted_score, adjustment_factors = self._apply_behavioral_adjustments(
                risk_context, risk_score
            )
            risk_score = adjusted_score
            risk_level = self._determine_risk_level(risk_score)

            if adjustment_factors["trust"]:
                trust_factors.extend(adjustment_factors["trust"])
            if adjustment_factors["risk"]:
                risk_factors.extend(adjustment_factors["risk"])

        # Prepare result
        result = {
            "success": True,
            "risk_score": round(risk_score, 3),
            "risk_level": risk_level.value,
            "risk_factors": list(set(risk_factors)),
            "trust_factors": list(set(trust_factors)) if trust_factors else [],
            "confidence": self._calculate_confidence(risk_context, risk_factors),
        }

        # Add mitigation recommendations if requested
        if include_mitigation:
            mitigation = self._generate_mitigation_recommendations(
                risk_level, risk_factors
            )
            result["mitigation_required"] = mitigation["required"]
            result["additional_checks"] = mitigation["additional_checks"]

        # Add location details if available
        if hasattr(risk_context, "location") and risk_context.location:
            result["location_details"] = risk_context.location

        return result

    def _record_assessment_attempt(self, context: RiskContext) -> None:
        """Record assessment attempt in history for velocity checks.

        Args:
            context: Risk context to record
        """
        user_id = context.user_id

        # Initialize user history if needed
        if user_id not in self.user_history:
            self.user_history[user_id] = []

        # Record in history
        auth_record = {
            "timestamp": context.timestamp,
            "ip_address": context.ip_address,
            "device_id": context.device_info.get("device_id"),
            "location": context.location,
        }

        self.user_history[user_id].append(auth_record)

        # Limit history size
        if len(self.user_history[user_id]) > 100:
            self.user_history[user_id] = self.user_history[user_id][-100:]

    def _parse_risk_context(self, context: Dict[str, Any]) -> RiskContext:
        """Parse risk context from input.

        Args:
            context: Raw context dictionary

        Returns:
            Parsed risk context
        """
        return RiskContext(
            user_id=context["user_id"],
            ip_address=context["ip_address"],
            device_info=context.get("device_info", {}),
            timestamp=context["timestamp"],
            location=context.get("location"),
            user_timezone=context.get("user_timezone"),
            usual_hours=context.get("usual_hours"),
            usual_locations=context.get("usual_locations", []),
        )

    def _evaluate_risk_factor(
        self, factor: str, context: RiskContext
    ) -> Dict[str, Any]:
        """Evaluate a specific risk factor.

        Args:
            factor: Risk factor to evaluate
            context: Risk context

        Returns:
            Factor evaluation result
        """
        if factor == "ip_reputation":
            return self._evaluate_ip_reputation(context)
        elif factor == "device_trust":
            return self._evaluate_device_trust(context)
        elif factor == "location":
            return self._evaluate_location_risk(context)
        elif factor == "behavior":
            return self._evaluate_behavioral_risk(context)
        elif factor == "time_pattern":
            return self._evaluate_time_pattern_risk(context)
        else:
            return {"score": 0.0, "risk_indicators": [], "trust_indicators": []}

    def _evaluate_ip_reputation(self, context: RiskContext) -> Dict[str, Any]:
        """Evaluate IP address reputation risk.

        Args:
            context: Risk context

        Returns:
            IP reputation evaluation
        """
        ip = context.ip_address
        risk_indicators = []
        trust_indicators = []
        score = 0.0

        # Check if IP is internal/corporate
        if ip.startswith(("10.", "172.", "192.168.")):
            trust_indicators.append("corporate_network")
            score = 0.05  # Very low risk for internal networks
        elif ip.startswith("127."):
            trust_indicators.append("localhost")
            score = 0.0
        else:
            # External IP - moderate risk
            risk_indicators.append("external_ip")
            score = 0.37

            # Check for known suspicious IP patterns
            if self._is_suspicious_ip(ip):
                risk_indicators.append("suspicious_ip")
                score = 0.8

        return {
            "score": score,
            "risk_indicators": risk_indicators,
            "trust_indicators": trust_indicators,
        }

    def _is_suspicious_ip(self, ip: str) -> bool:
        """Check if IP is suspicious (simplified check).

        Args:
            ip: IP address to check

        Returns:
            True if IP appears suspicious
        """
        # Simplified suspicious IP detection
        # In production, this would check against threat intelligence feeds
        suspicious_patterns = [
            "185.220.",  # Known Tor exits
            "198.96.",  # Known proxy services
            "198.98.",  # Known proxy services
        ]

        return any(ip.startswith(pattern) for pattern in suspicious_patterns)

    def _evaluate_device_trust(self, context: RiskContext) -> Dict[str, Any]:
        """Evaluate device trust level.

        Args:
            context: Risk context

        Returns:
            Device trust evaluation
        """
        device_info = context.device_info
        risk_indicators = []
        trust_indicators = []
        score = 0.0

        # Check if device is recognized
        if device_info.get("recognized", False):
            trust_indicators.append("recognized_device")
            score = 0.02  # Very low score for recognized devices
        else:
            risk_indicators.append("unrecognized_device")
            score = 0.4  # Moderate risk for unrecognized devices

        # Check device type
        device_type = device_info.get("device_type", "unknown")
        if device_type in ["desktop", "laptop"]:
            trust_indicators.append("managed_device_type")
            # No additional score for good device types
        elif device_type == "mobile":
            score += 0.1  # Slight risk for mobile
            risk_indicators.append("mobile_device")
        elif device_type == "unknown":
            risk_indicators.append("unknown_device_type")
            score += 0.2  # Moderate risk for unknown devices

        return {
            "score": score,
            "risk_indicators": risk_indicators,
            "trust_indicators": trust_indicators,
        }

    def _evaluate_location_risk(self, context: RiskContext) -> Dict[str, Any]:
        """Evaluate location-based risk.

        Args:
            context: Risk context

        Returns:
            Location risk evaluation
        """
        risk_indicators = []
        trust_indicators = []
        score = 0.0

        # If we have location information
        if context.location:
            country = context.location.get("country")
            city = context.location.get("city")

            # Check against usual locations
            if context.usual_locations and country:
                if country in context.usual_locations:
                    trust_indicators.append("usual_location")
                    score = 0.1
                else:
                    risk_indicators.append("unusual_location")
                    score = 0.6

            # Check for velocity (impossible travel)
            if self.velocity_check_enabled:
                velocity_risk = self._check_velocity(context)
                if velocity_risk["impossible_travel"]:
                    risk_indicators.append("impossible_travel")
                    score = max(score, 0.9)
        else:
            # No location info available - low risk since it's common
            risk_indicators.append("no_location_data")
            score = 0.05

        return {
            "score": score,
            "risk_indicators": risk_indicators,
            "trust_indicators": trust_indicators,
        }

    def _check_velocity(self, context: RiskContext) -> Dict[str, Any]:
        """Check for impossible travel velocity.

        Args:
            context: Risk context

        Returns:
            Velocity check result
        """
        user_id = context.user_id
        current_time = datetime.fromisoformat(context.timestamp.replace("Z", "+00:00"))

        # Get recent authentication history
        if user_id in self.user_history:
            recent_auths = [
                auth
                for auth in self.user_history[user_id]
                if (
                    current_time
                    - datetime.fromisoformat(auth["timestamp"].replace("Z", "+00:00"))
                ).total_seconds()
                < 3600
            ]

            # Get previous auths (excluding current timestamp to avoid comparing against self)
            previous_auths = [
                auth for auth in recent_auths if auth["timestamp"] != context.timestamp
            ]

            if previous_auths:
                # Check last auth location vs current
                last_auth = previous_auths[-1]
                if "location" in last_auth and context.location:
                    last_location = last_auth["location"]
                    current_location = context.location

                    # Check if locations are different
                    if last_location.get("city") != current_location.get(
                        "city"
                    ) or last_location.get("country") != current_location.get(
                        "country"
                    ):

                        # Check time difference - less than 1 hour for different cities/countries = impossible
                        time_diff = (
                            current_time
                            - datetime.fromisoformat(
                                last_auth["timestamp"].replace("Z", "+00:00")
                            )
                        ).total_seconds()
                        if time_diff < 3600:  # Less than 1 hour
                            return {"impossible_travel": True}

        return {"impossible_travel": False}

    def _evaluate_behavioral_risk(self, context: RiskContext) -> Dict[str, Any]:
        """Evaluate behavioral risk patterns.

        Args:
            context: Risk context

        Returns:
            Behavioral risk evaluation
        """
        risk_indicators = []
        trust_indicators = []
        score = 0.0

        # This would integrate with behavioral analysis in production
        # For now, basic heuristics

        user_id = context.user_id
        if user_id in self.successful_auths:
            recent_successes = len(self.successful_auths[user_id])
            if recent_successes > 10:
                trust_indicators.append("established_pattern")
                score = 0.05  # Very low risk for established users
            elif recent_successes < 3:
                risk_indicators.append("new_user_pattern")
                score = 0.2  # Lower risk for new users
        else:
            risk_indicators.append("no_auth_history")
            score = 0.15  # Lower risk for unknown users (first time is normal)

        return {
            "score": score,
            "risk_indicators": risk_indicators,
            "trust_indicators": trust_indicators,
        }

    def _evaluate_time_pattern_risk(self, context: RiskContext) -> Dict[str, Any]:
        """Evaluate time-based access pattern risk.

        Args:
            context: Risk context

        Returns:
            Time pattern risk evaluation
        """
        risk_indicators = []
        trust_indicators = []
        score = 0.0

        # Parse current time
        try:
            # Handle different timestamp formats
            timestamp = context.timestamp
            if timestamp.endswith("Z"):
                timestamp = timestamp.replace("Z", "+00:00")
            elif not timestamp.endswith("+00:00") and "+" not in timestamp[-6:]:
                # Add UTC timezone if none provided
                timestamp = timestamp + "+00:00"

            current_time = datetime.fromisoformat(timestamp)
            hour = current_time.hour

            # Check against usual hours
            if context.usual_hours:
                start_hour = context.usual_hours.get("start", 9)
                end_hour = context.usual_hours.get("end", 17)

                # Debug output
                # print(f"Time check: hour={hour}, start={start_hour}, end={end_hour}, usual_hours={context.usual_hours}")

                if start_hour <= hour <= end_hour:
                    trust_indicators.append("normal_hours")
                    score = 0.1
                else:
                    risk_indicators.append("unusual_time")
                    score = 0.5  # Higher risk for off-hours
            else:
                # No usual hours defined, check for very unusual times only
                if hour < 3 or hour > 23:
                    risk_indicators.append("unusual_time")
                    score = 0.3
        except Exception as e:
            # Could not parse time
            risk_indicators.append("invalid_timestamp")
            score = 0.2

        return {
            "score": score,
            "risk_indicators": risk_indicators,
            "trust_indicators": trust_indicators,
        }

    def _apply_behavioral_adjustments(
        self, context: RiskContext, base_score: float
    ) -> tuple[float, Dict[str, List[str]]]:
        """Apply behavioral adjustments to risk score.

        Args:
            context: Risk context
            base_score: Base risk score

        Returns:
            Tuple of (adjusted_score, adjustment_factors)
        """
        adjustment_factors = {"trust": [], "risk": []}
        adjusted_score = base_score

        user_id = context.user_id

        # Check for consistent pattern
        if user_id in self.successful_auths:
            successes = self.successful_auths[user_id]

            # Look for consistent IP/device patterns
            if len(successes) >= 5:
                recent_ips = [auth.get("ip_address") for auth in successes[-5:]]
                recent_devices = [auth.get("device_id") for auth in successes[-5:]]

                if (
                    context.ip_address in recent_ips
                    and context.device_info.get("device_id") in recent_devices
                ):
                    adjustment_factors["trust"].append("consistent_pattern")
                    adjusted_score *= (
                        0.3  # Significantly reduce risk for consistent patterns
                    )

        return adjusted_score, adjustment_factors

    def _determine_risk_level(self, risk_score: float) -> RiskLevel:
        """Determine risk level from score.

        Args:
            risk_score: Calculated risk score (0-1)

        Returns:
            Risk level enum
        """
        if risk_score >= self.threshold_high:
            return RiskLevel.HIGH
        elif risk_score >= self.threshold_medium:
            return RiskLevel.HIGH  # This should be HIGH for >= 0.6
        elif risk_score >= self.threshold_low:
            return RiskLevel.MEDIUM  # This should be MEDIUM for >= 0.3
        else:
            return RiskLevel.LOW

    def _calculate_confidence(
        self, context: RiskContext, risk_factors: List[str]
    ) -> float:
        """Calculate confidence in risk assessment.

        Args:
            context: Risk context
            risk_factors: Identified risk factors

        Returns:
            Confidence score (0-1)
        """
        # Base confidence
        confidence = 0.7

        # Increase confidence with more data points
        if context.location:
            confidence += 0.1
        if context.device_info:
            confidence += 0.1
        if context.usual_locations:
            confidence += 0.1

        # Decrease confidence for ambiguous factors
        if "no_location_data" in risk_factors:
            confidence -= 0.2
        if "unknown_device_type" in risk_factors:
            confidence -= 0.1

        return min(1.0, max(0.3, confidence))

    def _generate_mitigation_recommendations(
        self, risk_level: RiskLevel, risk_factors: List[str]
    ) -> Dict[str, List[str]]:
        """Generate mitigation recommendations based on risk.

        Args:
            risk_level: Assessed risk level
            risk_factors: Identified risk factors

        Returns:
            Mitigation recommendations
        """
        required = []
        additional_checks = []

        if risk_level == RiskLevel.HIGH:
            required.extend(["mfa"])
            additional_checks.extend(
                ["email_verification", "security_questions", "device_verification"]
            )
        elif risk_level == RiskLevel.MEDIUM:
            required.append("mfa")
            additional_checks.append("email_verification")

        # Factor-specific recommendations
        if "unrecognized_device" in risk_factors:
            additional_checks.append("device_registration")
        if "unusual_location" in risk_factors:
            additional_checks.append("location_verification")
        if "suspicious_ip" in risk_factors:
            required.append("admin_approval")

        return {
            "required": list(set(required)),
            "additional_checks": list(set(additional_checks)),
        }

    def _record_successful_auth(self, context: Dict[str, Any]) -> Dict[str, Any]:
        """Record successful authentication for behavioral learning.

        Args:
            context: Authentication context

        Returns:
            Recording result
        """
        user_id = context["user_id"]

        # Initialize user history if needed
        if user_id not in self.user_history:
            self.user_history[user_id] = []
        if user_id not in self.successful_auths:
            self.successful_auths[user_id] = []

        # Record in history
        auth_record = {
            "timestamp": context["timestamp"],
            "ip_address": context["ip_address"],
            "device_id": context.get("device_info", {}).get("device_id"),
            "location": context.get("location"),
        }

        self.user_history[user_id].append(auth_record)
        self.successful_auths[user_id].append(auth_record)

        # Limit history size
        if len(self.user_history[user_id]) > 100:
            self.user_history[user_id] = self.user_history[user_id][-100:]
        if len(self.successful_auths[user_id]) > 50:
            self.successful_auths[user_id] = self.successful_auths[user_id][-50:]

        return {"success": True, "recorded": True, "user_id": user_id}

    def _update_stats(self, processing_time_ms: float, risk_level: str) -> None:
        """Update assessment statistics.

        Args:
            processing_time_ms: Processing time in milliseconds
            risk_level: Assessed risk level
        """
        self.assessment_stats["total_assessments"] += 1

        if risk_level == "high":
            self.assessment_stats["high_risk_count"] += 1
        elif risk_level == "low":
            self.assessment_stats["low_risk_count"] += 1

        # Update average processing time
        if self.assessment_stats["avg_assessment_time_ms"] == 0:
            self.assessment_stats["avg_assessment_time_ms"] = processing_time_ms
        else:
            self.assessment_stats["avg_assessment_time_ms"] = (
                self.assessment_stats["avg_assessment_time_ms"] * 0.9
                + processing_time_ms * 0.1
            )

    def get_assessment_stats(self) -> Dict[str, Any]:
        """Get risk assessment statistics.

        Returns:
            Dictionary with assessment statistics
        """
        return {
            **self.assessment_stats,
            "risk_factors_enabled": self.risk_factors,
            "ml_enabled": self.ml_enabled,
            "velocity_check_enabled": self.velocity_check_enabled,
            "behavioral_analysis": self.behavioral_analysis,
        }
