"""Tests for deploy service orchestration functions."""

from unittest.mock import MagicMock, patch
import pytest
import tempfile
from pathlib import Path
import typer

from runlayer_cli.deploy.service import (
    _get_or_create_deployment,
    _build_docker_image,
    _push_to_ecr,
    _update_deployment_config,
    _extract_validation_error,
    _validate_runlayer_yaml_config,
    validate_service,
    deploy_service,
    init_deployment_config,
    destroy_deployment,
)
from runlayer_cli.api import (
    RunlayerClient,
    DeploymentPublic,
    ECRCredentials,
    ValidateYAMLResponse,
)
import datetime
import httpx
import yaml


@pytest.fixture
def mock_api_client():
    """Create a mock API client."""
    return MagicMock(spec=RunlayerClient)


@pytest.fixture
def sample_config():
    """Sample deployment configuration."""
    return {
        "name": "test-service",
        "runtime": "docker",
        "build": {
            "dockerfile": "Dockerfile",
            "context": ".",
            "platform": "x86",
        },
        "service": {"port": 8000},
    }


def test_get_or_create_deployment_existing(mock_api_client, sample_config):
    """Test using existing deployment ID."""
    deployment_id = "existing-deployment-id"
    sample_config["id"] = deployment_id

    mock_deployment = MagicMock(spec=DeploymentPublic)
    mock_deployment.id = deployment_id
    mock_deployment.name = "test-service"
    mock_deployment.deletion_status = None
    mock_api_client.get_deployment.return_value = mock_deployment

    result = _get_or_create_deployment(
        mock_api_client, sample_config, "test-config.yaml"
    )

    assert result == deployment_id
    mock_api_client.get_deployment.assert_called_once_with(deployment_id)
    mock_api_client.create_deployment.assert_not_called()


def test_get_or_create_deployment_new(mock_api_client, sample_config):
    """Test creating new deployment when ID is missing."""
    with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
        import yaml

        yaml.dump(sample_config, f)
        config_path = f.name

    try:
        mock_deployment = MagicMock(spec=DeploymentPublic)
        mock_deployment.id = "new-deployment-id"
        mock_api_client.create_deployment.return_value = mock_deployment

        with patch("runlayer_cli.deploy.service.update_config_id") as mock_update:
            result = _get_or_create_deployment(
                mock_api_client, sample_config, config_path
            )

            assert result == "new-deployment-id"
            mock_api_client.create_deployment.assert_called_once_with("test-service")
            mock_update.assert_called_once_with(config_path, "new-deployment-id")
    finally:
        Path(config_path).unlink()


def test_get_or_create_deployment_deleted_status(mock_api_client, sample_config):
    """Test that deleted deployment is rejected."""
    deployment_id = "deleted-deployment-id"
    sample_config["id"] = deployment_id

    mock_deployment = MagicMock(spec=DeploymentPublic)
    mock_deployment.id = deployment_id
    mock_deployment.deletion_status = "deleted"
    mock_api_client.get_deployment.return_value = mock_deployment

    with pytest.raises(typer.Exit):
        _get_or_create_deployment(mock_api_client, sample_config, "test-config.yaml")


def test_build_docker_image_success(sample_config):
    """Test successful Docker image build with platform config."""
    deployment_id = "test-deployment-id"

    with (
        patch("runlayer_cli.deploy.service.build_image") as mock_build,
        patch("runlayer_cli.deploy.service.typer") as mock_typer,
    ):
        mock_build.return_value = "sha256:abc123def456"

        result = _build_docker_image(sample_config, deployment_id)

        assert result == "sha256:abc123def456"
        mock_build.assert_called_once()
        call_kwargs = mock_build.call_args[1]
        assert call_kwargs["tag"] == f"runlayer-build:{deployment_id}"
        assert call_kwargs["platform"] == "linux/amd64"


def test_push_to_ecr_success(mock_api_client):
    """Test successful push to ECR and return digest URI."""
    image_id = "sha256:abc123"
    deployment_id = "test-deployment-id"

    expires_at = datetime.datetime.now(datetime.timezone.utc) + datetime.timedelta(
        hours=1
    )
    ecr_creds = ECRCredentials(
        username="AWS",
        password="test-password",
        registry_url="https://123456789.dkr.ecr.us-east-1.amazonaws.com",
        repository_url="123456789.dkr.ecr.us-east-1.amazonaws.com/my-repo",
        expires_at=expires_at,
    )
    mock_api_client.get_ecr_credentials.return_value = ecr_creds

    with (
        patch("runlayer_cli.deploy.service.authenticate_ecr") as mock_auth,
        patch("runlayer_cli.deploy.service.tag_image") as mock_tag,
        patch("runlayer_cli.deploy.service.push_image") as mock_push,
        patch("runlayer_cli.deploy.service.typer") as mock_typer,
    ):
        mock_tag.return_value = (
            "123456789.dkr.ecr.us-east-1.amazonaws.com/my-repo:test-deployment-id"
        )
        mock_push.return_value = "sha256:def789ghi012"

        result = _push_to_ecr(mock_api_client, image_id, deployment_id)

        expected_uri = (
            "123456789.dkr.ecr.us-east-1.amazonaws.com/my-repo@sha256:def789ghi012"
        )
        assert result == expected_uri
        mock_auth.assert_called_once()
        mock_tag.assert_called_once()
        mock_push.assert_called_once()


def test_update_deployment_config_success(mock_api_client):
    """Test successful deployment configuration update."""
    deployment_id = "test-deployment-id"
    image_uri = "repo@sha256:abc123"
    yaml_content = "name: test-service\nruntime: docker\n"

    mock_deployment = MagicMock(spec=DeploymentPublic)
    mock_api_client.update_deployment.return_value = mock_deployment

    with patch("runlayer_cli.deploy.service.typer") as mock_typer:
        _update_deployment_config(
            mock_api_client, deployment_id, image_uri, yaml_content
        )

        mock_api_client.update_deployment.assert_called_once_with(
            deployment_id,
            yaml_content=yaml_content,
            docker_image=image_uri,
        )


def test_init_deployment_config_success():
    """Test successful deployment initialization."""
    with tempfile.TemporaryDirectory() as tmpdir:
        config_path = Path(tmpdir) / "runlayer.yaml"

        mock_deployment = MagicMock(spec=DeploymentPublic)
        mock_deployment.id = "new-deployment-id"
        mock_deployment.template_yaml = "name: test-service\nruntime: docker\n"

        with (
            patch("runlayer_cli.deploy.service.RunlayerClient") as mock_client_class,
            patch("runlayer_cli.deploy.service.typer") as mock_typer,
        ):
            mock_client = MagicMock()
            mock_client.create_deployment.return_value = mock_deployment
            mock_client_class.return_value = mock_client

            init_deployment_config(
                name="test-service",
                config_path=str(config_path),
                secret="test-secret",
                host="http://localhost:3000",
            )

            mock_client.create_deployment.assert_called_once_with("test-service")
            assert config_path.exists()
            assert "test-service" in config_path.read_text()


def test_init_deployment_config_invalid_name():
    """Test that invalid deployment name is rejected."""
    with patch("runlayer_cli.deploy.service.typer.echo") as mock_echo:
        with pytest.raises(typer.Exit) as exc_info:
            init_deployment_config(
                name="Invalid Name With Spaces!",
                config_path="runlayer.yaml",
                secret="test-secret",
                host="http://localhost:3000",
            )

        assert exc_info.value.exit_code == 1


def test_destroy_deployment_success(mock_api_client):
    """Test successful deployment destruction."""
    deployment_id = "test-deployment-id"

    mock_deployment = MagicMock(spec=DeploymentPublic)
    mock_deployment.id = deployment_id
    mock_deployment.name = "test-service"
    mock_deployment.created_at = datetime.datetime.now(datetime.timezone.utc)
    mock_deployment.deletion_status = None
    mock_deployment.connected_servers = []

    with (
        patch("runlayer_cli.deploy.service.RunlayerClient") as mock_client_class,
        patch("runlayer_cli.deploy.service.load_config") as mock_load,
        patch("runlayer_cli.deploy.service.typer.confirm") as mock_confirm,
        patch("runlayer_cli.deploy.service.typer.echo") as mock_echo,
        patch("runlayer_cli.deploy.service.typer.secho") as mock_secho,
    ):
        mock_client = MagicMock()
        mock_client.get_deployment.return_value = mock_deployment
        mock_client_class.return_value = mock_client

        mock_load.return_value = {"id": deployment_id}
        mock_confirm.return_value = True

        destroy_deployment(
            config_path="runlayer.yaml",
            secret="test-secret",
            host="http://localhost:3000",
            deployment_id=deployment_id,
        )

        mock_client.delete_deployment.assert_called_once_with(deployment_id)


def test_destroy_deployment_already_deleted(mock_api_client):
    """Test handling of already deleted deployment."""
    deployment_id = "deleted-deployment-id"

    mock_deployment = MagicMock(spec=DeploymentPublic)
    mock_deployment.id = deployment_id
    mock_deployment.name = "test-service"
    mock_deployment.created_at = datetime.datetime.now(datetime.timezone.utc)
    mock_deployment.deletion_status = "deleted"

    with (
        patch("runlayer_cli.deploy.service.RunlayerClient") as mock_client_class,
        patch("runlayer_cli.deploy.service.typer.echo") as mock_echo,
        patch("runlayer_cli.deploy.service.typer.secho") as mock_secho,
    ):
        mock_client = MagicMock()
        mock_client.get_deployment.return_value = mock_deployment
        mock_client_class.return_value = mock_client

        with pytest.raises(typer.Exit) as exc_info:
            destroy_deployment(
                config_path="runlayer.yaml",
                secret="test-secret",
                host="http://localhost:3000",
                deployment_id=deployment_id,
            )

        assert exc_info.value.exit_code == 0
        mock_client.delete_deployment.assert_not_called()


def test_extract_validation_error_from_422():
    """Test error extraction from 422 HTTP status error."""
    # Create a mock HTTPStatusError with 422 status
    mock_response = MagicMock()
    mock_response.status_code = 422
    mock_response.json.return_value = {
        "detail": "Configuration error: service.port is required"
    }
    mock_response.text = ""

    error = httpx.HTTPStatusError(
        message="422 Unprocessable Entity",
        request=MagicMock(),
        response=mock_response,
    )

    result = _extract_validation_error(error)
    assert result == "Configuration error: service.port is required"


def test_extract_validation_error_from_422_no_detail():
    """Test error extraction from 422 when detail field is missing."""
    mock_response = MagicMock()
    mock_response.status_code = 422
    mock_response.json.return_value = {"error": "Something went wrong"}
    mock_response.text = ""

    error = httpx.HTTPStatusError(
        message="422 Unprocessable Entity",
        request=MagicMock(),
        response=mock_response,
    )

    result = _extract_validation_error(error)
    assert "error" in result.lower() or "something" in result.lower()


def test_extract_validation_error_from_non_422():
    """Test error extraction from non-422 HTTP errors."""
    mock_response = MagicMock()
    mock_response.status_code = 500
    mock_response.text = "Internal Server Error"

    error = httpx.HTTPStatusError(
        message="500 Internal Server Error",
        request=MagicMock(),
        response=mock_response,
    )

    result = _extract_validation_error(error)
    assert "500" in result
    assert "Internal Server Error" in result


def test_extract_validation_error_generic():
    """Test error extraction from generic exceptions."""
    error = ValueError("Something went wrong")
    result = _extract_validation_error(error)
    assert result == "Something went wrong"


def test_validate_runlayer_yaml_config_success(mock_api_client):
    """Test successful YAML validation."""
    yaml_content = "name: test-service\nruntime: docker\nservice:\n  port: 8000\n"

    mock_response = ValidateYAMLResponse(
        valid=True,
        error=None,
        parsed_config={"name": "test-service"},
    )
    mock_api_client.validate_yaml.return_value = mock_response

    with (
        patch("runlayer_cli.deploy.service.typer.echo") as _mock_echo,
        patch("runlayer_cli.deploy.service.typer.secho") as mock_secho,
    ):
        _validate_runlayer_yaml_config(mock_api_client, yaml_content)

        mock_api_client.validate_yaml.assert_called_once_with(yaml_content)
        mock_secho.assert_called()
        # Check that success message was shown
        calls = [str(call) for call in mock_secho.call_args_list]
        assert any("valid" in str(call).lower() for call in calls)


def test_validate_runlayer_yaml_config_invalid(mock_api_client):
    """Test YAML validation with invalid configuration."""
    yaml_content = "invalid: yaml"

    mock_response = ValidateYAMLResponse(
        valid=False,
        error="Configuration error: service.port is required",
        parsed_config=None,
    )
    mock_api_client.validate_yaml.return_value = mock_response

    with (
        patch("runlayer_cli.deploy.service.typer.echo") as _mock_echo,
        patch("runlayer_cli.deploy.service.typer.secho") as _mock_secho,
    ):
        with pytest.raises(typer.Exit) as exc_info:
            _validate_runlayer_yaml_config(mock_api_client, yaml_content)

        assert exc_info.value.exit_code == 1
        mock_api_client.validate_yaml.assert_called_once_with(yaml_content)


def test_validate_runlayer_yaml_config_http_error(mock_api_client):
    """Test YAML validation with HTTP error."""
    yaml_content = "name: test-service\nruntime: docker\n"

    mock_response = MagicMock()
    mock_response.status_code = 422
    mock_response.json.return_value = {"detail": "Validation failed"}
    mock_response.text = ""

    http_error = httpx.HTTPStatusError(
        message="422 Unprocessable Entity",
        request=MagicMock(),
        response=mock_response,
    )
    mock_api_client.validate_yaml.side_effect = http_error

    with (
        patch("runlayer_cli.deploy.service.typer.echo") as _mock_echo,
        patch("runlayer_cli.deploy.service.typer.secho") as _mock_secho,
    ):
        with pytest.raises(typer.Exit) as exc_info:
            _validate_runlayer_yaml_config(mock_api_client, yaml_content)

        assert exc_info.value.exit_code == 1


def test_validate_service_success():
    """Test successful validation service call."""
    with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
        config = {
            "name": "test-service",
            "runtime": "docker",
            "service": {"port": 8000},
        }
        yaml.dump(config, f)
        config_path = f.name

    try:
        mock_response = ValidateYAMLResponse(
            valid=True,
            error=None,
            parsed_config=config,
        )

        with (
            patch("runlayer_cli.deploy.service.RunlayerClient") as mock_client_class,
            patch("runlayer_cli.deploy.service.typer.echo") as _mock_echo,
            patch("runlayer_cli.deploy.service.typer.secho") as _mock_secho,
        ):
            mock_client = MagicMock()
            mock_client.validate_yaml.return_value = mock_response
            mock_client_class.return_value = mock_client

            validate_service(
                config_path=config_path,
                secret="test-secret",
                host="http://localhost:3000",
            )

            mock_client.validate_yaml.assert_called_once()
    finally:
        Path(config_path).unlink()


def test_validate_service_invalid_yaml():
    """Test validation service with invalid YAML."""
    with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
        f.write("invalid: yaml: content")
        config_path = f.name

    try:
        mock_response = ValidateYAMLResponse(
            valid=False,
            error="Configuration error: service.port is required",
            parsed_config=None,
        )

        with (
            patch("runlayer_cli.deploy.service.RunlayerClient") as mock_client_class,
            patch("runlayer_cli.deploy.service.typer.echo") as _mock_echo,
            patch("runlayer_cli.deploy.service.typer.secho") as _mock_secho,
        ):
            mock_client = MagicMock()
            mock_client.validate_yaml.return_value = mock_response
            mock_client_class.return_value = mock_client

            with pytest.raises(typer.Exit) as exc_info:
                validate_service(
                    config_path=config_path,
                    secret="test-secret",
                    host="http://localhost:3000",
                )

            assert exc_info.value.exit_code == 1
    finally:
        Path(config_path).unlink()


def test_deploy_service_validates_early(mock_api_client, sample_config):
    """Test that deploy_service validates configuration before Docker build."""
    with tempfile.NamedTemporaryFile(mode="w", suffix=".yaml", delete=False) as f:
        yaml.dump(sample_config, f)
        config_path = f.name

    try:
        mock_response = ValidateYAMLResponse(
            valid=True,
            error=None,
            parsed_config=sample_config,
        )
        mock_api_client.validate_yaml.return_value = mock_response

        mock_deployment = MagicMock(spec=DeploymentPublic)
        mock_deployment.id = "test-deployment-id"
        mock_deployment.name = "test-service"
        mock_deployment.deletion_status = None
        mock_api_client.get_deployment.return_value = mock_deployment

        with (
            patch("runlayer_cli.deploy.service.RunlayerClient") as mock_client_class,
            patch("runlayer_cli.deploy.service.load_config") as mock_load_config,
            patch("runlayer_cli.deploy.service.load_config_raw") as mock_load_raw,
            patch(
                "runlayer_cli.deploy.service.check_docker_available"
            ) as mock_docker_check,
            patch("runlayer_cli.deploy.service.typer") as _mock_typer,
        ):
            mock_client_class.return_value = mock_api_client
            mock_load_config.return_value = sample_config
            mock_load_raw.return_value = yaml.dump(sample_config)
            mock_docker_check.return_value = True

            # This should call validate_yaml before Docker build
            # We'll just verify the call order by checking validate_yaml was called
            # Since we're mocking everything, we expect it to fail at Docker build
            # but validation should have been called first
            try:
                deploy_service(
                    config_path=config_path,
                    secret="test-secret",
                    host="http://localhost:3000",
                )
            except (typer.Exit, AttributeError, TypeError):
                # Expected to fail at some point, but validation should have been called
                pass

            # Verify validation was called
            mock_api_client.validate_yaml.assert_called_once()
    finally:
        Path(config_path).unlink()
