"""Deploy command group for Runlayer CLI."""

import typer
from typing import Optional

from runlayer_cli.deploy import (
    deploy_service,
    init_deployment_config,
    destroy_deployment,
    validate_service,
)

app = typer.Typer(help="Deploy services to Runlayer infrastructure")


@app.callback(invoke_without_command=True)
def deploy_callback(
    ctx: typer.Context,
    config_path: str = typer.Option(
        "runlayer.yaml", "--config", "-c", help="Path to runlayer.yaml config file"
    ),
    secret: Optional[str] = typer.Option(
        None, "--secret", "-s", help="API secret for authentication"
    ),
    host: str = typer.Option(
        "http://localhost:3000", "--host", "-H", help="Runlayer API host URL"
    ),
):
    """
    Deploy a service based on runlayer.yaml configuration.

    This command will:
    1. Load and validate the configuration
    2. Create or update the deployment
    3. Build the Docker image
    4. Push the image to ECR
    5. Trigger the deployment
    """
    if ctx.invoked_subcommand is not None:
        return

    if secret is None:
        typer.echo("Error: Missing option '--secret' / '-s'", err=True)
        raise typer.Exit(1)

    deploy_service(config_path=config_path, secret=secret, host=host)


@app.command(name="init", help="Initialize a new deployment configuration")
def init(
    config_path: str = typer.Option(
        "runlayer.yaml",
        "--config",
        "-c",
        help="Path to create runlayer.yaml config file",
    ),
    secret: str = typer.Option(
        ..., "--secret", "-s", help="API secret for authentication"
    ),
    host: str = typer.Option(
        "http://localhost:3000", "--host", "-H", help="Runlayer API host URL"
    ),
):
    """Initialize a new deployment and create runlayer.yaml configuration file."""
    name = typer.prompt("Enter deployment name (lowercase, URL-friendly)")
    init_deployment_config(name=name, config_path=config_path, secret=secret, host=host)


@app.command(name="validate", help="Validate runlayer.yaml configuration")
def validate(
    config_path: str = typer.Option(
        "runlayer.yaml",
        "--config",
        "-c",
        help="Path to runlayer.yaml config file",
    ),
    secret: str = typer.Option(
        ..., "--secret", "-s", help="API secret for authentication"
    ),
    host: str = typer.Option(
        "http://localhost:3000", "--host", "-H", help="Runlayer API host URL"
    ),
):
    """
    Validate runlayer.yaml configuration without deploying.

    This command validates the configuration file by checking:
    - YAML syntax correctness
    - Required fields and structure
    - Configuration values against backend schema

    No deployment is created or modified. Use this to check your
    configuration before running the full deploy command.
    """
    validate_service(config_path=config_path, secret=secret, host=host)


@app.command(name="destroy", help="Destroy a deployment and its infrastructure")
def destroy(
    config_path: str = typer.Option(
        "runlayer.yaml",
        "--config",
        "-c",
        help="Path to runlayer.yaml config file (contains deployment ID)",
    ),
    deployment_id: Optional[str] = typer.Option(
        None,
        "--deployment-id",
        "-d",
        help="Deployment ID to destroy (overrides config file)",
    ),
    secret: str = typer.Option(
        ..., "--secret", "-s", help="API secret for authentication"
    ),
    host: str = typer.Option(
        "http://localhost:3000", "--host", "-H", help="Runlayer API host URL"
    ),
):
    """
    Destroy a deployment and tear down its infrastructure.

    This command will:
    1. Load the deployment ID from config or use provided ID
    2. Confirm the destruction with the user
    3. Trigger infrastructure teardown via the backend
    4. Queue the deletion process

    You can specify the deployment either by:
    - Using --config to read the ID from runlayer.yaml (default)
    - Using --deployment-id to specify the ID directly
    """
    destroy_deployment(
        config_path=config_path,
        secret=secret,
        host=host,
        deployment_id=deployment_id,
    )
