def axisymmetric_file(geometry_type, geometry_parameters, Nrank, wavelength, 
        index, index_m, kb=None, conducting=False, Nparam=1, 
        use_ds=True, complex_plane=True, eps_z_re_im=0.95, Nint=200):
    """Create input file for axisymmetric particles

    Arguments:
        geometry_type (int)          choose from 1 (spheroid), 2 (cylinder), 3 (rounded oblate cylinder)
        geometry_parameters (list)   geometric parameters ([radius along symmetry axius, radius along other axes])
        Nrank (int)                  maximum number of multipoles
        wavelength (float)           wavelength of incident light
        index (complex)              index of refraction of the particle
        index_m (float)              index of refraction of the medium
        kb (float)                   parameter of chirality (default: None [no chirality])
        conducting (bool)            if True, particle is conducting (default: False)
        Nparam (int)                 number of smooth curves used in approximate surface (default: 1)
        use_ds (bool)                if True, use discrete sources (default: True)
        complex_plane (bool)         if True, distribute discrete sources in complex plane (default: True)
        eps_z_re_im (float)          parameter used to distribute discrete sources (default: 0.95)
        Nint (int)                   number of points used in integration (default: 200)
    """
    geometry_xy = geometry_parameters[0]/wavelength
    geometry_z = geometry_parameters[1]/wavelength
    wavelength = 1

    if kb is None:
        chiral = False
        kb = 1
    else:
        chiral = True

    file_str_template = """OptProp
{wavelength}
{index_m.real}
({index.real}, {index.imag})
Variables:
 - wavelength - wavelength of the incident light in vacuo.
 - ind_refMed - refractive index of the ambient medium.
 - ind_refRel - relative refractive index of the particle.  

MatProp
.{conducting}.
.{chiral}.
{kb}
 Variables:
 - perfectcond - if perfectcond = t, the particle is perfectly conducting.
 - chiral      - if chiral = t, the particle is optical active (chiral).
 - kb          - parameter of chirality.

GeomProp
.false.
'../GEOMFILES/prolate.fem'
{geometry_type}
2
{geometry_xy}
{geometry_z}
{Nparam}
1.0
1.0
.false.
 Variables:
 - FileGeom - if FileGeom = t, the particle geometry is supplied by the 
              input file FileFEM. 
 - FileFEM  - name of the file containing the particle geometry. 
 - TypeGeom - parameter specifying the type of the particle geometry.
 - Nsurf	   - number of surface parameters. 
 - surf(1)  - surface parameter.
 - ...  
 - surf(Nsurf  
 - Nparam   - number of smooth curves forming the generatrix curve.    
 - anorm    - characteristic length of the particle which is used to 
              normalize the differential scattering cross sections.	 
 - Rcirc    - characteristic length of the particle for computing Nrank. 
 - miror    - if miror = t, the particle is mirror symmetric.	            
 NOTE: FOR CHIRAL PARTICLES AND DISTRIBUTED SOURCES SET miror = f.

ConvTest
.false.
.false.
 Variables:
 - DoConvTest   - if DoConvTest = t, the interactive convergence tests 
                  over Nint and Nrank are performed.   
 - MishConvTest - if MishConvTest = t, estimates of Nint and Nrank are  
                  computed with the convergence criterion proposed by 
                  Mishchenko.        
 NOTE: IF THE PARTICLE IS OPTICAL ACTIVE (chiral = t) OR THE PARTICLE
 GEOMETRY IS SUPPLIED BY THE FILE FileFEM (FileGeom = t), THE CODE SETS
 MishConvTest = f. IN FACT, MISHCHENKOS CONVERGENCE TEST WILL BE 
 PERFORMED IF (DS = f AND DoConvTest = t AND chiral = f AND FileGeom = f), 
 OR (DS = t AND autGenDS = t AND DoConvTest = t AND chiral = f AND 
 FileGeom = f).   

Sources
.{use_ds}.
.true.
 Variables:
 - DS       - if DS = t, distributed sources are used for T-matrix 
              calculation. 	
 - autGenDS - if autGenDS = t, the coordinates of the distributed sources
              are generated by the code.
 NOTE: IF THE PARTICLE GEOMETRY IS READ FROM FILE (FileGeom = t),
 THE CODE SETS autgenDS = f.                                 

SourcePosAut
.{complex_plane}.
{eps_z_re_im}
 Variables: 
 - ComplexPlane - if ComplexPlane = t, the distributed sources are placed
                  in the complex plane.
 - EpsZReIm     - parameter controlling the distribution of the discrete 
                  sources.
 NOTE: THESE VARIABLES MUST BE PROVIDED IF (DS = t AND autgenDS = t).

NintNrank
{Nint}
{Nrank}
 Variables: 
 - Nint  - number of integration points in computing integrals over the 
           generatrix curve.
 - Nrank - maximum expansion order.  
 NOTE: THESE VARIABLES MUST BE PROVIDED IF ((DoConvTest = f) OR 
 (DS = t AND autgenDS = f)).                  

Errors
5.e-2
5.e-2
1.e-2
4
50
 Variables:
 - epsNint    - error tolerance for the integration test.    
 - epsNrank   - error tolerance for the expansion order test.  
 - epsMrank   - error tolerance for the azimuthal order test.  
 - dNint	     - number of division points for the integration test 
                and Mishchenkos convergence test.   
 - dNintMrank - number of division points for azimuthal mode 
                calculation.

Tmat
'../TMATFILES/tmatrix.dat'
 Variable:
 - FileTmat - name of the file to which the T matrix is written.  

PrintProgress
.false.
 Variable:
 - PrnProgress - if PrnProgress = t, the progress of calculation 
                 is printed. 
"""
    return file_str_template.format(geometry_type=geometry_type, geometry_xy=geometry_xy, geometry_z=geometry_z,
             Nrank=Nrank, wavelength=wavelength, index=index/index_m, index_m=index_m, chiral=str(chiral).lower(), 
             kb=kb, conducting=str(conducting).lower(), Nparam=Nparam, use_ds=str(use_ds).lower(), 
             complex_plane=str(complex_plane).lower(), eps_z_re_im=eps_z_re_im, Nint=Nint)
