"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * An object that represents reasons to restart an InitService.
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *    ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *    ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 *
 * @stability stable
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
_a = JSII_RTTI_SYMBOL_1;
InitServiceRestartHandle[_a] = { fqn: "@aws-cdk/aws-ec2.InitServiceRestartHandle", version: "1.107.0" };
/**
 * Base class for all CloudFormation Init elements.
 *
 * @stability stable
 */
class InitElement {
}
exports.InitElement = InitElement;
_b = JSII_RTTI_SYMBOL_1;
InitElement[_b] = { fqn: "@aws-cdk/aws-ec2.InitElement", version: "1.107.0" };
/**
 * Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 *
 * @stability stable
 */
class InitCommandWaitDuration {
    /**
     * Wait for a specified duration after a command.
     *
     * @stability stable
     */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /**
     * Do not wait for this command.
     *
     * @stability stable
     */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /**
     * cfn-init will exit and resume only after a reboot.
     *
     * @stability stable
     */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
_c = JSII_RTTI_SYMBOL_1;
InitCommandWaitDuration[_c] = { fqn: "@aws-cdk/aws-ec2.InitCommandWaitDuration", version: "1.107.0" };
/**
 * Command to execute on the instance.
 *
 * @stability stable
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * Run a shell command.
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     *
     * @stability stable
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand(shellCommand, options);
    }
    /**
     * Run a command from an argv array.
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     *
     * @stability stable
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _l, _m;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_l = this.options.serviceRestartHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_m = this.options.waitAfterCompletion) === null || _m === void 0 ? void 0 : _m._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
_d = JSII_RTTI_SYMBOL_1;
InitCommand[_d] = { fqn: "@aws-cdk/aws-ec2.InitCommand", version: "1.107.0" };
/**
 * Create files on the EC2 instance.
 *
 * @stability stable
 */
class InitFile extends InitElement {
    /**
     * @stability stable
     */
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * Use a literal string as the file content.
     *
     * @stability stable
     */
    static fromString(fileName, content, options = {}) {
        if (!content) {
            throw new Error(`InitFile ${fileName}: cannot create empty file. Please supply at least one character of content.`);
        }
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Write a symlink with the given symlink target.
     *
     * @stability stable
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     *
     * @stability stable
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Read a file from disk and use its contents.
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     *
     * @stability stable
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * Download from a URL at instance startup time.
     *
     * @stability stable
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Download a file from an S3 bucket at instance startup time.
     *
     * @stability stable
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * Create an asset from the given file.
     *
     * This is appropriate for files that are too large to embed into the template.
     *
     * @stability stable
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /**
     * Use a file from an asset at instance startup time.
     *
     * @stability stable
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _l;
        for (const handle of (_l = this.options.serviceRestartHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {
                [this.fileName]: { ...contentVars },
            };
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
_e = JSII_RTTI_SYMBOL_1;
InitFile[_e] = { fqn: "@aws-cdk/aws-ec2.InitFile", version: "1.107.0" };
/**
 * Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 *
 * @stability stable
 */
class InitGroup extends InitElement {
    /**
     * @stability stable
     */
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * Create a group from its name, and optionally, group id.
     *
     * @stability stable
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
_f = JSII_RTTI_SYMBOL_1;
InitGroup[_f] = { fqn: "@aws-cdk/aws-ec2.InitGroup", version: "1.107.0" };
/**
 * Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 *
 * @stability stable
 */
class InitUser extends InitElement {
    /**
     * @stability stable
     */
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * Create a user from user name.
     *
     * @stability stable
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
_g = JSII_RTTI_SYMBOL_1;
InitUser[_g] = { fqn: "@aws-cdk/aws-ec2.InitUser", version: "1.107.0" };
/**
 * A package to be installed during cfn-init time.
 *
 * @stability stable
 */
class InitPackage extends InitElement {
    /**
     * @stability stable
     */
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * Install an RPM from an HTTP URL or a location on disk.
     *
     * @stability stable
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * Install a package using Yum.
     *
     * @stability stable
     */
    static yum(packageName, options = {}) {
        var _l;
        return new InitPackage('yum', (_l = options.version) !== null && _l !== void 0 ? _l : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package from RubyGems.
     *
     * @stability stable
     */
    static rubyGem(gemName, options = {}) {
        var _l;
        return new InitPackage('rubygems', (_l = options.version) !== null && _l !== void 0 ? _l : [], gemName, options.serviceRestartHandles);
    }
    /**
     * Install a package from PyPI.
     *
     * @stability stable
     */
    static python(packageName, options = {}) {
        var _l;
        return new InitPackage('python', (_l = options.version) !== null && _l !== void 0 ? _l : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package using APT.
     *
     * @stability stable
     */
    static apt(packageName, options = {}) {
        var _l;
        return new InitPackage('apt', (_l = options.version) !== null && _l !== void 0 ? _l : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install an MSI package from an HTTP URL or a location on disk.
     *
     * @stability stable
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _l;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_l = this.serviceHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    /**
     * @stability stable
     */
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
_h = JSII_RTTI_SYMBOL_1;
InitPackage[_h] = { fqn: "@aws-cdk/aws-ec2.InitPackage", version: "1.107.0" };
/**
 * A services that be enabled, disabled or restarted when the instance is launched.
 *
 * @stability stable
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * Enable and start the given service, optionally restarting it.
     *
     * @stability stable
     */
    static enable(serviceName, options = {}) {
        var _l;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_l = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _l !== void 0 ? _l : true,
            ...otherOptions,
        });
    }
    /**
     * Disable and stop the given service.
     *
     * @stability stable
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _l;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_l = this.serviceOptions.serviceRestartHandle) === null || _l === void 0 ? void 0 : _l._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
_j = JSII_RTTI_SYMBOL_1;
InitService[_j] = { fqn: "@aws-cdk/aws-ec2.InitService", version: "1.107.0" };
/**
 * Extract an archive into a directory.
 *
 * @stability stable
 */
class InitSource extends InitElement {
    /**
     * @stability stable
     */
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * Retrieve a URL and extract it into the given directory.
     *
     * @stability stable
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a GitHub branch into a given directory.
     *
     * @stability stable
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * Extract an archive stored in an S3 bucket into the given directory.
     *
     * @stability stable
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Create an InitSource from an asset created from the given path.
     *
     * @stability stable
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a directory from an existing directory asset.
     *
     * @stability stable
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _l;
        for (const handle of (_l = this.serviceHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
_k = JSII_RTTI_SYMBOL_1;
InitSource[_k] = { fqn: "@aws-cdk/aws-ec2.InitSource", version: "1.107.0" };
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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