"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mode = exports.StaticSiteAuthorization = exports.SpaAuthorization = exports.Authorization = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const core_1 = require("@aws-cdk/core");
const cdk_lambda_at_edge_pattern_1 = require("@cloudcomponents/cdk-lambda-at-edge-pattern");
const auth_flow_1 = require("./auth-flow");
const retrieve_user_pool_client_secret_1 = require("./retrieve-user-pool-client-secret");
const secret_generator_1 = require("./secret-generator");
const user_pool_client_redirects_1 = require("./user-pool-client-redirects");
const user_pool_domain_1 = require("./user-pool-domain");
class Authorization extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.userPool = props.userPool;
        this.redirectPaths = (_a = props.redirectPaths) !== null && _a !== void 0 ? _a : {
            signIn: '/parseauth',
            authRefresh: '/refreshauth',
            signOut: '/',
        };
        this.signOutUrlPath = (_b = props.signOutUrl) !== null && _b !== void 0 ? _b : '/signout';
        this.httpHeaders = (_c = props.httpHeaders) !== null && _c !== void 0 ? _c : {
            'Content-Security-Policy': "default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self' 'unsafe-inline'; object-src 'none'; connect-src 'self'",
            'Strict-Transport-Security': 'max-age=31536000; includeSubdomains; preload',
            'Referrer-Policy': 'same-origin',
            'X-XSS-Protection': '1; mode=block',
            'X-Frame-Options': 'DENY',
            'X-Content-Type-Options': 'nosniff',
            'Cache-Control': 'no-cache',
        };
        this.oauthScopes = (_d = props.oauthScopes) !== null && _d !== void 0 ? _d : [aws_cognito_1.OAuthScope.PHONE, aws_cognito_1.OAuthScope.EMAIL, aws_cognito_1.OAuthScope.PROFILE, aws_cognito_1.OAuthScope.OPENID, aws_cognito_1.OAuthScope.COGNITO_ADMIN];
        this.cookieSettings = props.cookieSettings;
        this.identityProviders = (_e = props.identityProviders) !== null && _e !== void 0 ? _e : [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO];
        this.userPoolClient = this.createUserPoolClient();
        this.nonceSigningSecret = this.generateNonceSigningSecret();
        this.cognitoAuthDomain = this.retrieveCognitoAuthDomain();
        this.authFlow = this.createAuthFlow((_f = props.logLevel) !== null && _f !== void 0 ? _f : cdk_lambda_at_edge_pattern_1.LogLevel.WARN);
    }
    updateUserPoolClientCallbacks(redirects) {
        const { callbackUrls, logoutUrls } = redirects;
        new user_pool_client_redirects_1.UserPoolClientRedirects(this, 'UserPoolClientRedirects', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            callbackUrls,
            logoutUrls,
            identityProviders: this.identityProviders,
        });
    }
    createDefaultBehavior(origin, options) {
        return {
            origin,
            compress: true,
            originRequestPolicy: aws_cloudfront_1.OriginRequestPolicy.ALL_VIEWER,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            edgeLambdas: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
            ...options,
        };
    }
    createLegacyDefaultBehavior(options) {
        return {
            isDefaultBehavior: true,
            forwardedValues: {
                queryString: true,
            },
            compress: true,
            lambdaFunctionAssociations: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
            ...options,
        };
    }
    createAdditionalBehaviors(origin, options) {
        return {
            [this.redirectPaths.signIn]: {
                origin,
                compress: true,
                originRequestPolicy: aws_cloudfront_1.OriginRequestPolicy.ALL_VIEWER,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.parseAuth],
                ...options,
            },
            [this.redirectPaths.authRefresh]: {
                origin,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.refreshAuth],
                ...options,
            },
            [this.signOutUrlPath]: {
                origin,
                compress: true,
                originRequestPolicy: aws_cloudfront_1.OriginRequestPolicy.ALL_VIEWER,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.signOut],
                ...options,
            },
        };
    }
    createLegacyAdditionalBehaviors(options) {
        return [
            {
                pathPattern: this.redirectPaths.signIn,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.parseAuth],
                ...options,
            },
            {
                pathPattern: this.redirectPaths.authRefresh,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.refreshAuth],
                ...options,
            },
            {
                pathPattern: this.signOutUrlPath,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.signOut],
                ...options,
            },
        ];
    }
    generateNonceSigningSecret() {
        const { secret } = new secret_generator_1.SecretGenerator(this, 'SecretGenerator');
        return secret;
    }
    retrieveCognitoAuthDomain() {
        const userPoolDomain = new user_pool_domain_1.UserPoolDomain(this, 'UserPoolDomain', {
            userPool: this.userPool,
        });
        return userPoolDomain.cognitoAuthDomain;
    }
}
exports.Authorization = Authorization;
class SpaAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.SPA;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: false,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: this.identityProviders,
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; SameSite=Lax',
                accessToken: 'Path=/; Secure; SameSite=Lax',
                refreshToken: 'Path=/; Secure; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
}
exports.SpaAuthorization = SpaAuthorization;
class StaticSiteAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.STATIC_SITE;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: true,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: this.identityProviders,
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        const clientSecret = this.retrieveUserPoolClientSecret();
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            clientSecret,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                accessToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                refreshToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
    retrieveUserPoolClientSecret() {
        const { clientSecret } = new retrieve_user_pool_client_secret_1.RetrieveUserPoolClientSecret(this, 'RetrieveUserPoolClientSecret', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
        });
        return clientSecret;
    }
}
exports.StaticSiteAuthorization = StaticSiteAuthorization;
var Mode;
(function (Mode) {
    Mode["SPA"] = "SPA";
    Mode["STATIC_SITE"] = "STATIC_SITE";
})(Mode = exports.Mode || (exports.Mode = {}));
//# sourceMappingURL=data:application/json;base64,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