"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
            ...props,
            restartExecutionOnUpdate: true,
            stages: [
                {
                    stageName: 'Source',
                    actions: [props.sourceAction],
                },
                {
                    stageName: 'Build',
                    actions: [props.synthAction],
                },
                {
                    stageName: 'UpdatePipeline',
                    actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                            cloudAssemblyInput: this._cloudAssemblyArtifact,
                            pipelineStackName: pipelineStack.stackName,
                            cdkCliVersion: props.cdkCliVersion,
                            projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        })],
                },
            ],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    onPrepare() {
        super.onPrepare();
        // TODO: Support this in a proper way in the upstream library. For now, we
        // "un-add" the Assets stage if it turns out to be empty.
        this._assets.removeAssetsStageIfEmpty();
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    this.node.addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
        // roles into one, and re-using across all assets, saves significant size of the final synthesized output.
        // Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
        // Late-binding here to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRole) {
            this.assetRole = new iam.Role(this, 'Role', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
            });
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRole,
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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