"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        switch (artifact.type) {
            case cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK:
                return new cloudformation_artifact_1.CloudFormationStackArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.CDK_TREE:
                return new tree_cloud_artifact_1.TreeCloudArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.ASSET_MANIFEST:
                return new asset_manifest_artifact_1.AssetManifestArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY:
                return new nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact(assembly, id, artifact);
            default:
                return undefined;
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
}
exports.CloudArtifact = CloudArtifact;
// needs to be defined at the end to avoid a cyclic dependency
const asset_manifest_artifact_1 = require("./artifacts/asset-manifest-artifact");
const cloudformation_artifact_1 = require("./artifacts/cloudformation-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
//# sourceMappingURL=data:application/json;base64,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