"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RememberingTokenResolver = exports.findTokens = exports.resolve = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x) { return resolve(x, { ...options, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return options.resolver.resolveString(fragments, makeContext()[0]);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) !== 'string') {
            // eslint-disable-next-line max-len
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}. Consider using "CfnJson" to delay resolution to deployment-time`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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