import * as events from "../../aws-events";
import * as iam from "../../aws-iam";
import * as kms from "../../aws-kms";
import { Construct, IResource, RemovalPolicy, Resource } from "../../core";
import { BucketPolicy } from './bucket-policy';
import { IBucketNotificationDestination } from './destination';
import { LifecycleRule } from './rule';
export interface IBucket extends IResource {
    /**
     * The ARN of the bucket.
     * @attribute
     */
    readonly bucketArn: string;
    /**
     * The name of the bucket.
     * @attribute
     */
    readonly bucketName: string;
    /**
     * The URL of the static website.
     * @attribute
     */
    readonly bucketWebsiteUrl: string;
    /**
     * The Domain name of the static website.
     * @attribute
     */
    readonly bucketWebsiteDomainName: string;
    /**
     * The IPv4 DNS name of the specified bucket.
     * @attribute
     */
    readonly bucketDomainName: string;
    /**
     * The IPv6 DNS name of the specified bucket.
     * @attribute
     */
    readonly bucketDualStackDomainName: string;
    /**
     * The regional domain name of the specified bucket.
     * @attribute
     */
    readonly bucketRegionalDomainName: string;
    /**
     * If this bucket has been configured for static website hosting.
     */
    readonly isWebsite?: boolean;
    /**
     * Optional KMS encryption key associated with this bucket.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     */
    policy?: BucketPolicy;
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or it's
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     */
    addToResourcePolicy(permission: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * The https URL of an S3 object. For example:
     * @example https://s3.us-west-1.amazonaws.com/onlybucket
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     * @example https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key?: string): string;
    /**
     * The S3 URL of an S3 object. For example:
     * @example s3://onlybucket
     * @example s3://bucket/key
     * @param key The S3 key of the object. If not specified, the S3 URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    s3UrlForObject(key?: string): string;
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     */
    arnForObjects(keyPattern: string): string;
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantReadWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     * @returns The `iam.PolicyStatement` object, which can be used to apply e.g. conditions.
     */
    grantPublicAccess(keyPrefix?: string, ...allowedActions: string[]): iam.Grant;
    /**
     * Defines a CloudWatch event that triggers when something happens to this bucket
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded
     * to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the
     * specified paths (keys) in this bucket are written to.  This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailWriteObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
}
/**
 * A reference to a bucket. The easiest way to instantiate is to call
 * `bucket.export()`. Then, the consumer can use `Bucket.import(this, ref)` and
 * get a `Bucket`.
 */
export interface BucketAttributes {
    /**
     * The ARN of the bucket. At least one of bucketArn or bucketName must be
     * defined in order to initialize a bucket ref.
     */
    readonly bucketArn?: string;
    /**
     * The name of the bucket. If the underlying value of ARN is a string, the
     * name will be parsed from the ARN. Otherwise, the name is optional, but
     * some features that require the bucket name such as auto-creating a bucket
     * policy, won't work.
     */
    readonly bucketName?: string;
    /**
     * The domain name of the bucket.
     *
     * @default Inferred from bucket name
     */
    readonly bucketDomainName?: string;
    /**
     * The website URL of the bucket (if static web hosting is enabled).
     *
     * @default Inferred from bucket name
     */
    readonly bucketWebsiteUrl?: string;
    /**
     * The regional domain name of the specified bucket.
     */
    readonly bucketRegionalDomainName?: string;
    /**
     * The IPv6 DNS name of the specified bucket.
     */
    readonly bucketDualStackDomainName?: string;
    /**
     * The format of the website URL of the bucket. This should be true for
     * regions launched since 2014.
     *
     * @default false
     */
    readonly bucketWebsiteNewUrlFormat?: boolean;
    readonly encryptionKey?: kms.IKey;
    /**
     * If this bucket has been configured for static website hosting.
     *
     * @default false
     */
    readonly isWebsite?: boolean;
}
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
declare abstract class BucketBase extends Resource implements IBucket {
    abstract readonly bucketArn: string;
    abstract readonly bucketName: string;
    abstract readonly bucketDomainName: string;
    abstract readonly bucketWebsiteUrl: string;
    abstract readonly bucketWebsiteDomainName: string;
    abstract readonly bucketRegionalDomainName: string;
    abstract readonly bucketDualStackDomainName: string;
    /**
     * Optional KMS encryption key associated with this bucket.
     */
    abstract readonly encryptionKey?: kms.IKey;
    /**
     * If this bucket has been configured for static website hosting.
     */
    abstract readonly isWebsite?: boolean;
    /**
     * The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     */
    abstract policy?: BucketPolicy;
    /**
     * Indicates if a bucket resource policy should automatically created upon
     * the first call to `addToResourcePolicy`.
     */
    protected abstract autoCreatePolicy: boolean;
    /**
     * Whether to disallow public access
     */
    protected abstract disallowPublicAccess?: boolean;
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded
     * to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the
     * specified paths (keys) in this bucket are written to.  This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailWriteObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or it's
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     */
    addToResourcePolicy(permission: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * The https URL of an S3 object. For example:
     * @example https://s3.us-west-1.amazonaws.com/onlybucket
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     * @example https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key?: string): string;
    /**
     * The S3 URL of an S3 object. For example:
     * @example s3://onlybucket
     * @example s3://bucket/key
     * @param key The S3 key of the object. If not specified, the S3 URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    s3UrlForObject(key?: string): string;
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     *
     * If you specify multiple components for keyPattern, they will be concatenated::
     *
     *   arnForObjects('home/', team, '/', user, '/*')
     *
     */
    arnForObjects(keyPattern: string): string;
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantReadWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     */
    grantPublicAccess(keyPrefix?: string, ...allowedActions: string[]): iam.Grant;
    private buildUrl;
    private grant;
    private isGranteeFromAnotherAccount;
}
export interface BlockPublicAccessOptions {
    /**
     * Whether to block public ACLs
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     */
    readonly blockPublicAcls?: boolean;
    /**
     * Whether to block public policy
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     */
    readonly blockPublicPolicy?: boolean;
    /**
     * Whether to ignore public ACLs
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     */
    readonly ignorePublicAcls?: boolean;
    /**
     * Whether to restrict public access
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     */
    readonly restrictPublicBuckets?: boolean;
}
export declare class BlockPublicAccess {
    static readonly BLOCK_ALL: BlockPublicAccess;
    static readonly BLOCK_ACLS: BlockPublicAccess;
    blockPublicAcls: boolean | undefined;
    blockPublicPolicy: boolean | undefined;
    ignorePublicAcls: boolean | undefined;
    restrictPublicBuckets: boolean | undefined;
    constructor(options: BlockPublicAccessOptions);
}
/**
 * Specifies a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket.
 */
export interface BucketMetrics {
    /**
     * The ID used to identify the metrics configuration.
     */
    readonly id: string;
    /**
     * The prefix that an object must have to be included in the metrics results.
     */
    readonly prefix?: string;
    /**
     * Specifies a list of tag filters to use as a metrics configuration filter.
     * The metrics configuration includes only objects that meet the filter's criteria.
     */
    readonly tagFilters?: {
        [tag: string]: any;
    };
}
/**
 * All http request methods
 */
export declare enum HttpMethods {
    /**
     * The GET method requests a representation of the specified resource.
     */
    GET = "GET",
    /**
     * The PUT method replaces all current representations of the target resource with the request payload.
     */
    PUT = "PUT",
    /**
     * The HEAD method asks for a response identical to that of a GET request, but without the response body.
     */
    HEAD = "HEAD",
    /**
     * The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     */
    POST = "POST",
    /**
     * The DELETE method deletes the specified resource.
     */
    DELETE = "DELETE"
}
/**
 * Specifies a cross-origin access rule for an Amazon S3 bucket.
 */
export interface CorsRule {
    /**
     * A unique identifier for this rule.
     *
     * @default - No id specified.
     */
    readonly id?: string;
    /**
     * The time in seconds that your browser is to cache the preflight response for the specified resource.
     *
     * @default - No caching.
     */
    readonly maxAge?: number;
    /**
     * Headers that are specified in the Access-Control-Request-Headers header.
     *
     * @default - No headers allowed.
     */
    readonly allowedHeaders?: string[];
    /**
     * An HTTP method that you allow the origin to execute.
     */
    readonly allowedMethods: HttpMethods[];
    /**
     * One or more origins you want customers to be able to access the bucket from.
     */
    readonly allowedOrigins: string[];
    /**
     * One or more headers in the response that you want customers to be able to access from their applications.
     *
     * @default - No headers exposed.
     */
    readonly exposedHeaders?: string[];
}
/**
 * All http request methods
 */
export declare enum RedirectProtocol {
    HTTP = "http",
    HTTPS = "https"
}
/**
 * Specifies a redirect behavior of all requests to a website endpoint of a bucket.
 */
export interface RedirectTarget {
    /**
     * Name of the host where requests are redirected
     */
    readonly hostName: string;
    /**
     * Protocol to use when redirecting requests
     *
     * @default - The protocol used in the original request.
     */
    readonly protocol?: RedirectProtocol;
}
export interface BucketProps {
    /**
     * The kind of server-side encryption to apply to this bucket.
     *
     * If you choose KMS, you can specify a KMS key via `encryptionKey`. If
     * encryption key is not specified, a key will automatically be created.
     *
     * @default - `Kms` if `encryptionKey` is specified, or `Unencrypted` otherwise.
     */
    readonly encryption?: BucketEncryption;
    /**
     * External KMS key to use for bucket encryption.
     *
     * The 'encryption' property must be either not specified or set to "Kms".
     * An error will be emitted if encryption is set to "Unencrypted" or
     * "Managed".
     *
     * @default - If encryption is set to "Kms" and this property is undefined,
     * a new KMS key will be created and associated with this bucket.
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Physical name of this bucket.
     *
     * @default - Assigned by CloudFormation (recommended).
     */
    readonly bucketName?: string;
    /**
     * Policy to apply when the bucket is removed from this stack.
     *
     * @default - The bucket will be orphaned.
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Whether this bucket should have versioning turned on or not.
     *
     * @default false
     */
    readonly versioned?: boolean;
    /**
     * Rules that define how Amazon S3 manages objects during their lifetime.
     *
     * @default - No lifecycle rules.
     */
    readonly lifecycleRules?: LifecycleRule[];
    /**
     * The name of the index document (e.g. "index.html") for the website. Enables static website
     * hosting for this bucket.
     *
     * @default - No index document.
     */
    readonly websiteIndexDocument?: string;
    /**
     * The name of the error document (e.g. "404.html") for the website.
     * `websiteIndexDocument` must also be set if this is set.
     *
     * @default - No error document.
     */
    readonly websiteErrorDocument?: string;
    /**
     * Specifies the redirect behavior of all requests to a website endpoint of a bucket.
     *
     * If you specify this property, you can't specify "websiteIndexDocument", "websiteErrorDocument" nor , "websiteRoutingRules".
     *
     * @default - No redirection.
     */
    readonly websiteRedirect?: RedirectTarget;
    /**
     * Rules that define when a redirect is applied and the redirect behavior
     *
     * @default - No redirection rules.
     */
    readonly websiteRoutingRules?: RoutingRule[];
    /**
     * Specifies a canned ACL that grants predefined permissions to the bucket.
     *
     * @default BucketAccessControl.PRIVATE
     */
    readonly accessControl?: BucketAccessControl;
    /**
     * Grants public read access to all objects in the bucket.
     * Similar to calling `bucket.grantPublicAccess()`
     *
     * @default false
     */
    readonly publicReadAccess?: boolean;
    /**
     * The block public access configuration of this bucket.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html
     *
     * @default false New buckets and objects don't allow public access, but users can modify bucket
     * policies or object permissions to allow public access.
     */
    readonly blockPublicAccess?: BlockPublicAccess;
    /**
     * The metrics configuration of this bucket.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-metricsconfiguration.html
     *
     * @default - No metrics configuration.
     */
    readonly metrics?: BucketMetrics[];
    /**
     * The CORS configuration of this bucket.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-cors.html
     *
     * @default - No CORS configuration.
     */
    readonly cors?: CorsRule[];
    /**
     * Destination bucket for the server access logs.
     * @default - If "serverAccessLogsPrefix" undefined - access logs disabled, otherwise - log to current bucket.
     */
    readonly serverAccessLogsBucket?: IBucket;
    /**
     * Optional log file prefix to use for the bucket's access logs.
     * If defined without "serverAccessLogsBucket", enables access logs to current bucket with this prefix.
     * @default - No log file prefix
     */
    readonly serverAccessLogsPrefix?: string;
}
/**
 * An S3 bucket with associated policy objects
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 */
export declare class Bucket extends BucketBase {
    static fromBucketArn(scope: Construct, id: string, bucketArn: string): IBucket;
    static fromBucketName(scope: Construct, id: string, bucketName: string): IBucket;
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object. Can be obtained from a call to
     * `bucket.export()` or manually created.
     */
    static fromBucketAttributes(scope: Construct, id: string, attrs: BucketAttributes): IBucket;
    readonly bucketArn: string;
    readonly bucketName: string;
    readonly bucketDomainName: string;
    readonly bucketWebsiteUrl: string;
    readonly bucketWebsiteDomainName: string;
    readonly bucketDualStackDomainName: string;
    readonly bucketRegionalDomainName: string;
    readonly encryptionKey?: kms.IKey;
    readonly isWebsite?: boolean;
    policy?: BucketPolicy;
    protected autoCreatePolicy: boolean;
    protected disallowPublicAccess?: boolean;
    private accessControl?;
    private readonly lifecycleRules;
    private readonly versioned?;
    private readonly notifications;
    private readonly metrics;
    private readonly cors;
    constructor(scope: Construct, id: string, props?: BucketProps);
    /**
     * Add a lifecycle rule to the bucket
     *
     * @param rule The rule to add
     */
    addLifecycleRule(rule: LifecycleRule): void;
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add
     */
    addMetric(metric: BucketMetrics): void;
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket
     *
     * @param rule The CORS configuration rule to add
     */
    addCorsRule(rule: CorsRule): void;
    /**
     * Adds a bucket notification event destination.
     * @param event The event to trigger the notification
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue)
     *
     * @param filters S3 object key filter rules to determine which objects
     * trigger this event. Each filter must include a `prefix` and/or `suffix`
     * that will be matched against the s3 object key. Refer to the S3 Developer Guide
     * for details about allowed filter rules.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering
     *
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     */
    addEventNotification(event: EventType, dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * Subscribes a destination to receive notificatins when an object is
     * created in the bucket. This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectCreatedNotification(dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * Subscribes a destination to receive notificatins when an object is
     * removed from the bucket. This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectRemovedNotification(dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    private validateBucketName;
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    private parseEncryption;
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    private parseLifecycleConfiguration;
    private parseServerAccessLogs;
    private parseMetricConfiguration;
    private parseCorsConfiguration;
    private parseTagFilters;
    private renderWebsiteConfiguration;
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    private allowLogDelivery;
}
/**
 * What kind of server-side encryption to apply to this bucket
 */
export declare enum BucketEncryption {
    /**
     * Objects in the bucket are not encrypted.
     */
    UNENCRYPTED = "NONE",
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     */
    KMS_MANAGED = "MANAGED",
    /**
     * Server-side encryption with a master key managed by S3.
     */
    S3_MANAGED = "S3MANAGED",
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    KMS = "KMS"
}
/**
 * Notification event types.
 */
export declare enum EventType {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    OBJECT_CREATED = "s3:ObjectCreated:*",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    OBJECT_CREATED_PUT = "s3:ObjectCreated:Put",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    OBJECT_CREATED_POST = "s3:ObjectCreated:Post",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    OBJECT_CREATED_COPY = "s3:ObjectCreated:Copy",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD = "s3:ObjectCreated:CompleteMultipartUpload",
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    OBJECT_REMOVED = "s3:ObjectRemoved:*",
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    OBJECT_REMOVED_DELETE = "s3:ObjectRemoved:Delete",
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    OBJECT_REMOVED_DELETE_MARKER_CREATED = "s3:ObjectRemoved:DeleteMarkerCreated",
    /**
     * You can use this event type to request Amazon S3 to send a notification
     * message when Amazon S3 detects that an object of the RRS storage class is
     * lost.
     */
    REDUCED_REDUNDANCY_LOST_OBJECT = "s3:ReducedRedundancyLostObject"
}
export interface NotificationKeyFilter {
    /**
     * S3 keys must have the specified prefix.
     */
    readonly prefix?: string;
    /**
     * S3 keys must have the specified suffix.
     */
    readonly suffix?: string;
}
/**
 * Options for the onCloudTrailPutObject method
 */
export interface OnCloudTrailBucketEventOptions extends events.OnEventOptions {
    /**
     * Only watch changes to these object paths
     *
     * @default - Watch changes to all objects
     */
    readonly paths?: string[];
}
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 */
export declare enum BucketAccessControl {
    /**
     * Owner gets FULL_CONTROL. No one else has access rights.
     */
    PRIVATE = "Private",
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ access.
     */
    PUBLIC_READ = "PublicRead",
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     */
    PUBLIC_READ_WRITE = "PublicReadWrite",
    /**
     * Owner gets FULL_CONTROL. The AuthenticatedUsers group gets READ access.
     */
    AUTHENTICATED_READ = "AuthenticatedRead",
    /**
     * The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     */
    LOG_DELIVERY_WRITE = "LogDeliveryWrite",
    /**
     * Object owner gets FULL_CONTROL. Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BUCKET_OWNER_READ = "BucketOwnerRead",
    /**
     * Both the object owner and the bucket owner get FULL_CONTROL over the object.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BUCKET_OWNER_FULL_CONTROL = "BucketOwnerFullControl",
    /**
     * Owner gets FULL_CONTROL. Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     */
    AWS_EXEC_READ = "AwsExecRead"
}
export interface RoutingRuleCondition {
    /**
     * The HTTP error code when the redirect is applied
     *
     * In the event of an error, if the error code equals this value, then the specified redirect is applied.
     *
     * If both condition properties are specified, both must be true for the redirect to be applied.
     *
     * @default - The HTTP error code will not be verified
     */
    readonly httpErrorCodeReturnedEquals?: string;
    /**
     * The object key name prefix when the redirect is applied
     *
     * If both condition properties are specified, both must be true for the redirect to be applied.
     *
     * @default - The object key name will not be verified
     */
    readonly keyPrefixEquals?: string;
}
export declare class ReplaceKey {
    readonly withKey?: string | undefined;
    readonly prefixWithKey?: string | undefined;
    /**
     * The specific object key to use in the redirect request
     */
    static with(keyReplacement: string): ReplaceKey;
    /**
     * The object key prefix to use in the redirect request
     */
    static prefixWith(keyReplacement: string): ReplaceKey;
    private constructor();
}
/**
 * Rule that define when a redirect is applied and the redirect behavior.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html
 */
export interface RoutingRule {
    /**
     * The host name to use in the redirect request
     *
     * @default - The host name used in the original request.
     */
    readonly hostName?: string;
    /**
     * The HTTP redirect code to use on the response
     *
     * @default "301" - Moved Permanently
     */
    readonly httpRedirectCode?: string;
    /**
     * Protocol to use when redirecting requests
     *
     * @default - The protocol used in the original request.
     */
    readonly protocol?: RedirectProtocol;
    /**
     * Specifies the object key prefix to use in the redirect request
     *
     * @default - The key will not be replaced
     */
    readonly replaceKey?: ReplaceKey;
    /**
     * Specifies a condition that must be met for the specified redirect to apply.
     *
     * @default - No condition
     */
    readonly condition?: RoutingRuleCondition;
}
export {};
