"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointGroup = exports.EndpointConfiguration = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ga = require("./globalaccelerator.generated");
/**
 * The class for endpoint configuration
 */
class EndpointConfiguration extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        props.endpointGroup._linkEndpoint(this);
    }
    /**
     * render the endpoint configuration for the endpoint group
     */
    renderEndpointConfiguration() {
        return {
            clientIpPreservationEnabled: this.props.clientIpReservation,
            endpointId: this.props.endpointId,
            weight: this.props.weight,
        };
    }
}
exports.EndpointConfiguration = EndpointConfiguration;
/**
 * EndpointGroup construct
 */
class EndpointGroup extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The array of the endpoints in this endpoint group
         */
        this.endpoints = new Array();
        const resource = new ga.CfnEndpointGroup(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            endpointGroupRegion: (_a = props.region) !== null && _a !== void 0 ? _a : cdk.Stack.of(this).region,
            endpointConfigurations: cdk.Lazy.anyValue({ produce: () => this.renderEndpoints() }, { omitEmptyArray: true }),
        });
        this.endpointGroupArn = resource.attrEndpointGroupArn;
        this.endpointGroupName = (_b = props.endpointGroupName) !== null && _b !== void 0 ? _b : resource.logicalId;
    }
    /**
     * import from ARN
     */
    static fromEndpointGroupArn(scope, id, endpointGroupArn) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointGroupArn = endpointGroupArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add an endpoint
     */
    addEndpoint(id, endpointId, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointGroup: this,
            endpointId,
            ...props,
        });
    }
    /**
     * Add an Elastic Load Balancer as an endpoint in this endpoint group
     */
    addLoadBalancer(id, lb, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: lb.loadBalancerArn,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Add an EIP as an endpoint in this endpoint group
     */
    addElasticIpAddress(id, eip, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: eip.attrAllocationId,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Add an EC2 Instance as an endpoint in this endpoint group
     */
    addEc2Instance(id, instance, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: instance.instanceId,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Links a endpoint to this endpoint group
     * @internal
     */
    _linkEndpoint(endpoint) {
        this.endpoints.push(endpoint);
    }
    renderEndpoints() {
        return this.endpoints.map(e => e.renderEndpointConfiguration());
    }
}
exports.EndpointGroup = EndpointGroup;
//# sourceMappingURL=data:application/json;base64,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