"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_7_DAYS"] = 0] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_14_DAYS"] = 1] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_30_DAYS"] = 2] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_60_DAYS"] = 3] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_90_DAYS"] = 4] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * This is the general purpose performance mode for most file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * This performance mode can scale to higher levels of aggregate throughput and operations per second with a
     * tradeoff of slightly higher latencies.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     *  This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? Array.of({
                transitionToIa: LifecyclePolicy[props.lifecyclePolicy],
            }) : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tag.add(this, 'Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
                this.mountTargetsAvailable = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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