"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InstanceDrainHook = void 0;
const autoscaling = require("../../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const hooks = require("../../../aws-autoscaling-hooktargets"); // Automatically re-written from '@aws-cdk/aws-autoscaling-hooktargets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
/**
 * A hook to drain instances from ECS traffic before they're terminated
 */
class InstanceDrainHook extends cdk.Construct {
    /**
     * Constructs a new instance of the InstanceDrainHook class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const drainTime = props.drainTime || cdk.Duration.minutes(5);
        // Invoke Lambda via SNS Topic
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline(fs.readFileSync(path.join(__dirname, 'lambda-source', 'index.py'), { encoding: 'utf-8' })),
            handler: 'index.lambda_handler',
            runtime: lambda.Runtime.PYTHON_3_6,
            // Timeout: some extra margin for additional API calls made by the Lambda,
            // up to a maximum of 15 minutes.
            timeout: cdk.Duration.seconds(Math.min(drainTime.toSeconds() + 10, 900)),
            environment: {
                CLUSTER: props.cluster.clusterName,
            },
        });
        // Hook everything up: ASG -> Topic, Topic -> Lambda
        props.autoScalingGroup.addLifecycleHook('DrainHook', {
            lifecycleTransition: autoscaling.LifecycleTransition.INSTANCE_TERMINATING,
            defaultResult: autoscaling.DefaultResult.CONTINUE,
            notificationTarget: new hooks.FunctionHook(fn),
            heartbeatTimeout: drainTime,
        });
        // Describe actions cannot be restricted and restrict the CompleteLifecycleAction to the ASG arn
        // https://docs.aws.amazon.com/autoscaling/ec2/userguide/control-access-using-iam.html
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceStatus',
                'ec2:DescribeHosts',
            ],
            resources: ['*'],
        }));
        // Restrict to the ASG
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['autoscaling:CompleteLifecycleAction'],
            resources: [props.autoScalingGroup.autoScalingGroupArn],
        }));
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['ecs:DescribeContainerInstances', 'ecs:DescribeTasks'],
            resources: ['*'],
        }));
        // Restrict to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:ListContainerInstances',
                'ecs:SubmitContainerStateChange',
                'ecs:SubmitTaskStateChange',
            ],
            resources: [props.cluster.clusterArn],
        }));
        // Restrict the container-instance operations to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:UpdateContainerInstancesState',
                'ecs:ListTasks',
            ],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
            resources: ['*'],
        }));
    }
}
exports.InstanceDrainHook = InstanceDrainHook;
//# sourceMappingURL=data:application/json;base64,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