"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3DeployAction = exports.CacheControl = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /** The 'must-revalidate' cache control directive. */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /** The 'no-cache' cache control directive. */
    static noCache() { return new CacheControl('no-cache'); }
    /** The 'no-transform' cache control directive. */
    static noTransform() { return new CacheControl('no-transform'); }
    /** The 'public' cache control directive. */
    static setPublic() { return new CacheControl('public'); }
    /** The 'private' cache control directive. */
    static setPrivate() { return new CacheControl('private'); }
    /** The 'proxy-revalidate' cache control directive. */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /** The 'max-age' cache control directive. */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /** The 's-max-age' cache control directive. */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive,
     * in case our support is missing a method for a particular directive.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 */
class S3DeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95LWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGxveS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMERBQTBELENBQUMsNERBQTREO0FBR3ZILCtCQUE0QztBQUM1QyxzQ0FBbUM7QUFDbkMsc0NBQWlEO0FBQ2pELDJEQUEyRDtBQUMzRCx5RUFBeUU7QUFDekUsOEZBQThGO0FBQzlGLDRDQUE0QztBQUM1Qzs7Ozs7O0dBTUc7QUFDSCxNQUFhLFlBQVk7SUFzQnJCLGtFQUFrRTtJQUNsRSxZQUEyQixLQUFhO1FBQWIsVUFBSyxHQUFMLEtBQUssQ0FBUTtJQUFJLENBQUM7SUF0QjdDLHFEQUFxRDtJQUM5QyxNQUFNLENBQUMsY0FBYyxLQUFLLE9BQU8sSUFBSSxZQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUUsOENBQThDO0lBQ3ZDLE1BQU0sQ0FBQyxPQUFPLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDaEUsa0RBQWtEO0lBQzNDLE1BQU0sQ0FBQyxXQUFXLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDeEUsNENBQTRDO0lBQ3JDLE1BQU0sQ0FBQyxTQUFTLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDaEUsNkNBQTZDO0lBQ3RDLE1BQU0sQ0FBQyxVQUFVLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbEUsc0RBQXNEO0lBQy9DLE1BQU0sQ0FBQyxlQUFlLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNoRiw2Q0FBNkM7SUFDdEMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFXLElBQUksT0FBTyxJQUFJLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzFGLCtDQUErQztJQUN4QyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQVcsSUFBSSxPQUFPLElBQUksWUFBWSxDQUFDLFlBQVksQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUY7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFTLElBQUksT0FBTyxJQUFJLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FHdEU7QUF4QkQsb0NBd0JDO0FBdUNEOztHQUVHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsZUFBTTtJQUV0QyxZQUFZLEtBQTBCO1FBQ2xDLEtBQUssQ0FBQztZQUNGLEdBQUcsS0FBSztZQUNSLFFBQVEsRUFBRSxLQUFLLENBQUMsTUFBTTtZQUN0QixRQUFRLEVBQUUsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNO1lBQzVDLFFBQVEsRUFBRSxJQUFJO1lBQ2QsY0FBYyxFQUFFLDZCQUFvQixFQUFFO1lBQ3RDLE1BQU0sRUFBRSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUM7U0FDeEIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDdkIsQ0FBQztJQUNTLEtBQUssQ0FBQyxNQUFpQixFQUFFLE1BQTJCLEVBQUUsT0FBdUM7UUFDbkcsdURBQXVEO1FBQ3ZELElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0MsZ0VBQWdFO1FBQ2hFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUNyQyxPQUFPO1lBQ0gsYUFBYSxFQUFFO2dCQUNYLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUN4QyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssS0FBSyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLE1BQU07Z0JBQ3hELFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7Z0JBQy9CLFNBQVMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLFlBQVcsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDeEQsWUFBWSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO2FBQ2xHO1NBQ0osQ0FBQztJQUNOLENBQUM7Q0FDSjtBQTdCRCx3Q0E2QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSBcIi4uLy4uLy4uL2F3cy1jb2RlcGlwZWxpbmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnXG5pbXBvcnQgKiBhcyBzMyBmcm9tIFwiLi4vLi4vLi4vYXdzLXMzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtczMnXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uIH0gZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBrZWJhYiBhcyB0b0tlYmFiQ2FzZSB9IGZyb20gJ2Nhc2UnO1xuaW1wb3J0IHsgQWN0aW9uIH0gZnJvbSAnLi4vYWN0aW9uJztcbmltcG9ydCB7IGRlcGxveUFydGlmYWN0Qm91bmRzIH0gZnJvbSAnLi4vY29tbW9uJztcbi8vIENsYXNzIGNvcGllZCB2ZXJiYXRpbSBmcm9tIHRoZSBhd3MtczMtZGVwbG95bWVudCBtb2R1bGUuXG4vLyBZZXMsIGl0IHN1Y2tzIHRoYXQgd2UgZGlkbid0IGFic3RyYWN0IHRoaXMgcHJvcGVybHkgaW4gYSBjb21tb24gY2xhc3MsXG4vLyBidXQgaGF2aW5nIDIgZGlmZmVyZW50IENhY2hlQ29udHJvbCBjbGFzc2VzIHRoYXQgYmVoYXZlIGRpZmZlcmVudGx5IHdvdWxkIGJlIHdvcnNlIEkgdGhpbmsuXG4vLyBTb21ldGhpbmcgdG8gZG8gd2hlbiBDREsgMi4wLjAgY29tZXMgb3V0LlxuLyoqXG4gKiBVc2VkIGZvciBIVFRQIGNhY2hlLWNvbnRyb2wgaGVhZGVyLCB3aGljaCBpbmZsdWVuY2VzIGRvd25zdHJlYW0gY2FjaGVzLlxuICogVXNlIHRoZSBwcm92aWRlZCBzdGF0aWMgZmFjdG9yeSBtZXRob2RzIHRvIGNvbnN0cnVjdCBpbnN0YW5jZXMgb2YgdGhpcyBjbGFzcy5cbiAqIFVzZWQgaW4gdGhlIHtAbGluayBTM0RlcGxveUFjdGlvblByb3BzLmNhY2hlQ29udHJvbH0gcHJvcGVydHkuXG4gKlxuICogQHNlZSBodHRwczovL3d3dy53My5vcmcvUHJvdG9jb2xzL3JmYzI2MTYvcmZjMjYxNi1zZWMxNC5odG1sI3NlYzE0LjlcbiAqL1xuZXhwb3J0IGNsYXNzIENhY2hlQ29udHJvbCB7XG4gICAgLyoqIFRoZSAnbXVzdC1yZXZhbGlkYXRlJyBjYWNoZSBjb250cm9sIGRpcmVjdGl2ZS4gKi9cbiAgICBwdWJsaWMgc3RhdGljIG11c3RSZXZhbGlkYXRlKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgnbXVzdC1yZXZhbGlkYXRlJyk7IH1cbiAgICAvKiogVGhlICduby1jYWNoZScgY2FjaGUgY29udHJvbCBkaXJlY3RpdmUuICovXG4gICAgcHVibGljIHN0YXRpYyBub0NhY2hlKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgnbm8tY2FjaGUnKTsgfVxuICAgIC8qKiBUaGUgJ25vLXRyYW5zZm9ybScgY2FjaGUgY29udHJvbCBkaXJlY3RpdmUuICovXG4gICAgcHVibGljIHN0YXRpYyBub1RyYW5zZm9ybSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ25vLXRyYW5zZm9ybScpOyB9XG4gICAgLyoqIFRoZSAncHVibGljJyBjYWNoZSBjb250cm9sIGRpcmVjdGl2ZS4gKi9cbiAgICBwdWJsaWMgc3RhdGljIHNldFB1YmxpYygpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ3B1YmxpYycpOyB9XG4gICAgLyoqIFRoZSAncHJpdmF0ZScgY2FjaGUgY29udHJvbCBkaXJlY3RpdmUuICovXG4gICAgcHVibGljIHN0YXRpYyBzZXRQcml2YXRlKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgncHJpdmF0ZScpOyB9XG4gICAgLyoqIFRoZSAncHJveHktcmV2YWxpZGF0ZScgY2FjaGUgY29udHJvbCBkaXJlY3RpdmUuICovXG4gICAgcHVibGljIHN0YXRpYyBwcm94eVJldmFsaWRhdGUoKSB7IHJldHVybiBuZXcgQ2FjaGVDb250cm9sKCdwcm94eS1yZXZhbGlkYXRlJyk7IH1cbiAgICAvKiogVGhlICdtYXgtYWdlJyBjYWNoZSBjb250cm9sIGRpcmVjdGl2ZS4gKi9cbiAgICBwdWJsaWMgc3RhdGljIG1heEFnZSh0OiBEdXJhdGlvbikgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbChgbWF4LWFnZT0ke3QudG9TZWNvbmRzKCl9YCk7IH1cbiAgICAvKiogVGhlICdzLW1heC1hZ2UnIGNhY2hlIGNvbnRyb2wgZGlyZWN0aXZlLiAqL1xuICAgIHB1YmxpYyBzdGF0aWMgc01heEFnZSh0OiBEdXJhdGlvbikgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbChgcy1tYXhhZ2U9JHt0LnRvU2Vjb25kcygpfWApOyB9XG4gICAgLyoqXG4gICAgICogQWxsb3dzIHlvdSB0byBjcmVhdGUgYW4gYXJiaXRyYXJ5IGNhY2hlIGNvbnRyb2wgZGlyZWN0aXZlLFxuICAgICAqIGluIGNhc2Ugb3VyIHN1cHBvcnQgaXMgbWlzc2luZyBhIG1ldGhvZCBmb3IgYSBwYXJ0aWN1bGFyIGRpcmVjdGl2ZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoczogc3RyaW5nKSB7IHJldHVybiBuZXcgQ2FjaGVDb250cm9sKHMpOyB9XG4gICAgLyoqIEBwYXJhbSB2YWx1ZSB0aGUgYWN0dWFsIHRleHQgdmFsdWUgb2YgdGhlIGNyZWF0ZWQgZGlyZWN0aXZlICovXG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgdmFsdWU6IHN0cmluZykgeyB9XG59XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIG9mIHRoZSB7QGxpbmsgUzNEZXBsb3lBY3Rpb24gUzMgZGVwbG95IEFjdGlvbn0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNEZXBsb3lBY3Rpb25Qcm9wcyBleHRlbmRzIGNvZGVwaXBlbGluZS5Db21tb25Bd3NBY3Rpb25Qcm9wcyB7XG4gICAgLyoqXG4gICAgICogU2hvdWxkIHRoZSBkZXBsb3kgYWN0aW9uIGV4dHJhY3QgdGhlIGFydGlmYWN0IGJlZm9yZSBkZXBsb3lpbmcgdG8gQW1hem9uIFMzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGV4dHJhY3Q/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBrZXkgb2YgdGhlIHRhcmdldCBvYmplY3QuIFRoaXMgaXMgcmVxdWlyZWQgaWYgZXh0cmFjdCBpcyBmYWxzZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBvYmplY3RLZXk/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGlucHV0IEFydGlmYWN0IHRvIGRlcGxveSB0byBBbWF6b24gUzMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5wdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICAvKipcbiAgICAgKiBUaGUgQW1hem9uIFMzIGJ1Y2tldCB0aGF0IGlzIHRoZSBkZXBsb3kgdGFyZ2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldDogczMuSUJ1Y2tldDtcbiAgICAvKipcbiAgICAgKiBUaGUgc3BlY2lmaWVkIGNhbm5lZCBBQ0wgdG8gb2JqZWN0cyBkZXBsb3llZCB0byBBbWF6b24gUzMuXG4gICAgICogVGhpcyBvdmVyd3JpdGVzIGFueSBleGlzdGluZyBBQ0wgdGhhdCB3YXMgYXBwbGllZCB0byB0aGUgb2JqZWN0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSB0aGUgb3JpZ2luYWwgb2JqZWN0IEFDTFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFjY2Vzc0NvbnRyb2w/OiBzMy5CdWNrZXRBY2Nlc3NDb250cm9sO1xuICAgIC8qKlxuICAgICAqIFRoZSBjYWNoaW5nIGJlaGF2aW9yIGZvciByZXF1ZXN0cy9yZXNwb25zZXMgZm9yIG9iamVjdHMgaW4gdGhlIGJ1Y2tldC5cbiAgICAgKiBUaGUgZmluYWwgY2FjaGUgY29udHJvbCBwcm9wZXJ0eSB3aWxsIGJlIHRoZSByZXN1bHQgb2Ygam9pbmluZyBhbGwgb2YgdGhlIHByb3ZpZGVkIGFycmF5IGVsZW1lbnRzIHdpdGggYSBjb21tYVxuICAgICAqIChwbHVzIGEgc3BhY2UgYWZ0ZXIgdGhlIGNvbW1hKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm9uZSwgZGVjaWRlZCBieSB0aGUgSFRUUCBjbGllbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBjYWNoZUNvbnRyb2w/OiBDYWNoZUNvbnRyb2xbXTtcbn1cbi8qKlxuICogRGVwbG95cyB0aGUgc291cmNlQXJ0aWZhY3QgdG8gQW1hem9uIFMzLlxuICovXG5leHBvcnQgY2xhc3MgUzNEZXBsb3lBY3Rpb24gZXh0ZW5kcyBBY3Rpb24ge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFMzRGVwbG95QWN0aW9uUHJvcHM7XG4gICAgY29uc3RydWN0b3IocHJvcHM6IFMzRGVwbG95QWN0aW9uUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoe1xuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICByZXNvdXJjZTogcHJvcHMuYnVja2V0LFxuICAgICAgICAgICAgY2F0ZWdvcnk6IGNvZGVwaXBlbGluZS5BY3Rpb25DYXRlZ29yeS5ERVBMT1ksXG4gICAgICAgICAgICBwcm92aWRlcjogJ1MzJyxcbiAgICAgICAgICAgIGFydGlmYWN0Qm91bmRzOiBkZXBsb3lBcnRpZmFjdEJvdW5kcygpLFxuICAgICAgICAgICAgaW5wdXRzOiBbcHJvcHMuaW5wdXRdLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgIH1cbiAgICBwcm90ZWN0ZWQgYm91bmQoX3Njb3BlOiBDb25zdHJ1Y3QsIF9zdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTogY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgICAgIC8vIHBpcGVsaW5lIG5lZWRzIHBlcm1pc3Npb25zIHRvIHdyaXRlIHRvIHRoZSBTMyBidWNrZXRcbiAgICAgICAgdGhpcy5wcm9wcy5idWNrZXQuZ3JhbnRXcml0ZShvcHRpb25zLnJvbGUpO1xuICAgICAgICAvLyB0aGUgQWN0aW9uIFJvbGUgYWxzbyBuZWVkcyB0byByZWFkIGZyb20gdGhlIFBpcGVsaW5lJ3MgYnVja2V0XG4gICAgICAgIG9wdGlvbnMuYnVja2V0LmdyYW50UmVhZChvcHRpb25zLnJvbGUpO1xuICAgICAgICBjb25zdCBhY2wgPSB0aGlzLnByb3BzLmFjY2Vzc0NvbnRyb2w7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBjb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgQnVja2V0TmFtZTogdGhpcy5wcm9wcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICAgICAgICBFeHRyYWN0OiB0aGlzLnByb3BzLmV4dHJhY3QgPT09IGZhbHNlID8gJ2ZhbHNlJyA6ICd0cnVlJyxcbiAgICAgICAgICAgICAgICBPYmplY3RLZXk6IHRoaXMucHJvcHMub2JqZWN0S2V5LFxuICAgICAgICAgICAgICAgIENhbm5lZEFDTDogYWNsID8gdG9LZWJhYkNhc2UoYWNsLnRvU3RyaW5nKCkpIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIENhY2hlQ29udHJvbDogdGhpcy5wcm9wcy5jYWNoZUNvbnRyb2wgJiYgdGhpcy5wcm9wcy5jYWNoZUNvbnRyb2wubWFwKGFjID0+IGFjLnZhbHVlKS5qb2luKCcsICcpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICB9XG59XG4iXX0=