"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpOrigin = exports.S3Origin = exports.Origin = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const distribution_1 = require("./distribution");
const origin_access_identity_1 = require("./origin_access_identity");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class Origin {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(scope, options) {
        this.originId = new core_1.Construct(scope, `Origin${options.originIndex}`).node.uniqueId;
    }
    /**
     * Creates and returns the CloudFormation representation of this origin.
     */
    renderOrigin() {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            domainName: this.domainName,
            id: this.id,
            originPath: this.originPath,
            connectionAttempts: this.connectionAttempts,
            connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            originCustomHeaders: this.renderCustomHeaders(),
            s3OriginConfig,
            customOriginConfig,
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.Origin = Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
class S3Origin extends Origin {
    constructor(props) {
        super(props.bucket.bucketRegionalDomainName, props);
        this.bucket = props.bucket;
    }
    bind(scope, options) {
        super.bind(scope, options);
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new origin_access_identity_1.OriginAccessIdentity(scope, `S3Origin${options.originIndex}`);
            this.bucket.grantRead(this.originAccessIdentity);
        }
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
class HttpOrigin extends Origin {
    constructor(domainName, props = {}) {
        var _a, _b;
        super(domainName, props);
        this.props = props;
        validateIntInRangeOrUndefined('readTimeout', 1, 60, (_a = props.readTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('keepaliveTimeout', 1, 60, (_b = props.keepaliveTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
    }
    renderCustomOriginConfig() {
        var _a, _b, _c;
        return {
            originProtocolPolicy: (_a = this.props.protocolPolicy) !== null && _a !== void 0 ? _a : distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: this.props.httpPort,
            httpsPort: this.props.httpsPort,
            originReadTimeout: (_b = this.props.readTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            originKeepaliveTimeout: (_c = this.props.keepaliveTimeout) === null || _c === void 0 ? void 0 : _c.toSeconds(),
        };
    }
}
exports.HttpOrigin = HttpOrigin;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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