"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.Distribution = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.additionalBehaviors = [];
        this.origins = new Set();
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
        }
        this.defaultBehavior = new cache_behavior_1.CacheBehavior({ pathPattern: '*', ...props.defaultBehavior });
        this.addOrigin(this.defaultBehavior.origin);
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig: {
                enabled: true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                customErrorResponses: this.renderErrorResponses(),
                priceClass: (_b = props.priceClass) !== null && _b !== void 0 ? _b : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior({ pathPattern, origin, ...behaviorOptions }));
        this.addOrigin(origin);
    }
    addOrigin(origin) {
        if (!this.origins.has(origin)) {
            this.origins.add(origin);
            origin.bind(this, { originIndex: this.origins.size });
        }
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.origins.forEach(origin => renderedOrigins.push(origin.renderOrigin()));
        return renderedOrigins;
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2018,
        };
    }
}
exports.Distribution = Distribution;
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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