"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mesh = exports.MeshFilterType = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
const virtual_service_1 = require("./virtual-service");
/**
 * A utility enum defined for the egressFilter type property, the default of DROP_ALL,
 * allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 */
var MeshFilterType;
(function (MeshFilterType) {
    /**
     * Allows all outbound traffic
     */
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    /**
     * Allows traffic only to other resources inside the mesh, or API calls to amazon resources
     */
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * Adds a VirtualRouter to the Mesh with the given id and props
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualService with the given id
     */
    addVirtualService(id, props = {}) {
        return new virtual_service_1.VirtualService(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualNode to the Mesh
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * Define a new AppMesh mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 */
class Mesh extends MeshBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing mesh by arn
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing mesh by name
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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