"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApi = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const apiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: apiName,
            protocolType: 'HTTP',
            corsConfiguration,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
                domainMapping: props === null || props === void 0 ? void 0 : props.defaultDomainMapping,
            });
            // to ensure the domain is ready before creating the default stage
            if (props === null || props === void 0 ? void 0 : props.defaultDomainMapping) {
                this.defaultStage.node.addDependency(props.defaultDomainMapping.domainName);
            }
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === false && props.defaultDomainMapping) {
            throw new Error('defaultDomainMapping not supported with createDefaultStage disabled');
        }
    }
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * Add a new stage.
     */
    addStage(id, options) {
        const stage = new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
        return stage;
    }
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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