"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const aws_credentials_1 = require("./private/aws-credentials");
const github = require("./workflows-model");
const yaml_file_1 = require("./yaml-file");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.gitHubActionRoleArn = props.gitHubActionRoleArn;
        this.useGitHubActionRole = this.gitHubActionRoleArn ? true : false;
        this.jobSettings = props.jobSettings;
        this.awsCredentials = props.awsCredentials ?? {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        this.workflowFile.update(workflow);
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const diffProtection = this.node.tryGetContext('cdk-pipelines-github:diffProtection') ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || this.workflowFile.toYaml() !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        this.workflowFile.writeFile();
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Worflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
            default:
                // The 'as any' is temporary, until the change upstream rolls out
                throw new Error(`GitHubWorfklow does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -x'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '::set-output name=${ASSET_HASH_NAME}::${assetId}'`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.jobSettings,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region: this.publishAssetsAuthRegion }),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.jobSettings,
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                ...(this.stackProperties[stack.stackArtifactId]?.environment ? {
                    environment: this.stackProperties[stack.stackArtifactId].environment,
                } : {}),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region, assumeRoleArn }),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    stepsToConfigureAws(openId, { region, assumeRoleArn }) {
        function getDeployRole(arn) {
            return arn.replace('cfn-exec', 'deploy');
        }
        let steps = [];
        if (openId) {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via OIDC Role', {
                region,
                gitHubActionRoleArn: this.gitHubActionRoleArn,
            }));
            if (assumeRoleArn) {
                // Result of initial credentials with GitHub Action role are these environment variables
                steps.push(aws_credentials_1.awsCredentialStep('Assume CDK Deploy Role', {
                    region,
                    accessKeyId: '${{ env.AWS_ACCESS_KEY_ID }}',
                    secretAccessKey: '${{ env.AWS_SECRET_ACCESS_KEY }}',
                    sessionToken: '${{ env.AWS_SESSION_TOKEN }}',
                    roleToAssume: getDeployRole(assumeRoleArn),
                }));
            }
        }
        else {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via GitHub Secrets', {
                region,
                accessKeyId: `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
                secretAccessKey: `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
                ...(this.awsCredentials.sessionToken ? {
                    sessionToken: `\${{ secrets.${this.awsCredentials.sessionToken} }}`,
                } : undefined),
                roleToAssume: assumeRoleArn,
            }));
        }
        return steps;
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.2.197" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvcGlwZWxpbmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwyQkFBd0U7QUFDeEUsNkJBQTZCO0FBQzdCLDZDQUFvQztBQUNwQywrQ0FBNkQ7QUFDN0QscURBQTZKO0FBQzdKLGlGQUF1RztBQUV2Ryx5Q0FBeUM7QUFFekMsK0RBQThEO0FBRTlELDRDQUE0QztBQUM1QywyQ0FBdUM7QUFFdkMsTUFBTSxlQUFlLEdBQUcsU0FBUyxDQUFDO0FBQ2xDLE1BQU0sZUFBZSxHQUFHLFlBQVksQ0FBQztBQWtJckM7O0dBRUc7QUFDSCxNQUFhLGNBQWUsU0FBUSx3QkFBWTtJQXNCOUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwQjtRQUNsRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQVJULGVBQVUsR0FBMkMsRUFBRSxDQUFDO1FBQ3hELGlCQUFZLEdBQTJCLEVBQUUsQ0FBQztRQUcxQyxvQkFBZSxHQUF3QyxFQUFFLENBQUM7UUFNekUsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQ3pDLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUM7UUFDNUMsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBQzNDLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsSUFBSSxFQUFFLENBQUM7UUFDL0MsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxJQUFJLEVBQUUsQ0FBQztRQUNqRCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDO1FBQ3JELElBQUksQ0FBQyxtQkFBbUIsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1FBQ25FLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUVyQyxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLElBQUk7WUFDNUMsV0FBVyxFQUFFLG1CQUFtQjtZQUNoQyxlQUFlLEVBQUUsdUJBQXVCO1NBQ3pDLENBQUM7UUFFRixJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixJQUFJLEVBQUUsQ0FBQztRQUV2RCxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksOEJBQThCLENBQUM7UUFDekUsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDL0UsTUFBTSxJQUFJLEtBQUssQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1NBQ2hFO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RkFBeUYsQ0FBQyxDQUFDO1NBQzVHO1FBRUQsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLG9CQUFRLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksSUFBSSxRQUFRLENBQUM7UUFDbkQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSTtZQUNoRCxJQUFJLEVBQUUsRUFBRSxRQUFRLEVBQUUsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUM1QixnQkFBZ0IsRUFBRSxFQUFFO1NBQ3JCLENBQUM7UUFFRixJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUM7UUFDMUQsSUFBSSxDQUFDLHVCQUF1QixHQUFHLEtBQUssQ0FBQyx1QkFBdUIsSUFBSSxXQUFXLENBQUM7SUFDOUUsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0kseUJBQXlCLENBQUMsS0FBWSxFQUFFLE9BQStCO1FBQzVFLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXRELHdDQUF3QztRQUN4QyxNQUFNLE1BQU0sR0FBRyxlQUFlLENBQUMsTUFBTSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxjQUFjLEVBQUUsT0FBTyxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDdkUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUU3RCxPQUFPLGVBQWUsQ0FBQztJQUN6QixDQUFDO0lBRU8sYUFBYSxDQUFDLE1BQXlCLEVBQUUsR0FBVyxFQUFFLEtBQVU7UUFDdEUsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQUUsT0FBTztTQUFFO1FBQ3BDLEtBQUssTUFBTSxLQUFLLElBQUksTUFBTSxFQUFFO1lBQzFCLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxHQUFHO2dCQUM1QyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQztnQkFDOUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxLQUFLO2FBQ2IsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVTLGVBQWU7UUFDdkIsTUFBTSxHQUFHLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0IsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztTQUMvRTtRQUNELE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUM7UUFFN0IsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLEVBQU8sQ0FBQztRQUU5QixNQUFNLFNBQVMsR0FBRyxJQUFJLGdDQUFhLENBQUMsSUFBSSxFQUFFO1lBQ3hDLFlBQVksRUFBRSxLQUFLO1lBQ25CLGVBQWUsRUFBRSxJQUFJO1lBQ3JCLFdBQVcsRUFBRSxLQUFLO1NBQ25CLENBQUMsQ0FBQztRQUVILEtBQUssTUFBTSxTQUFTLElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLENBQUMsRUFBRTtZQUNqRSxJQUFJLENBQUMsMEJBQU8sQ0FBQyxTQUFTLENBQUMsRUFBRTtnQkFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQywyQ0FBMkMsU0FBUyxHQUFHLENBQUMsQ0FBQzthQUMxRTtZQUVELE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUUxQyxLQUFLLE1BQU0sT0FBTyxJQUFJLFFBQVEsRUFBRTtnQkFDOUIsS0FBSyxNQUFNLElBQUksSUFBSSxPQUFPLEVBQUU7b0JBQzFCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFO3dCQUNoQyxXQUFXLEVBQUUsU0FBUzt3QkFDdEIsU0FBUztxQkFDVixDQUFDLENBQUM7b0JBRUgsSUFBSSxHQUFHLEVBQUU7d0JBQ1AsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGO1FBRUQsOERBQThEO1FBQzlELE1BQU0sTUFBTSxHQUErQixFQUFFLENBQUM7UUFDOUMsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDdEIsSUFBSSxHQUFHLENBQUMsRUFBRSxJQUFJLE1BQU0sRUFBRTtnQkFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7YUFDL0M7WUFDRCxNQUFNLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLGFBQWEsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDaEQ7UUFFRCw4Q0FBOEM7UUFDOUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRTlCLE1BQU0sUUFBUSxHQUFHO1lBQ2YsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3ZCLEVBQUUsRUFBRSxhQUFhLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLEdBQUcsQ0FBQztZQUM3QyxJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUM7UUFFRix1QkFBdUI7UUFDdkIsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFbkMsd0NBQXdDO1FBQ3hDLGNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRWhFLHNFQUFzRTtRQUN0RSxtSEFBbUg7UUFDbkgsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMscUNBQXFDLENBQUMsSUFBSSxJQUFJLENBQUM7UUFDOUYsSUFBSSxjQUFjLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxlQUFlLEtBQUssSUFBSSxDQUFDLFlBQVksRUFBRTtZQUN2RSxxQ0FBcUM7WUFDckMsSUFBSSxDQUFDLGVBQVUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsS0FBSyxpQkFBWSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsTUFBTSxDQUFDLEVBQUU7Z0JBQzVHLE1BQU0sSUFBSSxLQUFLLENBQUMsMkNBQTJDLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsNENBQTRDLENBQUMsQ0FBQzthQUNySjtTQUNGO1FBRUQsSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUNoQyxDQUFDO0lBRU8sZ0JBQWdCLENBQUMsTUFBa0M7UUFDekQsS0FBSyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ2pFLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRztnQkFDZCxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUM7Z0JBQ2hCLE9BQU8sRUFBRTtvQkFDUCxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPO29CQUN4QixHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUM7aUJBQ3JDO2FBQ0YsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVPLGdCQUFnQixDQUFDLE9BQStCO1FBQ3RELE1BQU0sZUFBZSxHQUEyQixFQUFFLENBQUM7UUFDbkQsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDNUIsZUFBZSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBRyxjQUFjLE1BQU0sQ0FBQyxNQUFNLFlBQVksTUFBTSxDQUFDLFVBQVUsS0FBSyxDQUFDO1NBQ3BHO1FBQ0QsT0FBTyxlQUFlLENBQUM7SUFDekIsQ0FBQztJQUVEOztPQUVHO0lBQ0ssVUFBVSxDQUFDLElBQWdCLEVBQUUsT0FBZ0I7UUFDbkQsUUFBUSxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRTtZQUN2Qiw2RUFBNkU7WUFDN0UsS0FBSyxPQUFPLENBQUM7WUFDYixLQUFLLGFBQWEsQ0FBQztZQUNuQixLQUFLLFNBQVM7Z0JBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyxrREFBa0QsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBRXZGLEtBQUssYUFBYTtnQkFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1lBRXJFLEtBQUssZ0JBQWdCO2dCQUNuQixPQUFPLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFFbEUsS0FBSyxTQUFTO2dCQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztZQUVuRSxLQUFLLFNBQVM7Z0JBQ1osT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBRTVFLEtBQUssTUFBTTtnQkFDVCxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFO29CQUN6QixPQUFPLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQ25EO3FCQUFNLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLFlBQVkscUJBQVMsRUFBRTtvQkFDOUMsT0FBTyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQ3BEO3FCQUFNO29CQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsMEJBQTBCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO2lCQUM5RTtZQUVIO2dCQUNFLGlFQUFpRTtnQkFDakUsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBeUQsSUFBSSxDQUFDLElBQVksRUFBRSxJQUFJLHlGQUF5RixDQUFDLENBQUM7U0FDOUw7SUFDSCxDQUFDO0lBRU8sa0JBQWtCLENBQUMsSUFBZ0IsRUFBRSxNQUFvQixFQUFFLE9BQWdCO1FBQ2pGLElBQUksTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xFO1FBRUQsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUN6RSxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQ3RDLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDNUIsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUVsQyxtRUFBbUU7UUFDbkUsTUFBTSxnQkFBZ0IsR0FBcUIsRUFBRSxDQUFDO1FBQzlDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLElBQUksSUFBSSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDOUUsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsaUJBQWlCLEVBQUU7Z0JBQzFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO2FBQzlEO1NBQ0Y7UUFFRCxvQ0FBb0M7UUFDcEMsTUFBTSxrQkFBa0IsR0FBRyxDQUFDLENBQVMsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2hILE1BQU0sWUFBWSxHQUFhLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRTtZQUNwRSxPQUFPLDBCQUEwQixrQkFBa0IsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsd0JBQXdCLEtBQUssQ0FBQyxhQUFhLEdBQUcsQ0FBQztRQUM3SCxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRUosbURBQW1EO1FBQ25ELElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLEdBQUcsS0FBSyxDQUFDO1FBQ25DLFlBQVksQ0FBQyxJQUFJLENBQUMsMkJBQTJCLGVBQWUsS0FBSyxPQUFPLEdBQUcsQ0FBQyxDQUFDO1FBRTdFLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFdBQVcsS0FBSyxVQUFVLENBQUMsQ0FBQztRQUN6RSxjQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzlELGtCQUFhLENBQUMsZUFBZSxFQUFFLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUUvRSxNQUFNLFdBQVcsR0FBbUI7WUFDbEMsRUFBRSxFQUFFLFNBQVM7WUFDYixJQUFJLEVBQUUsV0FBVyxLQUFLLEVBQUU7WUFDeEIsR0FBRyxFQUFFLHVCQUF1QixJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxlQUFlLENBQUMsRUFBRTtTQUN4RSxDQUFDO1FBRUYsT0FBTztZQUNMLEVBQUUsRUFBRSxLQUFLO1lBQ1QsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRSxrQkFBa0IsS0FBSyxFQUFFO2dCQUMvQixHQUFHLElBQUksQ0FBQyxXQUFXO2dCQUNuQixLQUFLLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQztnQkFDcEMsV0FBVyxFQUFFO29CQUNYLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQ25DLE9BQU8sRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7aUJBQzNGO2dCQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU07Z0JBQzFCLE9BQU8sRUFBRTtvQkFDUCxDQUFDLGVBQWUsQ0FBQyxFQUFFLDhCQUE4QixlQUFlLEtBQUs7aUJBQ3RFO2dCQUNELEtBQUssRUFBRTtvQkFDTCxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxTQUFTLENBQUM7b0JBQzFDO3dCQUNFLElBQUksRUFBRSxTQUFTO3dCQUNmLEdBQUcsRUFBRSxtQ0FBbUMsYUFBYSxFQUFFO3FCQUN4RDtvQkFDRCxHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsRUFBRSxNQUFNLEVBQUUsSUFBSSxDQUFDLHVCQUF1QixFQUFFLENBQUM7b0JBQy9GLEdBQUcsZ0JBQWdCO29CQUNuQixXQUFXO2lCQUNaO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVPLFlBQVksQ0FBQyxJQUFnQixFQUFFLEtBQXNCLEVBQUUsZUFBd0I7UUFDckYsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUM1QixNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzlCLElBQUksQ0FBQyxNQUFNLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FDeEY7UUFFRCxNQUFNLE9BQU8sR0FBRyxDQUFDLENBQVMsRUFBVSxFQUFFO1lBQ3BDLE9BQU8sZ0NBQXVCLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRTtnQkFDeEMsU0FBUyxFQUFFLE9BQU87Z0JBQ2xCLE1BQU0sRUFBRSxNQUFNO2dCQUNkLFNBQVMsRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxRQUFnQixFQUFFLEVBQUU7WUFDNUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQztZQUM5RCxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFO2dCQUN6QyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixJQUFJLGFBQWEsQ0FBQyxDQUFDO2FBQzNEO1lBQ0QsT0FBTyxRQUFRLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxjQUFjLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFlBQVksZUFBZSxLQUFLLENBQUMsQ0FBQztRQUN2RyxDQUFDLENBQUM7UUFFRixNQUFNLE1BQU0sR0FBd0I7WUFDbEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ3ZCLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ3hELDRCQUE0QixFQUFFLEdBQUc7U0FDbEMsQ0FBQztRQUVGLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFLFlBQVksQ0FBQztRQUMvRSxJQUFJLFlBQVksRUFBRTtZQUNoQixNQUFNLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDckQ7UUFFRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1NBQ3REO1FBQ0QsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRXJGLE9BQU87WUFDTCxFQUFFLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDakIsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRSxVQUFVLEtBQUssQ0FBQyxlQUFlLEVBQUU7Z0JBQ3ZDLEdBQUcsSUFBSSxDQUFDLFdBQVc7Z0JBQ25CLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLEVBQUUsUUFBUTtnQkFDeEQsV0FBVyxFQUFFO29CQUNYLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQ25DLE9BQU8sRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7aUJBQzNGO2dCQUNELEdBQUcsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsRUFBRSxXQUFXLENBQUMsQ0FBQyxDQUFDO29CQUM3RCxXQUFXLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUMsV0FBVztpQkFDckUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUNQLEtBQUssRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDO2dCQUNwQyxNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2dCQUMxQixLQUFLLEVBQUU7b0JBQ0wsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLG1CQUFtQixFQUFFLEVBQUUsTUFBTSxFQUFFLGFBQWEsRUFBRSxDQUFDO29CQUNoRjt3QkFDRSxFQUFFLEVBQUUsUUFBUTt3QkFDWixJQUFJLEVBQUUsaURBQWlEO3dCQUN2RCxJQUFJLEVBQUUsTUFBTTtxQkFDYjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFTyxlQUFlLENBQUMsSUFBZ0IsRUFBRSxJQUFVO1FBQ2xELElBQUksQ0FBQyxDQUFDLElBQUksWUFBWSxxQkFBUyxDQUFDLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1NBQ25EO1FBRUQsSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDMUIsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1NBQ2pEO1FBRUQsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUUvQixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3RELElBQUksRUFBRSxTQUFTO2dCQUNmLEdBQUcsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7YUFDckMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFUixPQUFPO1lBQ0wsRUFBRSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ2pCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUUsWUFBWTtnQkFDbEIsR0FBRyxJQUFJLENBQUMsV0FBVztnQkFDbkIsV0FBVyxFQUFFO29CQUNYLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7b0JBQ25DLHVGQUF1RjtvQkFDdkYsOENBQThDO29CQUM5QyxPQUFPLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFJO2lCQUMzRjtnQkFDRCxNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNO2dCQUMxQixLQUFLLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQztnQkFDcEMsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLFNBQVMsRUFBRSxJQUFJLENBQUMsY0FBYztnQkFDOUIsS0FBSyxFQUFFO29CQUNMLEdBQUcsSUFBSSxDQUFDLGVBQWUsRUFBRTtvQkFDekIsR0FBRyxJQUFJLENBQUMsYUFBYTtvQkFDckIsR0FBRyxZQUFZO29CQUNmO3dCQUNFLElBQUksRUFBRSxPQUFPO3dCQUNiLEdBQUcsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7cUJBQzlCO29CQUNELEdBQUcsSUFBSSxDQUFDLGNBQWM7b0JBQ3RCLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUM7aUJBQ2hEO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNLLGlCQUFpQixDQUFDLEdBQXlCLEVBQUUsSUFBZ0I7UUFDbkUsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQzlCLElBQUksR0FBRyxDQUFDLElBQUksRUFBRSxJQUFJLEtBQUssU0FBUyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDcEUsT0FBTyxHQUFHLENBQUMsUUFBUSxDQUFDO2FBQ3JCO1NBQ0Y7UUFDRCxzQkFBc0I7UUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEdBQUcsQ0FBQyxVQUFVLG9EQUFvRCxDQUFDLENBQUM7SUFDcEcsQ0FBQztJQUVPLFlBQVksQ0FBQyxLQUFhLEVBQUUsTUFBNEI7UUFDOUQsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLFNBQVMsRUFBRTtZQUN4QyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDbkM7YUFBTTtZQUNMLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3JDO0lBQ0gsQ0FBQztJQUVPLGdCQUFnQixDQUFDLElBQWdCLEVBQUUsSUFBZTtRQUN4RCxNQUFNLFlBQVksR0FBMkIsRUFBRSxDQUFDO1FBQ2hELEtBQUssTUFBTSxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFO1lBQ25FLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDaEQsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUU7Z0JBQ3ZCLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVTtnQkFDMUIsTUFBTSxFQUFFLFFBQVE7YUFDakIsQ0FBQyxDQUFDO1lBQ0gsWUFBWSxDQUFDLE9BQU8sQ0FBQyxHQUFHLGNBQWMsS0FBSyxZQUFZLEdBQUcsQ0FBQyxVQUFVLEtBQUssQ0FBQztTQUM1RTtRQUVELE1BQU0sY0FBYyxHQUFHLElBQUksS0FBSyxFQUFrQixDQUFDO1FBQ25ELE1BQU0sYUFBYSxHQUFHLElBQUksS0FBSyxFQUFrQixDQUFDO1FBRWxELEtBQUssTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUMvQixjQUFjLENBQUMsSUFBSSxDQUFDO2dCQUNsQixJQUFJLEVBQUUsOEJBQThCO2dCQUNwQyxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtvQkFDdEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxTQUFTO2lCQUN0QjthQUNGLENBQUMsQ0FBQztTQUNKO1FBRUQsS0FBSyxNQUFNLE1BQU0sSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ2pDLGFBQWEsQ0FBQyxJQUFJLENBQUM7Z0JBQ2pCLElBQUksRUFBRSxnQ0FBZ0M7Z0JBQ3RDLElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFO29CQUN2QixJQUFJLEVBQUUsTUFBTSxDQUFDLFNBQVM7aUJBQ3ZCO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3RELElBQUksRUFBRSxTQUFTO2dCQUNmLEdBQUcsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7YUFDckMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFUixPQUFPO1lBQ0wsRUFBRSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ2pCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUUsSUFBSSxDQUFDLEVBQUU7Z0JBQ2IsR0FBRyxJQUFJLENBQUMsV0FBVztnQkFDbkIsV0FBVyxFQUFFO29CQUNYLFFBQVEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7aUJBQ3BDO2dCQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU07Z0JBQzFCLEtBQUssRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDO2dCQUNwQyxHQUFHLEVBQUU7b0JBQ0gsR0FBRyxJQUFJLENBQUMsR0FBRztvQkFDWCxHQUFHLFlBQVk7aUJBQ2hCO2dCQUNELEtBQUssRUFBRTtvQkFDTCxHQUFHLGNBQWM7b0JBQ2pCLEdBQUcsWUFBWTtvQkFDZixFQUFFLEdBQUcsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtvQkFDakMsR0FBRyxhQUFhO2lCQUNqQjthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFTyxtQkFBbUIsQ0FBQyxNQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsYUFBYSxFQUE4QztRQUNoSCxTQUFTLGFBQWEsQ0FBQyxHQUFXO1lBQ2hDLE9BQU8sR0FBRyxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDM0MsQ0FBQztRQUVELElBQUksS0FBSyxHQUFxQixFQUFFLENBQUM7UUFFakMsSUFBSSxNQUFNLEVBQUU7WUFDVixLQUFLLENBQUMsSUFBSSxDQUFDLG1DQUFpQixDQUFDLDRCQUE0QixFQUFFO2dCQUN6RCxNQUFNO2dCQUNOLG1CQUFtQixFQUFFLElBQUksQ0FBQyxtQkFBbUI7YUFDOUMsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLGFBQWEsRUFBRTtnQkFDakIsd0ZBQXdGO2dCQUN4RixLQUFLLENBQUMsSUFBSSxDQUFDLG1DQUFpQixDQUFDLHdCQUF3QixFQUFFO29CQUNyRCxNQUFNO29CQUNOLFdBQVcsRUFBRSw4QkFBOEI7b0JBQzNDLGVBQWUsRUFBRSxrQ0FBa0M7b0JBQ25ELFlBQVksRUFBRSw4QkFBOEI7b0JBQzVDLFlBQVksRUFBRSxhQUFhLENBQUMsYUFBYSxDQUFDO2lCQUMzQyxDQUFDLENBQUMsQ0FBQzthQUNMO1NBQ0Y7YUFBTTtZQUNMLEtBQUssQ0FBQyxJQUFJLENBQUMsbUNBQWlCLENBQUMsaUNBQWlDLEVBQUU7Z0JBQzlELE1BQU07Z0JBQ04sV0FBVyxFQUFFLGdCQUFnQixJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsS0FBSztnQkFDakUsZUFBZSxFQUFFLGdCQUFnQixJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWUsS0FBSztnQkFDekUsR0FBRyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztvQkFDckMsWUFBWSxFQUFFLGdCQUFnQixJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksS0FBSztpQkFDcEUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNkLFlBQVksRUFBRSxhQUFhO2FBQzVCLENBQUMsQ0FBQyxDQUFDO1NBQ0w7UUFFRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFTyxzQkFBc0IsQ0FBQyxnQkFBa0M7UUFDL0QsSUFBSSxNQUEyQixDQUFDO1FBRWhDLElBQUksZ0JBQWdCLENBQUMsSUFBSSxLQUFLLFFBQVEsRUFBRTtZQUN0QyxNQUFNLEdBQUc7Z0JBQ1AsUUFBUSxFQUFFLGdCQUFnQixnQkFBZ0IsQ0FBQyxXQUFXLEtBQUs7Z0JBQzNELFFBQVEsRUFBRSxnQkFBZ0IsZ0JBQWdCLENBQUMsV0FBVyxLQUFLO2FBQzVELENBQUM7U0FDSDthQUFNLElBQUksZ0JBQWdCLENBQUMsSUFBSSxLQUFLLEtBQUssRUFBRTtZQUMxQyxNQUFNLEdBQUc7Z0JBQ1AsUUFBUSxFQUFFLGdCQUFnQixDQUFDLFFBQVE7YUFDcEMsQ0FBQztTQUNIO2FBQU07WUFDTCxNQUFNLEdBQUc7Z0JBQ1AsUUFBUSxFQUFFLGdCQUFnQixDQUFDLFFBQVE7Z0JBQ25DLFFBQVEsRUFBRSxnQkFBZ0IsZ0JBQWdCLENBQUMsV0FBVyxLQUFLO2dCQUMzRCxRQUFRLEVBQUUsZ0JBQWdCLGdCQUFnQixDQUFDLFdBQVcsS0FBSzthQUM1RCxDQUFDO1NBQ0g7UUFFRCxPQUFPO1lBQ0w7Z0JBQ0UsSUFBSSxFQUFFLHdCQUF3QjtnQkFDOUIsSUFBSSxFQUFFLE1BQU07YUFDYjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sdUJBQXVCLENBQUMsU0FBaUI7UUFDL0MsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE9BQU8sSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO1NBQy9CO1FBRUQsT0FBTyxDQUFDO2dCQUNOLElBQUksRUFBRSxZQUFZLGVBQWUsRUFBRTtnQkFDbkMsSUFBSSxFQUFFLDhCQUE4QjtnQkFDcEMsSUFBSSxFQUFFO29CQUNKLElBQUksRUFBRSxlQUFlO29CQUNyQixJQUFJLEVBQUUsU0FBUztpQkFDaEI7YUFDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sZUFBZTtRQUNyQixPQUFPLENBQUM7Z0JBQ04sSUFBSSxFQUFFLFVBQVU7Z0JBQ2hCLElBQUksRUFBRSxxQkFBcUI7YUFDNUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLHFCQUFxQixDQUFDLEdBQVc7UUFDdkMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ25CLE9BQU8sRUFBRSxDQUFDO1NBQ1g7UUFFRCxPQUFPLENBQUM7Z0JBQ04sSUFBSSxFQUFFLFVBQVUsZUFBZSxFQUFFO2dCQUNqQyxJQUFJLEVBQUUsZ0NBQWdDO2dCQUN0QyxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLElBQUksRUFBRSxHQUFHO2lCQUNWO2FBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGtCQUFrQixDQUFDLElBQWdCO1FBQ3pDLE1BQU0sSUFBSSxHQUFHLElBQUksS0FBSyxFQUFjLENBQUM7UUFFckMsS0FBSyxNQUFNLENBQUMsSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQzVCLElBQUksQ0FBQyxZQUFZLHdCQUFLLEVBQUU7Z0JBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDbkM7aUJBQU07Z0JBQ0wsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUNkO1NBQ0Y7UUFFRCxPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDbkMsQ0FBQzs7QUFwbUJILHdDQXFtQkM7OztBQW1CRCxTQUFTLGFBQWEsQ0FBYyxHQUFNLEVBQUUsR0FBRyxHQUFHLEdBQUc7SUFDbkQsSUFBSSxPQUFPLEdBQUcsS0FBSyxRQUFRLElBQUksR0FBRyxJQUFJLElBQUksRUFBRTtRQUMxQyxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3RCLE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQVEsQ0FBQztLQUNuRDtJQUVELE1BQU0sTUFBTSxHQUE0QixFQUFFLENBQUM7SUFDM0MsS0FBSyxJQUFJLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDdEMsb0RBQW9EO1FBQ3BELElBQUksQ0FBQyxLQUFLLEtBQUssSUFBSSxPQUFPLENBQUMsS0FBSyxRQUFRLElBQUksQ0FBQyxJQUFJLElBQUksRUFBRTtZQUNyRCxDQUFDLEdBQUcsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3RCO1FBQ0QsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUMvQztJQUNELE9BQU8sTUFBYSxDQUFDO0FBQ3ZCLENBQUM7QUFzQkQsUUFBZSxDQUFDLENBQUMsT0FBTyxDQUFJLEVBQWlCO0lBQzNDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ2xCLEtBQUssTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2pCLE1BQU0sQ0FBQyxDQUFDO1NBQ1Q7S0FDRjtBQUNILENBQUM7QUFORCwwQkFNQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IG1rZGlyU3luYywgd3JpdGVGaWxlU3luYywgcmVhZEZpbGVTeW5jLCBleGlzdHNTeW5jIH0gZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IFN0YWdlIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgRW52aXJvbm1lbnRQbGFjZWhvbGRlcnMgfSBmcm9tICdhd3MtY2RrLWxpYi9jeC1hcGknO1xuaW1wb3J0IHsgUGlwZWxpbmVCYXNlLCBQaXBlbGluZUJhc2VQcm9wcywgU2hlbGxTdGVwLCBTdGFja0Fzc2V0LCBTdGFja0RlcGxveW1lbnQsIFN0YWNrT3V0cHV0UmVmZXJlbmNlLCBTdGFnZURlcGxveW1lbnQsIFN0ZXAgfSBmcm9tICdhd3MtY2RrLWxpYi9waXBlbGluZXMnO1xuaW1wb3J0IHsgQUdyYXBoTm9kZSwgUGlwZWxpbmVHcmFwaCwgR3JhcGgsIGlzR3JhcGggfSBmcm9tICdhd3MtY2RrLWxpYi9waXBlbGluZXMvbGliL2hlbHBlcnMtaW50ZXJuYWwnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBkZWNhbWVsaXplIGZyb20gJ2RlY2FtZWxpemUnO1xuaW1wb3J0IHsgRG9ja2VyQ3JlZGVudGlhbCB9IGZyb20gJy4vZG9ja2VyLWNyZWRlbnRpYWxzJztcbmltcG9ydCB7IGF3c0NyZWRlbnRpYWxTdGVwIH0gZnJvbSAnLi9wcml2YXRlL2F3cy1jcmVkZW50aWFscyc7XG5pbXBvcnQgeyBBZGRHaXRIdWJTdGFnZU9wdGlvbnMgfSBmcm9tICcuL3N0YWdlLW9wdGlvbnMnO1xuaW1wb3J0ICogYXMgZ2l0aHViIGZyb20gJy4vd29ya2Zsb3dzLW1vZGVsJztcbmltcG9ydCB7IFlhbWxGaWxlIH0gZnJvbSAnLi95YW1sLWZpbGUnO1xuXG5jb25zdCBDREtPVVRfQVJUSUZBQ1QgPSAnY2RrLm91dCc7XG5jb25zdCBBU1NFVF9IQVNIX05BTUUgPSAnYXNzZXQtaGFzaCc7XG5cbi8qKlxuICogSm9iIGxldmVsIHNldHRpbmdzIGFwcGxpZWQgdG8gYWxsIGpvYnMgaW4gdGhlIHdvcmtmbG93LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYlNldHRpbmdzIHtcbiAgLyoqXG4gICAqIGpvYnMuPGpvYl9pZD4uaWYuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vYWN0aW9ucy91c2luZy13b3JrZmxvd3Mvd29ya2Zsb3ctc3ludGF4LWZvci1naXRodWItYWN0aW9ucyNqb2Jzam9iX2lkaWZcbiAgICovXG4gIHJlYWRvbmx5IGlmPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BzIGZvciBgR2l0SHViV29ya2Zsb3dgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdpdEh1YldvcmtmbG93UHJvcHMgZXh0ZW5kcyBQaXBlbGluZUJhc2VQcm9wcyB7XG4gIC8qKlxuICAgKiBGaWxlIHBhdGggZm9yIHRoZSBHaXRIdWIgd29ya2Zsb3cuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiLmdpdGh1Yi93b3JrZmxvd3MvZGVwbG95LnltbFwiXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd1BhdGg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIHdvcmtmbG93LlxuICAgKlxuICAgKiBAZGVmYXVsdCBcImRlcGxveVwiXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd05hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEdpdEh1YiB3b3JrZmxvdyB0cmlnZ2Vycy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBCeSBkZWZhdWx0LCB3b3JrZmxvdyBpcyB0cmlnZ2VyZWQgb24gcHVzaCB0byB0aGUgYG1haW5gIGJyYW5jaFxuICAgKiBhbmQgY2FuIGFsc28gYmUgdHJpZ2dlcmVkIG1hbnVhbGx5IChgd29ya2Zsb3dfZGlzcGF0Y2hgKS5cbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93VHJpZ2dlcnM/OiBnaXRodWIuV29ya2Zsb3dUcmlnZ2VycztcblxuICAvKipcbiAgICogVmVyc2lvbiBvZiB0aGUgQ0RLIENMSSB0byB1c2UuXG4gICAqIEBkZWZhdWx0IC0gYXV0b21hdGljXG4gICAqL1xuICByZWFkb25seSBjZGtDbGlWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgdGhlIHJlcG9zaXRvcnkgYWxyZWFkeSBjb250YWlucyBhIHN5bnRoZXNpemVkIGBjZGsub3V0YCBkaXJlY3RvcnksIGluIHdoaWNoXG4gICAqIGNhc2Ugd2Ugd2lsbCBzaW1wbHkgY2hlY2tvdXQgdGhlIHJlcG8gaW4gam9icyB0aGF0IHJlcXVpcmUgYGNkay5vdXRgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJlU3ludGhlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE5hbWVzIG9mIEdpdEh1YiByZXBvc2l0b3J5IHNlY3JldHMgdGhhdCBpbmNsdWRlIEFXUyBjcmVkZW50aWFscyBmb3JcbiAgICogZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgQVdTX0FDQ0VTU19LRVlfSURgIGFuZCBgQVdTX1NFQ1JFVF9BQ0NFU1NfS0VZYC5cbiAgICovXG4gIHJlYWRvbmx5IGF3c0NyZWRlbnRpYWxzPzogQXdzQ3JlZGVudGlhbHNTZWNyZXRzO1xuXG4gIC8qKlxuICAgKiBBIHJvbGUgdGhhdCB1dGlsaXplcyB0aGUgR2l0SHViIE9JREMgSWRlbnRpdHkgUHJvdmlkZXIgaW4geW91ciBBV1MgYWNjb3VudC5cbiAgICogSWYgc3VwcGxpZWQsIHRoaXMgd2lsbCBiZSB1c2VkIGluc3RlYWQgb2YgYGF3c0NyZWRlbnRpYWxzYC5cbiAgICpcbiAgICogWW91IGNhbiBjcmVhdGUgeW91ciBvd24gcm9sZSBpbiB0aGUgY29uc29sZSB3aXRoIHRoZSBuZWNlc3NhcnkgdHJ1c3QgcG9saWN5XG4gICAqIHRvIGFsbG93IGdpdEh1YiBhY3Rpb25zIGZyb20geW91ciBnaXRIdWIgcmVwb3NpdG9yeSB0byBhc3N1bWUgdGhlIHJvbGUsIG9yXG4gICAqIHlvdSBjYW4gdXRpbGl6ZSB0aGUgYEdpdEh1YkFjdGlvblJvbGVgIGNvbnN0cnVjdCB0byBjcmVhdGUgYSByb2xlIGZvciB5b3UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gR2l0SHViIHJlcG9zaXRvcnkgc2VjcmV0cyBhcmUgdXNlZCBpbnN0ZWFkIG9mIE9wZW5JZCBDb25uZWN0IHJvbGUuXG4gICAqL1xuICByZWFkb25seSBnaXRIdWJBY3Rpb25Sb2xlQXJuPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBCdWlsZCBjb250YWluZXIgb3B0aW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBHaXRIdWIgZGVmYXVsdHNcbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkQ29udGFpbmVyPzogZ2l0aHViLkNvbnRhaW5lck9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEdpdEh1YiB3b3JrZmxvdyBzdGVwcyB0byBleGVjdXRlIGJlZm9yZSBidWlsZC5cbiAgICpcbiAgICogQGRlZmF1bHQgW11cbiAgICovXG4gIHJlYWRvbmx5IHByZUJ1aWxkU3RlcHM/OiBnaXRodWIuSm9iU3RlcFtdO1xuXG4gIC8qKlxuICAgKiBHaXRIdWIgd29ya2Zsb3cgc3RlcHMgdG8gZXhlY3V0ZSBhZnRlciBidWlsZC5cbiAgICpcbiAgICogQGRlZmF1bHQgW11cbiAgICovXG4gIHJlYWRvbmx5IHBvc3RCdWlsZFN0ZXBzPzogZ2l0aHViLkpvYlN0ZXBbXTtcblxuICAvKipcbiAgICogVGhlIERvY2tlciBDcmVkZW50aWFscyB0byB1c2UgdG8gbG9naW4uIElmIHlvdSBzZXQgdGhpcyB2YXJpYWJsZSxcbiAgICogeW91IHdpbGwgYmUgbG9nZ2VkIGluIHRvIGRvY2tlciB3aGVuIHlvdSB1cGxvYWQgRG9ja2VyIEFzc2V0cy5cbiAgICovXG4gIHJlYWRvbmx5IGRvY2tlckNyZWRlbnRpYWxzPzogRG9ja2VyQ3JlZGVudGlhbFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBydW5uZXIgdG8gcnVuIHRoZSBqb2Igb24uIFRoZSBydW5uZXIgY2FuIGJlIGVpdGhlciBhXG4gICAqIEdpdEh1Yi1ob3N0ZWQgcnVubmVyIG9yIGEgc2VsZi1ob3N0ZWQgcnVubmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBSdW5uZXIuVUJVTlRVX0xBVEVTVFxuICAgKi9cbiAgcmVhZG9ubHkgcnVubmVyPzogZ2l0aHViLlJ1bm5lcjtcblxuICAvKipcbiAgICogV2lsbCBhc3N1bWUgdGhlIEdpdEh1YkFjdGlvblJvbGUgaW4gdGhpcyByZWdpb24gd2hlbiBwdWJsaXNoaW5nIGFzc2V0cy5cbiAgICogVGhpcyBpcyBOT1QgdGhlIHJlZ2lvbiBpbiB3aGljaCB0aGUgYXNzZXRzIGFyZSBwdWJsaXNoZWQuXG4gICAqXG4gICAqIEluIG1vc3QgY2FzZXMsIHlvdSBkbyBub3QgaGF2ZSB0byB3b3JyeSBhYm91dCB0aGlzIHByb3BlcnR5LCBhbmQgY2FuIHNhZmVseVxuICAgKiBpZ25vcmUgaXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwidXMtd2VzdC0yXCJcbiAgICovXG4gIHJlYWRvbmx5IHB1Ymxpc2hBc3NldHNBdXRoUmVnaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBKb2IgbGV2ZWwgc2V0dGluZ3MgdGhhdCB3aWxsIGJlIGFwcGxpZWQgdG8gYWxsIGpvYnMgaW4gdGhlIHdvcmtmbG93LFxuICAgKiBpbmNsdWRpbmcgc3ludGggYW5kIGFzc2V0IGRlcGxveSBqb2JzLiBDdXJyZW50bHkgdGhlIG9ubHkgdmFsaWQgc2V0dGluZ1xuICAgKiBpcyAnaWYnLiBZb3UgY2FuIHVzZSB0aGlzIHRvIHJ1biBqb2JzIG9ubHkgaW4gc3BlY2lmaWMgcmVwb3NpdG9yaWVzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5naXRodWIuY29tL2VuL2FjdGlvbnMvdXNpbmctd29ya2Zsb3dzL3dvcmtmbG93LXN5bnRheC1mb3ItZ2l0aHViLWFjdGlvbnMjZXhhbXBsZS1vbmx5LXJ1bi1qb2ItZm9yLXNwZWNpZmljLXJlcG9zaXRvcnlcbiAgICovXG4gIHJlYWRvbmx5IGpvYlNldHRpbmdzPzogSm9iU2V0dGluZ3M7XG59XG5cbi8qKlxuICogQ0RLIFBpcGVsaW5lcyBmb3IgR2l0SHViIHdvcmtmbG93cy5cbiAqL1xuZXhwb3J0IGNsYXNzIEdpdEh1YldvcmtmbG93IGV4dGVuZHMgUGlwZWxpbmVCYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHdvcmtmbG93UGF0aDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgd29ya2Zsb3dOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSB3b3JrZmxvd0ZpbGU6IFlhbWxGaWxlO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya2Zsb3dUcmlnZ2VyczogZ2l0aHViLldvcmtmbG93VHJpZ2dlcnM7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJlU3ludGhlZDogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBhd3NDcmVkZW50aWFsczogQXdzQ3JlZGVudGlhbHNTZWNyZXRzO1xuICBwcml2YXRlIHJlYWRvbmx5IGdpdEh1YkFjdGlvblJvbGVBcm4/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgdXNlR2l0SHViQWN0aW9uUm9sZTogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBkb2NrZXJDcmVkZW50aWFsczogRG9ja2VyQ3JlZGVudGlhbFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGNka0NsaVZlcnNpb24/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgYnVpbGRDb250YWluZXI/OiBnaXRodWIuQ29udGFpbmVyT3B0aW9ucztcbiAgcHJpdmF0ZSByZWFkb25seSBwcmVCdWlsZFN0ZXBzOiBnaXRodWIuSm9iU3RlcFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IHBvc3RCdWlsZFN0ZXBzOiBnaXRodWIuSm9iU3RlcFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGpvYk91dHB1dHM6IFJlY29yZDxzdHJpbmcsIGdpdGh1Yi5Kb2JTdGVwT3V0cHV0W10+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgYXNzZXRIYXNoTWFwOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgcnVubmVyOiBnaXRodWIuUnVubmVyO1xuICBwcml2YXRlIHJlYWRvbmx5IHB1Ymxpc2hBc3NldHNBdXRoUmVnaW9uOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgc3RhY2tQcm9wZXJ0aWVzOiBSZWNvcmQ8c3RyaW5nLCBSZWNvcmQ8c3RyaW5nLCBhbnk+PiA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IGpvYlNldHRpbmdzPzogSm9iU2V0dGluZ3M7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEdpdEh1YldvcmtmbG93UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIHRoaXMuY2RrQ2xpVmVyc2lvbiA9IHByb3BzLmNka0NsaVZlcnNpb247XG4gICAgdGhpcy5wcmVTeW50aGVkID0gcHJvcHMucHJlU3ludGhlZCA/PyBmYWxzZTtcbiAgICB0aGlzLmJ1aWxkQ29udGFpbmVyID0gcHJvcHMuYnVpbGRDb250YWluZXI7XG4gICAgdGhpcy5wcmVCdWlsZFN0ZXBzID0gcHJvcHMucHJlQnVpbGRTdGVwcyA/PyBbXTtcbiAgICB0aGlzLnBvc3RCdWlsZFN0ZXBzID0gcHJvcHMucG9zdEJ1aWxkU3RlcHMgPz8gW107XG4gICAgdGhpcy5naXRIdWJBY3Rpb25Sb2xlQXJuID0gcHJvcHMuZ2l0SHViQWN0aW9uUm9sZUFybjtcbiAgICB0aGlzLnVzZUdpdEh1YkFjdGlvblJvbGUgPSB0aGlzLmdpdEh1YkFjdGlvblJvbGVBcm4gPyB0cnVlIDogZmFsc2U7XG4gICAgdGhpcy5qb2JTZXR0aW5ncyA9IHByb3BzLmpvYlNldHRpbmdzO1xuXG4gICAgdGhpcy5hd3NDcmVkZW50aWFscyA9IHByb3BzLmF3c0NyZWRlbnRpYWxzID8/IHtcbiAgICAgIGFjY2Vzc0tleUlkOiAnQVdTX0FDQ0VTU19LRVlfSUQnLFxuICAgICAgc2VjcmV0QWNjZXNzS2V5OiAnQVdTX1NFQ1JFVF9BQ0NFU1NfS0VZJyxcbiAgICB9O1xuXG4gICAgdGhpcy5kb2NrZXJDcmVkZW50aWFscyA9IHByb3BzLmRvY2tlckNyZWRlbnRpYWxzID8/IFtdO1xuXG4gICAgdGhpcy53b3JrZmxvd1BhdGggPSBwcm9wcy53b3JrZmxvd1BhdGggPz8gJy5naXRodWIvd29ya2Zsb3dzL2RlcGxveS55bWwnO1xuICAgIGlmICghdGhpcy53b3JrZmxvd1BhdGguZW5kc1dpdGgoJy55bWwnKSAmJiAhdGhpcy53b3JrZmxvd1BhdGguZW5kc1dpdGgoJy55YW1sJykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignd29ya2Zsb3cgZmlsZSBpcyBleHBlY3RlZCB0byBiZSBhIHlhbWwgZmlsZScpO1xuICAgIH1cbiAgICBpZiAoIXRoaXMud29ya2Zsb3dQYXRoLmluY2x1ZGVzKCcuZ2l0aHViL3dvcmtmbG93cy8nKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCd3b3JrZmxvdyBmaWxlcyBtdXN0IGJlIHN0b3JlZCBpbiB0aGUgXFwnLmdpdGh1Yi93b3JrZmxvd3NcXCcgZGlyZWN0b3J5IG9mIHlvdXIgcmVwb3NpdG9yeScpO1xuICAgIH1cblxuICAgIHRoaXMud29ya2Zsb3dGaWxlID0gbmV3IFlhbWxGaWxlKHRoaXMud29ya2Zsb3dQYXRoKTtcbiAgICB0aGlzLndvcmtmbG93TmFtZSA9IHByb3BzLndvcmtmbG93TmFtZSA/PyAnZGVwbG95JztcbiAgICB0aGlzLndvcmtmbG93VHJpZ2dlcnMgPSBwcm9wcy53b3JrZmxvd1RyaWdnZXJzID8/IHtcbiAgICAgIHB1c2g6IHsgYnJhbmNoZXM6IFsnbWFpbiddIH0sXG4gICAgICB3b3JrZmxvd0Rpc3BhdGNoOiB7fSxcbiAgICB9O1xuXG4gICAgdGhpcy5ydW5uZXIgPSBwcm9wcy5ydW5uZXIgPz8gZ2l0aHViLlJ1bm5lci5VQlVOVFVfTEFURVNUO1xuICAgIHRoaXMucHVibGlzaEFzc2V0c0F1dGhSZWdpb24gPSBwcm9wcy5wdWJsaXNoQXNzZXRzQXV0aFJlZ2lvbiA/PyAndXMtd2VzdC0yJztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXBsb3kgYSBzaW5nbGUgU3RhZ2UgYnkgaXRzZWxmIHdpdGggb3B0aW9ucyBmb3IgZnVydGhlciBHaXRIdWIgY29uZmlndXJhdGlvbi5cbiAgICpcbiAgICogQWRkIGEgU3RhZ2UgdG8gdGhlIHBpcGVsaW5lLCB0byBiZSBkZXBsb3llZCBpbiBzZXF1ZW5jZSB3aXRoIG90aGVyIFN0YWdlcyBhZGRlZCB0byB0aGUgcGlwZWxpbmUuXG4gICAqIEFsbCBTdGFja3MgaW4gdGhlIHN0YWdlIHdpbGwgYmUgZGVwbG95ZWQgaW4gYW4gb3JkZXIgYXV0b21hdGljYWxseSBkZXRlcm1pbmVkIGJ5IHRoZWlyIHJlbGF0aXZlIGRlcGVuZGVuY2llcy5cbiAgICovXG4gIHB1YmxpYyBhZGRTdGFnZVdpdGhHaXRIdWJPcHRpb25zKHN0YWdlOiBTdGFnZSwgb3B0aW9ucz86IEFkZEdpdEh1YlN0YWdlT3B0aW9ucyk6IFN0YWdlRGVwbG95bWVudCB7XG4gICAgY29uc3Qgc3RhZ2VEZXBsb3ltZW50ID0gdGhpcy5hZGRTdGFnZShzdGFnZSwgb3B0aW9ucyk7XG5cbiAgICAvLyBrZWVwIHRyYWNrIG9mIEdpdEh1YiBzcGVjaWZpYyBvcHRpb25zXG4gICAgY29uc3Qgc3RhY2tzID0gc3RhZ2VEZXBsb3ltZW50LnN0YWNrcztcbiAgICB0aGlzLmFkZFN0YWNrUHJvcHMoc3RhY2tzLCAnZW52aXJvbm1lbnQnLCBvcHRpb25zPy5naXRIdWJFbnZpcm9ubWVudCk7XG4gICAgdGhpcy5hZGRTdGFja1Byb3BzKHN0YWNrcywgJ2NhcGFiaWxpdGllcycsIG9wdGlvbnM/LnN0YWNrQ2FwYWJpbGl0aWVzKTtcbiAgICB0aGlzLmFkZFN0YWNrUHJvcHMoc3RhY2tzLCAnc2V0dGluZ3MnLCBvcHRpb25zPy5qb2JTZXR0aW5ncyk7XG5cbiAgICByZXR1cm4gc3RhZ2VEZXBsb3ltZW50O1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRTdGFja1Byb3BzKHN0YWNrczogU3RhY2tEZXBsb3ltZW50W10sIGtleTogc3RyaW5nLCB2YWx1ZTogYW55KSB7XG4gICAgaWYgKHZhbHVlID09PSB1bmRlZmluZWQpIHsgcmV0dXJuOyB9XG4gICAgZm9yIChjb25zdCBzdGFjayBvZiBzdGFja3MpIHtcbiAgICAgIHRoaXMuc3RhY2tQcm9wZXJ0aWVzW3N0YWNrLnN0YWNrQXJ0aWZhY3RJZF0gPSB7XG4gICAgICAgIC4uLnRoaXMuc3RhY2tQcm9wZXJ0aWVzW3N0YWNrLnN0YWNrQXJ0aWZhY3RJZF0sXG4gICAgICAgIFtrZXldOiB2YWx1ZSxcbiAgICAgIH07XG4gICAgfVxuICB9XG5cbiAgcHJvdGVjdGVkIGRvQnVpbGRQaXBlbGluZSgpIHtcbiAgICBjb25zdCBhcHAgPSBTdGFnZS5vZih0aGlzKTtcbiAgICBpZiAoIWFwcCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgR2l0SHViIFdvcmtmbG93IG11c3QgYmUgZGVmaW5lZCBpbiB0aGUgc2NvcGUgb2YgYW4gQXBwJyk7XG4gICAgfVxuICAgIGNvbnN0IGNka291dERpciA9IGFwcC5vdXRkaXI7XG5cbiAgICBjb25zdCBqb2JzID0gbmV3IEFycmF5PEpvYj4oKTtcblxuICAgIGNvbnN0IHN0cnVjdHVyZSA9IG5ldyBQaXBlbGluZUdyYXBoKHRoaXMsIHtcbiAgICAgIHNlbGZNdXRhdGlvbjogZmFsc2UsXG4gICAgICBwdWJsaXNoVGVtcGxhdGU6IHRydWUsXG4gICAgICBwcmVwYXJlU3RlcDogZmFsc2UsIC8vIHdlIGNyZWF0ZSBhbmQgZXhlY3V0ZSB0aGUgY2hhbmdlc2V0IGluIGEgc2luZ2xlIGpvYlxuICAgIH0pO1xuXG4gICAgZm9yIChjb25zdCBzdGFnZU5vZGUgb2YgZmxhdHRlbihzdHJ1Y3R1cmUuZ3JhcGguc29ydGVkQ2hpbGRyZW4oKSkpIHtcbiAgICAgIGlmICghaXNHcmFwaChzdGFnZU5vZGUpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVG9wLWxldmVsIGNoaWxkcmVuIG11c3QgYmUgZ3JhcGhzLCBnb3QgJyR7c3RhZ2VOb2RlfSdgKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgdHJhbmNoZXMgPSBzdGFnZU5vZGUuc29ydGVkTGVhdmVzKCk7XG5cbiAgICAgIGZvciAoY29uc3QgdHJhbmNoZSBvZiB0cmFuY2hlcykge1xuICAgICAgICBmb3IgKGNvbnN0IG5vZGUgb2YgdHJhbmNoZSkge1xuICAgICAgICAgIGNvbnN0IGpvYiA9IHRoaXMuam9iRm9yTm9kZShub2RlLCB7XG4gICAgICAgICAgICBhc3NlbWJseURpcjogY2Rrb3V0RGlyLFxuICAgICAgICAgICAgc3RydWN0dXJlLFxuICAgICAgICAgIH0pO1xuXG4gICAgICAgICAgaWYgKGpvYikge1xuICAgICAgICAgICAgam9icy5wdXNoKGpvYik7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gY29udmVydCBqb2JzIHRvIGEgbWFwIGFuZCBtYWtlIHN1cmUgdGhlcmUgYXJlIG5vIGR1cGxpY2F0ZXNcbiAgICBjb25zdCBqb2JtYXA6IFJlY29yZDxzdHJpbmcsIGdpdGh1Yi5Kb2I+ID0ge307XG4gICAgZm9yIChjb25zdCBqb2Igb2Ygam9icykge1xuICAgICAgaWYgKGpvYi5pZCBpbiBqb2JtYXApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBkdXBsaWNhdGUgam9iIGlkICR7am9iLmlkfWApO1xuICAgICAgfVxuICAgICAgam9ibWFwW2pvYi5pZF0gPSBzbmFrZUNhc2VLZXlzKGpvYi5kZWZpbml0aW9uKTtcbiAgICB9XG5cbiAgICAvLyBVcGRhdGUgam9icyB3aXRoIGxhdGUtYm91bmQgb3V0cHV0IHJlcXVlc3RzXG4gICAgdGhpcy5pbnNlcnRKb2JPdXRwdXRzKGpvYm1hcCk7XG5cbiAgICBjb25zdCB3b3JrZmxvdyA9IHtcbiAgICAgIG5hbWU6IHRoaXMud29ya2Zsb3dOYW1lLFxuICAgICAgb246IHNuYWtlQ2FzZUtleXModGhpcy53b3JrZmxvd1RyaWdnZXJzLCAnXycpLFxuICAgICAgam9iczogam9ibWFwLFxuICAgIH07XG5cbiAgICAvLyB3cml0ZSBhcyBhIHlhbWwgZmlsZVxuICAgIHRoaXMud29ya2Zsb3dGaWxlLnVwZGF0ZSh3b3JrZmxvdyk7XG5cbiAgICAvLyBjcmVhdGUgZGlyZWN0b3J5IGlmIGl0IGRvZXMgbm90IGV4aXN0XG4gICAgbWtkaXJTeW5jKHBhdGguZGlybmFtZSh0aGlzLndvcmtmbG93UGF0aCksIHsgcmVjdXJzaXZlOiB0cnVlIH0pO1xuXG4gICAgLy8gR0lUSFVCX1dPUktGTE9XIGlzIHNldCB3aGVuIEdpdEh1YiBBY3Rpb25zIGlzIHJ1bm5pbmcgdGhlIHdvcmtmbG93LlxuICAgIC8vIHNlZTogaHR0cHM6Ly9kb2NzLmdpdGh1Yi5jb20vZW4vYWN0aW9ucy9sZWFybi1naXRodWItYWN0aW9ucy9lbnZpcm9ubWVudC12YXJpYWJsZXMjZGVmYXVsdC1lbnZpcm9ubWVudC12YXJpYWJsZXNcbiAgICBjb25zdCBkaWZmUHJvdGVjdGlvbiA9IHRoaXMubm9kZS50cnlHZXRDb250ZXh0KCdjZGstcGlwZWxpbmVzLWdpdGh1YjpkaWZmUHJvdGVjdGlvbicpID8/IHRydWU7XG4gICAgaWYgKGRpZmZQcm90ZWN0aW9uICYmIHByb2Nlc3MuZW52LkdJVEhVQl9XT1JLRkxPVyA9PT0gdGhpcy53b3JrZmxvd05hbWUpIHtcbiAgICAgIC8vIGNoZWNrIGlmIHdvcmtmbG93IGZpbGUgaGFzIGNoYW5nZWRcbiAgICAgIGlmICghZXhpc3RzU3luYyh0aGlzLndvcmtmbG93UGF0aCkgfHwgdGhpcy53b3JrZmxvd0ZpbGUudG9ZYW1sKCkgIT09IHJlYWRGaWxlU3luYyh0aGlzLndvcmtmbG93UGF0aCwgJ3V0ZjgnKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFBsZWFzZSBjb21taXQgdGhlIHVwZGF0ZWQgd29ya2Zsb3cgZmlsZSAke3BhdGgucmVsYXRpdmUoX19kaXJuYW1lLCB0aGlzLndvcmtmbG93UGF0aCl9IHdoZW4geW91IGNoYW5nZSB5b3VyIHBpcGVsaW5lIGRlZmluaXRpb24uYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdGhpcy53b3JrZmxvd0ZpbGUud3JpdGVGaWxlKCk7XG4gIH1cblxuICBwcml2YXRlIGluc2VydEpvYk91dHB1dHMoam9ibWFwOiBSZWNvcmQ8c3RyaW5nLCBnaXRodWIuSm9iPikge1xuICAgIGZvciAoY29uc3QgW2pvYklkLCBqb2JPdXRwdXRzXSBvZiBPYmplY3QuZW50cmllcyh0aGlzLmpvYk91dHB1dHMpKSB7XG4gICAgICBqb2JtYXBbam9iSWRdID0ge1xuICAgICAgICAuLi5qb2JtYXBbam9iSWRdLFxuICAgICAgICBvdXRwdXRzOiB7XG4gICAgICAgICAgLi4uam9ibWFwW2pvYklkXS5vdXRwdXRzLFxuICAgICAgICAgIC4uLnRoaXMucmVuZGVySm9iT3V0cHV0cyhqb2JPdXRwdXRzKSxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJKb2JPdXRwdXRzKG91dHB1dHM6IGdpdGh1Yi5Kb2JTdGVwT3V0cHV0W10pIHtcbiAgICBjb25zdCByZW5kZXJlZE91dHB1dHM6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4gPSB7fTtcbiAgICBmb3IgKGNvbnN0IG91dHB1dCBvZiBvdXRwdXRzKSB7XG4gICAgICByZW5kZXJlZE91dHB1dHNbb3V0cHV0Lm91dHB1dE5hbWVdID0gYFxcJHt7IHN0ZXBzLiR7b3V0cHV0LnN0ZXBJZH0ub3V0cHV0cy4ke291dHB1dC5vdXRwdXROYW1lfSB9fWA7XG4gICAgfVxuICAgIHJldHVybiByZW5kZXJlZE91dHB1dHM7XG4gIH1cblxuICAvKipcbiAgICogTWFrZSBhbiBhY3Rpb24gZnJvbSB0aGUgZ2l2ZW4gbm9kZSBhbmQvb3Igc3RlcFxuICAgKi9cbiAgcHJpdmF0ZSBqb2JGb3JOb2RlKG5vZGU6IEFHcmFwaE5vZGUsIG9wdGlvbnM6IENvbnRleHQpOiBKb2IgfCB1bmRlZmluZWQge1xuICAgIHN3aXRjaCAobm9kZS5kYXRhPy50eXBlKSB7XG4gICAgICAvLyBOb3RoaW5nIGZvciB0aGVzZSwgdGhleSBhcmUgZ3JvdXBpbmdzIChzaG91bGRuJ3QgZXZlbiBoYXZlIHBvcHBlZCB1cCBoZXJlKVxuICAgICAgY2FzZSAnZ3JvdXAnOlxuICAgICAgY2FzZSAnc3RhY2stZ3JvdXAnOlxuICAgICAgY2FzZSB1bmRlZmluZWQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgam9iRm9yTm9kZTogZGlkIG5vdCBleHBlY3QgdG8gZ2V0IGdyb3VwIG5vZGVzOiAke25vZGUuZGF0YT8udHlwZX1gKTtcblxuICAgICAgY2FzZSAnc2VsZi11cGRhdGUnOlxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0dpdEh1YiBXb3JrZmxvd3MgZG9lcyBub3Qgc3VwcG9ydCBzZWxmIG11dGF0aW9uJyk7XG5cbiAgICAgIGNhc2UgJ3B1Ymxpc2gtYXNzZXRzJzpcbiAgICAgICAgcmV0dXJuIHRoaXMuam9iRm9yQXNzZXRQdWJsaXNoKG5vZGUsIG5vZGUuZGF0YS5hc3NldHMsIG9wdGlvbnMpO1xuXG4gICAgICBjYXNlICdwcmVwYXJlJzpcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdcInByZXBhcmVcIiBpcyBub3Qgc3VwcG9ydGVkIGJ5IEdpdEh1YiBXb3JmbG93cycpO1xuXG4gICAgICBjYXNlICdleGVjdXRlJzpcbiAgICAgICAgcmV0dXJuIHRoaXMuam9iRm9yRGVwbG95KG5vZGUsIG5vZGUuZGF0YS5zdGFjaywgbm9kZS5kYXRhLmNhcHR1cmVPdXRwdXRzKTtcblxuICAgICAgY2FzZSAnc3RlcCc6XG4gICAgICAgIGlmIChub2RlLmRhdGEuaXNCdWlsZFN0ZXApIHtcbiAgICAgICAgICByZXR1cm4gdGhpcy5qb2JGb3JCdWlsZFN0ZXAobm9kZSwgbm9kZS5kYXRhLnN0ZXApO1xuICAgICAgICB9IGVsc2UgaWYgKG5vZGUuZGF0YS5zdGVwIGluc3RhbmNlb2YgU2hlbGxTdGVwKSB7XG4gICAgICAgICAgcmV0dXJuIHRoaXMuam9iRm9yU2NyaXB0U3RlcChub2RlLCBub2RlLmRhdGEuc3RlcCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGB1bnN1cHBvcnRlZCBzdGVwIHR5cGU6ICR7bm9kZS5kYXRhLnN0ZXAuY29uc3RydWN0b3IubmFtZX1gKTtcbiAgICAgICAgfVxuXG4gICAgICBkZWZhdWx0OlxuICAgICAgICAvLyBUaGUgJ2FzIGFueScgaXMgdGVtcG9yYXJ5LCB1bnRpbCB0aGUgY2hhbmdlIHVwc3RyZWFtIHJvbGxzIG91dFxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEdpdEh1YldvcmZrbG93IGRvZXMgbm90IHN1cHBvcnQgZ3JhcGggbm9kZXMgb2YgdHlwZSAnJHsobm9kZS5kYXRhIGFzIGFueSk/LnR5cGV9Jy4gWW91IGFyZSBwcm9iYWJseSB1c2luZyBhIGZlYXR1cmUgdGhpcyBDREsgUGlwZWxpbmVzIGltcGxlbWVudGF0aW9uIGRvZXMgbm90IHN1cHBvcnQuYCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBqb2JGb3JBc3NldFB1Ymxpc2gobm9kZTogQUdyYXBoTm9kZSwgYXNzZXRzOiBTdGFja0Fzc2V0W10sIG9wdGlvbnM6IENvbnRleHQpOiBKb2Ige1xuICAgIGlmIChhc3NldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Fzc2V0IFB1Ymxpc2ggc3RlcCBtdXN0IGhhdmUgYXQgbGVhc3QgMSBhc3NldCcpO1xuICAgIH1cblxuICAgIGNvbnN0IGluc3RhbGxTdWZmaXggPSB0aGlzLmNka0NsaVZlcnNpb24gPyBgQCR7dGhpcy5jZGtDbGlWZXJzaW9ufWAgOiAnJztcbiAgICBjb25zdCBjZGtvdXREaXIgPSBvcHRpb25zLmFzc2VtYmx5RGlyO1xuICAgIGNvbnN0IGpvYklkID0gbm9kZS51bmlxdWVJZDtcbiAgICBjb25zdCBhc3NldElkID0gYXNzZXRzWzBdLmFzc2V0SWQ7XG5cbiAgICAvLyBjaGVjayBpZiBhc3NldCBpcyBkb2NrZXIgYXNzZXQgYW5kIGlmIHdlIGhhdmUgZG9ja2VyIGNyZWRlbnRpYWxzXG4gICAgY29uc3QgZG9ja2VyTG9naW5TdGVwczogZ2l0aHViLkpvYlN0ZXBbXSA9IFtdO1xuICAgIGlmIChub2RlLnVuaXF1ZUlkLmluY2x1ZGVzKCdEb2NrZXJBc3NldCcpICYmIHRoaXMuZG9ja2VyQ3JlZGVudGlhbHMubGVuZ3RoID4gMCkge1xuICAgICAgZm9yIChjb25zdCBjcmVkcyBvZiB0aGlzLmRvY2tlckNyZWRlbnRpYWxzKSB7XG4gICAgICAgIGRvY2tlckxvZ2luU3RlcHMucHVzaCguLi50aGlzLnN0ZXBzVG9Db25maWd1cmVEb2NrZXIoY3JlZHMpKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBjcmVhdGUgb25lIGZpbGUgYW5kIG1ha2Ugb25lIHN0ZXBcbiAgICBjb25zdCByZWxhdGl2ZVRvQXNzZW1ibHkgPSAocDogc3RyaW5nKSA9PiBwYXRoLnBvc2l4LmpvaW4oY2Rrb3V0RGlyLCBwYXRoLnJlbGF0aXZlKHBhdGgucmVzb2x2ZShjZGtvdXREaXIpLCBwKSk7XG4gICAgY29uc3QgZmlsZUNvbnRlbnRzOiBzdHJpbmdbXSA9IFsnc2V0IC14J10uY29uY2F0KGFzc2V0cy5tYXAoKGFzc2V0KSA9PiB7XG4gICAgICByZXR1cm4gYG5weCBjZGstYXNzZXRzIC0tcGF0aCBcIiR7cmVsYXRpdmVUb0Fzc2VtYmx5KGFzc2V0LmFzc2V0TWFuaWZlc3RQYXRoKX1cIiAtLXZlcmJvc2UgcHVibGlzaCBcIiR7YXNzZXQuYXNzZXRTZWxlY3Rvcn1cImA7XG4gICAgfSkpO1xuXG4gICAgLy8gd2UgbmVlZCB0aGUgam9iSWQgdG8gcmVmZXJlbmNlIHRoZSBvdXRwdXRzIGxhdGVyXG4gICAgdGhpcy5hc3NldEhhc2hNYXBbYXNzZXRJZF0gPSBqb2JJZDtcbiAgICBmaWxlQ29udGVudHMucHVzaChgZWNobyAnOjpzZXQtb3V0cHV0IG5hbWU9JHtBU1NFVF9IQVNIX05BTUV9Ojoke2Fzc2V0SWR9J2ApO1xuXG4gICAgY29uc3QgcHVibGlzaFN0ZXBGaWxlID0gcGF0aC5qb2luKGNka291dERpciwgYHB1Ymxpc2gtJHtqb2JJZH0tc3RlcC5zaGApO1xuICAgIG1rZGlyU3luYyhwYXRoLmRpcm5hbWUocHVibGlzaFN0ZXBGaWxlKSwgeyByZWN1cnNpdmU6IHRydWUgfSk7XG4gICAgd3JpdGVGaWxlU3luYyhwdWJsaXNoU3RlcEZpbGUsIGZpbGVDb250ZW50cy5qb2luKCdcXG4nKSwgeyBlbmNvZGluZzogJ3V0Zi04JyB9KTtcblxuICAgIGNvbnN0IHB1Ymxpc2hTdGVwOiBnaXRodWIuSm9iU3RlcCA9IHtcbiAgICAgIGlkOiAnUHVibGlzaCcsXG4gICAgICBuYW1lOiBgUHVibGlzaCAke2pvYklkfWAsXG4gICAgICBydW46IGAvYmluL2Jhc2ggLi9jZGsub3V0LyR7cGF0aC5yZWxhdGl2ZShjZGtvdXREaXIsIHB1Ymxpc2hTdGVwRmlsZSl9YCxcbiAgICB9O1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiBqb2JJZCxcbiAgICAgIGRlZmluaXRpb246IHtcbiAgICAgICAgbmFtZTogYFB1Ymxpc2ggQXNzZXRzICR7am9iSWR9YCxcbiAgICAgICAgLi4udGhpcy5qb2JTZXR0aW5ncyxcbiAgICAgICAgbmVlZHM6IHRoaXMucmVuZGVyRGVwZW5kZW5jaWVzKG5vZGUpLFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiBnaXRodWIuSm9iUGVybWlzc2lvbi5SRUFELFxuICAgICAgICAgIGlkVG9rZW46IHRoaXMudXNlR2l0SHViQWN0aW9uUm9sZSA/IGdpdGh1Yi5Kb2JQZXJtaXNzaW9uLldSSVRFIDogZ2l0aHViLkpvYlBlcm1pc3Npb24uTk9ORSxcbiAgICAgICAgfSxcbiAgICAgICAgcnVuc09uOiB0aGlzLnJ1bm5lci5ydW5zT24sXG4gICAgICAgIG91dHB1dHM6IHtcbiAgICAgICAgICBbQVNTRVRfSEFTSF9OQU1FXTogYFxcJHt7IHN0ZXBzLlB1Ymxpc2gub3V0cHV0cy4ke0FTU0VUX0hBU0hfTkFNRX0gfX1gLFxuICAgICAgICB9LFxuICAgICAgICBzdGVwczogW1xuICAgICAgICAgIC4uLnRoaXMuc3RlcHNUb0Rvd25sb2FkQXNzZW1ibHkoY2Rrb3V0RGlyKSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBuYW1lOiAnSW5zdGFsbCcsXG4gICAgICAgICAgICBydW46IGBucG0gaW5zdGFsbCAtLW5vLXNhdmUgY2RrLWFzc2V0cyR7aW5zdGFsbFN1ZmZpeH1gLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgLi4udGhpcy5zdGVwc1RvQ29uZmlndXJlQXdzKHRoaXMudXNlR2l0SHViQWN0aW9uUm9sZSwgeyByZWdpb246IHRoaXMucHVibGlzaEFzc2V0c0F1dGhSZWdpb24gfSksXG4gICAgICAgICAgLi4uZG9ja2VyTG9naW5TdGVwcyxcbiAgICAgICAgICBwdWJsaXNoU3RlcCxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgam9iRm9yRGVwbG95KG5vZGU6IEFHcmFwaE5vZGUsIHN0YWNrOiBTdGFja0RlcGxveW1lbnQsIF9jYXB0dXJlT3V0cHV0czogYm9vbGVhbik6IEpvYiB7XG4gICAgY29uc3QgcmVnaW9uID0gc3RhY2sucmVnaW9uO1xuICAgIGNvbnN0IGFjY291bnQgPSBzdGFjay5hY2NvdW50O1xuICAgIGlmICghcmVnaW9uIHx8ICFhY2NvdW50KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wiYWNjb3VudFwiIGFuZCBcInJlZ2lvblwiIGFyZSByZXF1aXJlZCcpO1xuICAgIH1cblxuICAgIGlmICghc3RhY2sudGVtcGxhdGVVcmwpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgdW5hYmxlIHRvIGRldGVybWluZSB0ZW1wbGF0ZSBVUkwgZm9yIHN0YWNrICR7c3RhY2suc3RhY2tBcnRpZmFjdElkfWApO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc29sdmUgPSAoczogc3RyaW5nKTogc3RyaW5nID0+IHtcbiAgICAgIHJldHVybiBFbnZpcm9ubWVudFBsYWNlaG9sZGVycy5yZXBsYWNlKHMsIHtcbiAgICAgICAgYWNjb3VudElkOiBhY2NvdW50LFxuICAgICAgICByZWdpb246IHJlZ2lvbixcbiAgICAgICAgcGFydGl0aW9uOiAnYXdzJyxcbiAgICAgIH0pO1xuICAgIH07XG5cbiAgICBjb25zdCByZXBsYWNlQXNzZXRIYXNoID0gKHRlbXBsYXRlOiBzdHJpbmcpID0+IHtcbiAgICAgIGNvbnN0IGhhc2ggPSBwYXRoLnBhcnNlKHRlbXBsYXRlLnNwbGl0KCcvJykucG9wKCkgPz8gJycpLm5hbWU7XG4gICAgICBpZiAodGhpcy5hc3NldEhhc2hNYXBbaGFzaF0gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRlbXBsYXRlIGFzc2V0IGhhc2ggJHtoYXNofSBub3QgZm91bmQuYCk7XG4gICAgICB9XG4gICAgICByZXR1cm4gdGVtcGxhdGUucmVwbGFjZShoYXNoLCBgXFwke3sgbmVlZHMuJHt0aGlzLmFzc2V0SGFzaE1hcFtoYXNoXX0ub3V0cHV0cy4ke0FTU0VUX0hBU0hfTkFNRX0gfX1gKTtcbiAgICB9O1xuXG4gICAgY29uc3QgcGFyYW1zOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge1xuICAgICAgJ25hbWUnOiBzdGFjay5zdGFja05hbWUsXG4gICAgICAndGVtcGxhdGUnOiByZXBsYWNlQXNzZXRIYXNoKHJlc29sdmUoc3RhY2sudGVtcGxhdGVVcmwpKSxcbiAgICAgICduby1mYWlsLW9uLWVtcHR5LWNoYW5nZXNldCc6ICcxJyxcbiAgICB9O1xuXG4gICAgY29uc3QgY2FwYWJpbGl0aWVzID0gdGhpcy5zdGFja1Byb3BlcnRpZXNbc3RhY2suc3RhY2tBcnRpZmFjdElkXT8uY2FwYWJpbGl0aWVzO1xuICAgIGlmIChjYXBhYmlsaXRpZXMpIHtcbiAgICAgIHBhcmFtcy5jYXBhYmlsaXRpZXMgPSBBcnJheShjYXBhYmlsaXRpZXMpLmpvaW4oJywnKTtcbiAgICB9XG5cbiAgICBpZiAoc3RhY2suZXhlY3V0aW9uUm9sZUFybikge1xuICAgICAgcGFyYW1zWydyb2xlLWFybiddID0gcmVzb2x2ZShzdGFjay5leGVjdXRpb25Sb2xlQXJuKTtcbiAgICB9XG4gICAgY29uc3QgYXNzdW1lUm9sZUFybiA9IHN0YWNrLmFzc3VtZVJvbGVBcm4gPyByZXNvbHZlKHN0YWNrLmFzc3VtZVJvbGVBcm4pIDogdW5kZWZpbmVkO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiBub2RlLnVuaXF1ZUlkLFxuICAgICAgZGVmaW5pdGlvbjoge1xuICAgICAgICBuYW1lOiBgRGVwbG95ICR7c3RhY2suc3RhY2tBcnRpZmFjdElkfWAsXG4gICAgICAgIC4uLnRoaXMuam9iU2V0dGluZ3MsXG4gICAgICAgIC4uLnRoaXMuc3RhY2tQcm9wZXJ0aWVzW3N0YWNrLnN0YWNrQXJ0aWZhY3RJZF0/LnNldHRpbmdzLFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiBnaXRodWIuSm9iUGVybWlzc2lvbi5SRUFELFxuICAgICAgICAgIGlkVG9rZW46IHRoaXMudXNlR2l0SHViQWN0aW9uUm9sZSA/IGdpdGh1Yi5Kb2JQZXJtaXNzaW9uLldSSVRFIDogZ2l0aHViLkpvYlBlcm1pc3Npb24uTk9ORSxcbiAgICAgICAgfSxcbiAgICAgICAgLi4uKHRoaXMuc3RhY2tQcm9wZXJ0aWVzW3N0YWNrLnN0YWNrQXJ0aWZhY3RJZF0/LmVudmlyb25tZW50ID8ge1xuICAgICAgICAgIGVudmlyb25tZW50OiB0aGlzLnN0YWNrUHJvcGVydGllc1tzdGFjay5zdGFja0FydGlmYWN0SWRdLmVudmlyb25tZW50LFxuICAgICAgICB9IDoge30pLFxuICAgICAgICBuZWVkczogdGhpcy5yZW5kZXJEZXBlbmRlbmNpZXMobm9kZSksXG4gICAgICAgIHJ1bnNPbjogdGhpcy5ydW5uZXIucnVuc09uLFxuICAgICAgICBzdGVwczogW1xuICAgICAgICAgIC4uLnRoaXMuc3RlcHNUb0NvbmZpZ3VyZUF3cyh0aGlzLnVzZUdpdEh1YkFjdGlvblJvbGUsIHsgcmVnaW9uLCBhc3N1bWVSb2xlQXJuIH0pLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGlkOiAnRGVwbG95JyxcbiAgICAgICAgICAgIHVzZXM6ICdhd3MtYWN0aW9ucy9hd3MtY2xvdWRmb3JtYXRpb24tZ2l0aHViLWRlcGxveUB2MScsXG4gICAgICAgICAgICB3aXRoOiBwYXJhbXMsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgam9iRm9yQnVpbGRTdGVwKG5vZGU6IEFHcmFwaE5vZGUsIHN0ZXA6IFN0ZXApOiBKb2Ige1xuICAgIGlmICghKHN0ZXAgaW5zdGFuY2VvZiBTaGVsbFN0ZXApKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3N5bnRoU3RlcCBtdXN0IGJlIGEgU2NyaXB0U3RlcCcpO1xuICAgIH1cblxuICAgIGlmIChzdGVwLmlucHV0cy5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3N5bnRoU3RlcCBjYW5ub3QgaGF2ZSBpbnB1dHMnKTtcbiAgICB9XG5cbiAgICBpZiAoc3RlcC5vdXRwdXRzLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignc3ludGhTdGVwIG11c3QgaGF2ZSBhIHNpbmdsZSBvdXRwdXQnKTtcbiAgICB9XG5cbiAgICBpZiAoIXN0ZXAucHJpbWFyeU91dHB1dCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdzeW50aFN0ZXAgcmVxdWlyZXMgYSBwcmltYXJ5T3V0cHV0IHdoaWNoIGNvbnRhaW5zIGNkay5vdXQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBjZGtPdXQgPSBzdGVwLm91dHB1dHNbMF07XG5cbiAgICBjb25zdCBpbnN0YWxsU3RlcHMgPSBzdGVwLmluc3RhbGxDb21tYW5kcy5sZW5ndGggPiAwID8gW3tcbiAgICAgIG5hbWU6ICdJbnN0YWxsJyxcbiAgICAgIHJ1bjogc3RlcC5pbnN0YWxsQ29tbWFuZHMuam9pbignXFxuJyksXG4gICAgfV0gOiBbXTtcblxuICAgIHJldHVybiB7XG4gICAgICBpZDogbm9kZS51bmlxdWVJZCxcbiAgICAgIGRlZmluaXRpb246IHtcbiAgICAgICAgbmFtZTogJ1N5bnRoZXNpemUnLFxuICAgICAgICAuLi50aGlzLmpvYlNldHRpbmdzLFxuICAgICAgICBwZXJtaXNzaW9uczoge1xuICAgICAgICAgIGNvbnRlbnRzOiBnaXRodWIuSm9iUGVybWlzc2lvbi5SRUFELFxuICAgICAgICAgIC8vIFRoZSBTeW50aGVzaXplIGpvYiBkb2VzIG5vdCB1c2UgdGhlIEdpdEh1YiBBY3Rpb24gUm9sZSBvbiBpdHMgb3duLCBidXQgaXQncyBwb3NzaWJsZVxuICAgICAgICAgIC8vIHRoYXQgaXQgaXMgYmVpbmcgdXNlZCBpbiB0aGUgcHJlQnVpbGRTdGVwcy5cbiAgICAgICAgICBpZFRva2VuOiB0aGlzLnVzZUdpdEh1YkFjdGlvblJvbGUgPyBnaXRodWIuSm9iUGVybWlzc2lvbi5XUklURSA6IGdpdGh1Yi5Kb2JQZXJtaXNzaW9uLk5PTkUsXG4gICAgICAgIH0sXG4gICAgICAgIHJ1bnNPbjogdGhpcy5ydW5uZXIucnVuc09uLFxuICAgICAgICBuZWVkczogdGhpcy5yZW5kZXJEZXBlbmRlbmNpZXMobm9kZSksXG4gICAgICAgIGVudjogc3RlcC5lbnYsXG4gICAgICAgIGNvbnRhaW5lcjogdGhpcy5idWlsZENvbnRhaW5lcixcbiAgICAgICAgc3RlcHM6IFtcbiAgICAgICAgICAuLi50aGlzLnN0ZXBzVG9DaGVja291dCgpLFxuICAgICAgICAgIC4uLnRoaXMucHJlQnVpbGRTdGVwcyxcbiAgICAgICAgICAuLi5pbnN0YWxsU3RlcHMsXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgIHJ1bjogc3RlcC5jb21tYW5kcy5qb2luKCdcXG4nKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIC4uLnRoaXMucG9zdEJ1aWxkU3RlcHMsXG4gICAgICAgICAgLi4udGhpcy5zdGVwc1RvVXBsb2FkQXNzZW1ibHkoY2RrT3V0LmRpcmVjdG9yeSksXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogU2VhcmNoZXMgZm9yIHRoZSBzdGFjayB0aGF0IHByb2R1Y2VkIHRoZSBvdXRwdXQgdmlhIHRoZSBjdXJyZW50XG4gICAqIGpvYidzIGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogVGhpcyBmdW5jdGlvbiBzaG91bGQgYWx3YXlzIGZpbmQgYSBzdGFjaywgc2luY2UgaXQgaXMgZ3VhcmFudGVlZFxuICAgKiB0aGF0IGEgQ2ZuT3V0cHV0IGNvbWVzIGZyb20gYSByZWZlcmVuY2VkIHN0YWNrLlxuICAgKi9cbiAgcHJpdmF0ZSBmaW5kU3RhY2tPZk91dHB1dChyZWY6IFN0YWNrT3V0cHV0UmVmZXJlbmNlLCBub2RlOiBBR3JhcGhOb2RlKSB7XG4gICAgZm9yIChjb25zdCBkZXAgb2Ygbm9kZS5hbGxEZXBzKSB7XG4gICAgICBpZiAoZGVwLmRhdGE/LnR5cGUgPT09ICdleGVjdXRlJyAmJiByZWYuaXNQcm9kdWNlZEJ5KGRlcC5kYXRhLnN0YWNrKSkge1xuICAgICAgICByZXR1cm4gZGVwLnVuaXF1ZUlkO1xuICAgICAgfVxuICAgIH1cbiAgICAvLyBTaG91bGQgbmV2ZXIgaGFwcGVuXG4gICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgb3V0cHV0ICR7cmVmLm91dHB1dE5hbWV9IGlzIG5vdCByZWZlcmVuY2VkIGJ5IGFueSBvZiB0aGUgZGVwZW5kZW50IHN0YWNrcyFgKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkSm9iT3V0cHV0KGpvYklkOiBzdHJpbmcsIG91dHB1dDogZ2l0aHViLkpvYlN0ZXBPdXRwdXQpIHtcbiAgICBpZiAodGhpcy5qb2JPdXRwdXRzW2pvYklkXSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLmpvYk91dHB1dHNbam9iSWRdID0gW291dHB1dF07XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuam9iT3V0cHV0c1tqb2JJZF0ucHVzaChvdXRwdXQpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgam9iRm9yU2NyaXB0U3RlcChub2RlOiBBR3JhcGhOb2RlLCBzdGVwOiBTaGVsbFN0ZXApOiBKb2Ige1xuICAgIGNvbnN0IGVudlZhcmlhYmxlczogUmVjb3JkPHN0cmluZywgc3RyaW5nPiA9IHt9O1xuICAgIGZvciAoY29uc3QgW2Vudk5hbWUsIHJlZl0gb2YgT2JqZWN0LmVudHJpZXMoc3RlcC5lbnZGcm9tQ2ZuT3V0cHV0cykpIHtcbiAgICAgIGNvbnN0IGpvYklkID0gdGhpcy5maW5kU3RhY2tPZk91dHB1dChyZWYsIG5vZGUpO1xuICAgICAgdGhpcy5hZGRKb2JPdXRwdXQoam9iSWQsIHtcbiAgICAgICAgb3V0cHV0TmFtZTogcmVmLm91dHB1dE5hbWUsXG4gICAgICAgIHN0ZXBJZDogJ0RlcGxveScsXG4gICAgICB9KTtcbiAgICAgIGVudlZhcmlhYmxlc1tlbnZOYW1lXSA9IGBcXCR7eyBuZWVkcy4ke2pvYklkfS5vdXRwdXRzLiR7cmVmLm91dHB1dE5hbWV9IH19YDtcbiAgICB9XG5cbiAgICBjb25zdCBkb3dubG9hZElucHV0cyA9IG5ldyBBcnJheTxnaXRodWIuSm9iU3RlcD4oKTtcbiAgICBjb25zdCB1cGxvYWRPdXRwdXRzID0gbmV3IEFycmF5PGdpdGh1Yi5Kb2JTdGVwPigpO1xuXG4gICAgZm9yIChjb25zdCBpbnB1dCBvZiBzdGVwLmlucHV0cykge1xuICAgICAgZG93bmxvYWRJbnB1dHMucHVzaCh7XG4gICAgICAgIHVzZXM6ICdhY3Rpb25zL2Rvd25sb2FkLWFydGlmYWN0QHYyJyxcbiAgICAgICAgd2l0aDoge1xuICAgICAgICAgIG5hbWU6IGlucHV0LmZpbGVTZXQuaWQsXG4gICAgICAgICAgcGF0aDogaW5wdXQuZGlyZWN0b3J5LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBvdXRwdXQgb2Ygc3RlcC5vdXRwdXRzKSB7XG4gICAgICB1cGxvYWRPdXRwdXRzLnB1c2goe1xuICAgICAgICB1c2VzOiAnYWN0aW9ucy91cGxvYWQtYXJ0aWZhY3RAdjIuMS4xJyxcbiAgICAgICAgd2l0aDoge1xuICAgICAgICAgIG5hbWU6IG91dHB1dC5maWxlU2V0LmlkLFxuICAgICAgICAgIHBhdGg6IG91dHB1dC5kaXJlY3RvcnksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBpbnN0YWxsU3RlcHMgPSBzdGVwLmluc3RhbGxDb21tYW5kcy5sZW5ndGggPiAwID8gW3tcbiAgICAgIG5hbWU6ICdJbnN0YWxsJyxcbiAgICAgIHJ1bjogc3RlcC5pbnN0YWxsQ29tbWFuZHMuam9pbignXFxuJyksXG4gICAgfV0gOiBbXTtcblxuICAgIHJldHVybiB7XG4gICAgICBpZDogbm9kZS51bmlxdWVJZCxcbiAgICAgIGRlZmluaXRpb246IHtcbiAgICAgICAgbmFtZTogc3RlcC5pZCxcbiAgICAgICAgLi4udGhpcy5qb2JTZXR0aW5ncyxcbiAgICAgICAgcGVybWlzc2lvbnM6IHtcbiAgICAgICAgICBjb250ZW50czogZ2l0aHViLkpvYlBlcm1pc3Npb24uUkVBRCxcbiAgICAgICAgfSxcbiAgICAgICAgcnVuc09uOiB0aGlzLnJ1bm5lci5ydW5zT24sXG4gICAgICAgIG5lZWRzOiB0aGlzLnJlbmRlckRlcGVuZGVuY2llcyhub2RlKSxcbiAgICAgICAgZW52OiB7XG4gICAgICAgICAgLi4uc3RlcC5lbnYsXG4gICAgICAgICAgLi4uZW52VmFyaWFibGVzLFxuICAgICAgICB9LFxuICAgICAgICBzdGVwczogW1xuICAgICAgICAgIC4uLmRvd25sb2FkSW5wdXRzLFxuICAgICAgICAgIC4uLmluc3RhbGxTdGVwcyxcbiAgICAgICAgICB7IHJ1bjogc3RlcC5jb21tYW5kcy5qb2luKCdcXG4nKSB9LFxuICAgICAgICAgIC4uLnVwbG9hZE91dHB1dHMsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHN0ZXBzVG9Db25maWd1cmVBd3Mob3BlbklkOiBib29sZWFuLCB7IHJlZ2lvbiwgYXNzdW1lUm9sZUFybiB9OiB7IHJlZ2lvbjogc3RyaW5nOyBhc3N1bWVSb2xlQXJuPzogc3RyaW5nIH0pOiBnaXRodWIuSm9iU3RlcFtdIHtcbiAgICBmdW5jdGlvbiBnZXREZXBsb3lSb2xlKGFybjogc3RyaW5nKSB7XG4gICAgICByZXR1cm4gYXJuLnJlcGxhY2UoJ2Nmbi1leGVjJywgJ2RlcGxveScpO1xuICAgIH1cblxuICAgIGxldCBzdGVwczogZ2l0aHViLkpvYlN0ZXBbXSA9IFtdO1xuXG4gICAgaWYgKG9wZW5JZCkge1xuICAgICAgc3RlcHMucHVzaChhd3NDcmVkZW50aWFsU3RlcCgnQXV0aGVudGljYXRlIFZpYSBPSURDIFJvbGUnLCB7XG4gICAgICAgIHJlZ2lvbixcbiAgICAgICAgZ2l0SHViQWN0aW9uUm9sZUFybjogdGhpcy5naXRIdWJBY3Rpb25Sb2xlQXJuLFxuICAgICAgfSkpO1xuXG4gICAgICBpZiAoYXNzdW1lUm9sZUFybikge1xuICAgICAgICAvLyBSZXN1bHQgb2YgaW5pdGlhbCBjcmVkZW50aWFscyB3aXRoIEdpdEh1YiBBY3Rpb24gcm9sZSBhcmUgdGhlc2UgZW52aXJvbm1lbnQgdmFyaWFibGVzXG4gICAgICAgIHN0ZXBzLnB1c2goYXdzQ3JlZGVudGlhbFN0ZXAoJ0Fzc3VtZSBDREsgRGVwbG95IFJvbGUnLCB7XG4gICAgICAgICAgcmVnaW9uLFxuICAgICAgICAgIGFjY2Vzc0tleUlkOiAnJHt7IGVudi5BV1NfQUNDRVNTX0tFWV9JRCB9fScsXG4gICAgICAgICAgc2VjcmV0QWNjZXNzS2V5OiAnJHt7IGVudi5BV1NfU0VDUkVUX0FDQ0VTU19LRVkgfX0nLFxuICAgICAgICAgIHNlc3Npb25Ub2tlbjogJyR7eyBlbnYuQVdTX1NFU1NJT05fVE9LRU4gfX0nLFxuICAgICAgICAgIHJvbGVUb0Fzc3VtZTogZ2V0RGVwbG95Um9sZShhc3N1bWVSb2xlQXJuKSxcbiAgICAgICAgfSkpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBzdGVwcy5wdXNoKGF3c0NyZWRlbnRpYWxTdGVwKCdBdXRoZW50aWNhdGUgVmlhIEdpdEh1YiBTZWNyZXRzJywge1xuICAgICAgICByZWdpb24sXG4gICAgICAgIGFjY2Vzc0tleUlkOiBgXFwke3sgc2VjcmV0cy4ke3RoaXMuYXdzQ3JlZGVudGlhbHMuYWNjZXNzS2V5SWR9IH19YCxcbiAgICAgICAgc2VjcmV0QWNjZXNzS2V5OiBgXFwke3sgc2VjcmV0cy4ke3RoaXMuYXdzQ3JlZGVudGlhbHMuc2VjcmV0QWNjZXNzS2V5fSB9fWAsXG4gICAgICAgIC4uLih0aGlzLmF3c0NyZWRlbnRpYWxzLnNlc3Npb25Ub2tlbiA/IHtcbiAgICAgICAgICBzZXNzaW9uVG9rZW46IGBcXCR7eyBzZWNyZXRzLiR7dGhpcy5hd3NDcmVkZW50aWFscy5zZXNzaW9uVG9rZW59IH19YCxcbiAgICAgICAgfSA6IHVuZGVmaW5lZCksXG4gICAgICAgIHJvbGVUb0Fzc3VtZTogYXNzdW1lUm9sZUFybixcbiAgICAgIH0pKTtcbiAgICB9XG5cbiAgICByZXR1cm4gc3RlcHM7XG4gIH1cblxuICBwcml2YXRlIHN0ZXBzVG9Db25maWd1cmVEb2NrZXIoZG9ja2VyQ3JlZGVudGlhbDogRG9ja2VyQ3JlZGVudGlhbCk6IGdpdGh1Yi5Kb2JTdGVwW10ge1xuICAgIGxldCBwYXJhbXM6IFJlY29yZDxzdHJpbmcsIGFueT47XG5cbiAgICBpZiAoZG9ja2VyQ3JlZGVudGlhbC5uYW1lID09PSAnZG9ja2VyJykge1xuICAgICAgcGFyYW1zID0ge1xuICAgICAgICB1c2VybmFtZTogYFxcJHt7IHNlY3JldHMuJHtkb2NrZXJDcmVkZW50aWFsLnVzZXJuYW1lS2V5fSB9fWAsXG4gICAgICAgIHBhc3N3b3JkOiBgXFwke3sgc2VjcmV0cy4ke2RvY2tlckNyZWRlbnRpYWwucGFzc3dvcmRLZXl9IH19YCxcbiAgICAgIH07XG4gICAgfSBlbHNlIGlmIChkb2NrZXJDcmVkZW50aWFsLm5hbWUgPT09ICdlY3InKSB7XG4gICAgICBwYXJhbXMgPSB7XG4gICAgICAgIHJlZ2lzdHJ5OiBkb2NrZXJDcmVkZW50aWFsLnJlZ2lzdHJ5LFxuICAgICAgfTtcbiAgICB9IGVsc2Uge1xuICAgICAgcGFyYW1zID0ge1xuICAgICAgICByZWdpc3RyeTogZG9ja2VyQ3JlZGVudGlhbC5yZWdpc3RyeSxcbiAgICAgICAgdXNlcm5hbWU6IGBcXCR7eyBzZWNyZXRzLiR7ZG9ja2VyQ3JlZGVudGlhbC51c2VybmFtZUtleX0gfX1gLFxuICAgICAgICBwYXNzd29yZDogYFxcJHt7IHNlY3JldHMuJHtkb2NrZXJDcmVkZW50aWFsLnBhc3N3b3JkS2V5fSB9fWAsXG4gICAgICB9O1xuICAgIH1cblxuICAgIHJldHVybiBbXG4gICAgICB7XG4gICAgICAgIHVzZXM6ICdkb2NrZXIvbG9naW4tYWN0aW9uQHYxJyxcbiAgICAgICAgd2l0aDogcGFyYW1zLFxuICAgICAgfSxcbiAgICBdO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGVwc1RvRG93bmxvYWRBc3NlbWJseSh0YXJnZXREaXI6IHN0cmluZyk6IGdpdGh1Yi5Kb2JTdGVwW10ge1xuICAgIGlmICh0aGlzLnByZVN5bnRoZWQpIHtcbiAgICAgIHJldHVybiB0aGlzLnN0ZXBzVG9DaGVja291dCgpO1xuICAgIH1cblxuICAgIHJldHVybiBbe1xuICAgICAgbmFtZTogYERvd25sb2FkICR7Q0RLT1VUX0FSVElGQUNUfWAsXG4gICAgICB1c2VzOiAnYWN0aW9ucy9kb3dubG9hZC1hcnRpZmFjdEB2MicsXG4gICAgICB3aXRoOiB7XG4gICAgICAgIG5hbWU6IENES09VVF9BUlRJRkFDVCxcbiAgICAgICAgcGF0aDogdGFyZ2V0RGlyLFxuICAgICAgfSxcbiAgICB9XTtcbiAgfVxuXG4gIHByaXZhdGUgc3RlcHNUb0NoZWNrb3V0KCk6IGdpdGh1Yi5Kb2JTdGVwW10ge1xuICAgIHJldHVybiBbe1xuICAgICAgbmFtZTogJ0NoZWNrb3V0JyxcbiAgICAgIHVzZXM6ICdhY3Rpb25zL2NoZWNrb3V0QHYyJyxcbiAgICB9XTtcbiAgfVxuXG4gIHByaXZhdGUgc3RlcHNUb1VwbG9hZEFzc2VtYmx5KGRpcjogc3RyaW5nKTogZ2l0aHViLkpvYlN0ZXBbXSB7XG4gICAgaWYgKHRoaXMucHJlU3ludGhlZCkge1xuICAgICAgcmV0dXJuIFtdO1xuICAgIH1cblxuICAgIHJldHVybiBbe1xuICAgICAgbmFtZTogYFVwbG9hZCAke0NES09VVF9BUlRJRkFDVH1gLFxuICAgICAgdXNlczogJ2FjdGlvbnMvdXBsb2FkLWFydGlmYWN0QHYyLjEuMScsXG4gICAgICB3aXRoOiB7XG4gICAgICAgIG5hbWU6IENES09VVF9BUlRJRkFDVCxcbiAgICAgICAgcGF0aDogZGlyLFxuICAgICAgfSxcbiAgICB9XTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyRGVwZW5kZW5jaWVzKG5vZGU6IEFHcmFwaE5vZGUpIHtcbiAgICBjb25zdCBkZXBzID0gbmV3IEFycmF5PEFHcmFwaE5vZGU+KCk7XG5cbiAgICBmb3IgKGNvbnN0IGQgb2Ygbm9kZS5hbGxEZXBzKSB7XG4gICAgICBpZiAoZCBpbnN0YW5jZW9mIEdyYXBoKSB7XG4gICAgICAgIGRlcHMucHVzaCguLi5kLmFsbExlYXZlcygpLm5vZGVzKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGRlcHMucHVzaChkKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gZGVwcy5tYXAoeCA9PiB4LnVuaXF1ZUlkKTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgQ29udGV4dCB7XG4gIC8qKlxuICAgKiBUaGUgcGlwZWxpbmUgZ3JhcGguXG4gICAqL1xuICByZWFkb25seSBzdHJ1Y3R1cmU6IFBpcGVsaW5lR3JhcGg7XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgY2xvdWQgYXNzZW1ibHkgZGlyZWN0b3J5LlxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZW1ibHlEaXI6IHN0cmluZztcbn1cblxuaW50ZXJmYWNlIEpvYiB7XG4gIHJlYWRvbmx5IGlkOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGRlZmluaXRpb246IGdpdGh1Yi5Kb2I7XG59XG5cbmZ1bmN0aW9uIHNuYWtlQ2FzZUtleXM8VCA9IHVua25vd24+KG9iajogVCwgc2VwID0gJy0nKTogVCB7XG4gIGlmICh0eXBlb2Ygb2JqICE9PSAnb2JqZWN0JyB8fCBvYmogPT0gbnVsbCkge1xuICAgIHJldHVybiBvYmo7XG4gIH1cblxuICBpZiAoQXJyYXkuaXNBcnJheShvYmopKSB7XG4gICAgcmV0dXJuIG9iai5tYXAobyA9PiBzbmFrZUNhc2VLZXlzKG8sIHNlcCkpIGFzIGFueTtcbiAgfVxuXG4gIGNvbnN0IHJlc3VsdDogUmVjb3JkPHN0cmluZywgdW5rbm93bj4gPSB7fTtcbiAgZm9yIChsZXQgW2ssIHZdIG9mIE9iamVjdC5lbnRyaWVzKG9iaikpIHtcbiAgICAvLyB3ZSBkb24ndCB3YW50IHRvIHNuYWtlIGNhc2UgZW52aXJvbm1lbnQgdmFyaWFibGVzXG4gICAgaWYgKGsgIT09ICdlbnYnICYmIHR5cGVvZiB2ID09PSAnb2JqZWN0JyAmJiB2ICE9IG51bGwpIHtcbiAgICAgIHYgPSBzbmFrZUNhc2VLZXlzKHYpO1xuICAgIH1cbiAgICByZXN1bHRbZGVjYW1lbGl6ZShrLCB7IHNlcGFyYXRvcjogc2VwIH0pXSA9IHY7XG4gIH1cbiAgcmV0dXJuIHJlc3VsdCBhcyBhbnk7XG59XG5cbi8qKlxuICogTmFtZXMgb2Ygc2VjcmV0cyBmb3IgQVdTIGNyZWRlbnRpYWxzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF3c0NyZWRlbnRpYWxzU2VjcmV0cyB7XG4gIC8qKlxuICAgKiBAZGVmYXVsdCBcIkFXU19BQ0NFU1NfS0VZX0lEXCJcbiAgICovXG4gIHJlYWRvbmx5IGFjY2Vzc0tleUlkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAZGVmYXVsdCBcIkFXU19TRUNSRVRfQUNDRVNTX0tFWVwiXG4gICAqL1xuICByZWFkb25seSBzZWNyZXRBY2Nlc3NLZXk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBkZWZhdWx0IC0gbm8gc2Vzc2lvbiB0b2tlbiBpcyB1c2VkXG4gICAqL1xuICByZWFkb25seSBzZXNzaW9uVG9rZW4/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiogZmxhdHRlbjxBPih4czogSXRlcmFibGU8QVtdPik6IEl0ZXJhYmxlSXRlcmF0b3I8QT4ge1xuICBmb3IgKGNvbnN0IHggb2YgeHMpIHtcbiAgICBmb3IgKGNvbnN0IHkgb2YgeCkge1xuICAgICAgeWllbGQgeTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==