# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/01b_base.ipynb (unless otherwise specified).

__all__ = ['USER_DIR', 'parse_datetime', 'read_S1_json', 'read_S2_json', 'get_s2_patch_directories',
           'get_s1_patch_directories', 'PATCHES_WITH_SNOW_URL', 'PATCHES_WITH_CLOUD_AND_SHADOW_URL',
           'get_complete_s1_to_s2_patch_name_mapping', 'get_complete_s2_to_s1_patch_name_mapping',
           's1_to_s2_patch_name', 's2_to_s1_patch_name', 'get_s2_patches_with_seasonal_snow',
           'get_s2_patches_with_cloud_and_shadow', 'get_s1_patches_with_seasonal_snow',
           'get_s1_patches_with_cloud_and_shadow', 'is_snowy_patch', 'is_cloudy_shadowy_patch',
           'get_s2_patches_from_original_train_split', 'get_s1_patches_from_original_train_split',
           'get_s2_patches_from_original_validation_split', 'get_s1_patches_from_original_validation_split',
           'get_s2_patches_from_original_test_split', 'get_s1_patches_from_original_test_split',
           'get_original_split_from_patch_name', 'old2new_labels', 'ben_19_labels_to_multi_hot',
           'ben_43_labels_to_multi_hot']

# Cell
import bz2
import csv
import functools
import json
import warnings
from datetime import datetime
from importlib import resources
from pathlib import Path
from typing import Dict, List, Optional, Sequence, Set, Union

import appdirs
import dateutil
import fastcore.all as fc
from fastcore.basics import compose
from fastcore.dispatch import typedispatch
from pydantic import DirectoryPath, FilePath, validate_arguments

import bigearthnet_common
import bigearthnet_common.constants as ben_constants
from .constants import OLD2NEW_LABELS_DICT

_patches_with_cloud_and_snow_resource = "patches_with_cloud_and_shadow.csv.bz2"
_patches_with_seasonal_snow_resource = "patches_with_seasonal_snow.csv.bz2"
_s1_s2_mapping_resource = "s1_s2_mapping.csv.bz2"
_test_resource = "test.csv.bz2"
_train_csv_resource = "train.csv.bz2"
_val_csv_resource = "val.csv.bz2"


# Cell
USER_DIR = Path(appdirs.user_data_dir("bigearthnet"))
USER_DIR.mkdir(exist_ok=True, parents=True)


# Cell
def parse_datetime(inp: Union[str, datetime]) -> datetime:
    """
    Parses an input into a `datetime` object.
    Will try its best to infer the correct format from a string.
    If a `datetime` object is already provided it will be returned.
    Otherwise it will raise an error.
    """
    return _parse_datetime(inp)


@typedispatch
def _parse_datetime(acquisition_date: str) -> datetime:
    return compose(dateutil.parser.parse, _parse_datetime)(acquisition_date)


@typedispatch
def _parse_datetime(acquisition_date: datetime) -> datetime:
    return acquisition_date


@typedispatch
def _parse_datetime(acquisition_date: object) -> None:
    raise TypeError("Could not parse acquisition_date!")


# Cell
@validate_arguments
def _read_json(
    json_fp: FilePath, expected_keys: Set, read_only_expected: bool = True
) -> Dict[str, str]:
    """
    Parse the json file given with the file path `json_fp`.
    The function checks if all of the `expected_keys` are present, which
    ensures that no keys have been accidentilly deleted (this has happend before).
    If `read_only_expected` is set, only the keys provided in `expected_keys` are read
    and returned.
    This prevents accidental processing of injected metadata.

    Args:
        json_fp (FilePath): Path to json file
        expected_keys (Set): Keys that are expected to be present in the json file
        read_only_expected (bool, optional): Read only the keys given in `expected_keys`. Defaults to True.

    Returns:
        [Dict[str, str]]: A dictionary of the keys.
    """
    try:
        complete_data = json.loads(json_fp.read_text())
    except json.JSONDecodeError:
        raise ValueError(f"Error trying to read json from: ", json_fp)

    missing_elements = expected_keys - complete_data.keys()
    if len(missing_elements) > 0:
        raise ValueError(f"{json_fp} is missing entries!", missing_elements)

    # ensure that the original values are loaded, as some users may customize the original json files
    if read_only_expected:
        return {k: v for k, v in complete_data.items() if k in expected_keys}
    return complete_data


def read_S1_json(json_fp: FilePath) -> Dict[str, str]:
    """
    A helper function that *safely* reads a BigEarthNet-S1 json file.
    It will ensure that all expected entries are present and only read those
    entries.
    This helps to avoid issues where the JSON files were accidentally modified
    or partially deleted.

    Note: This function will also silently fix a typo present in the `coordinates` key
    from version: S1_v1.0. A coordinates key is named `lly` and it should be `lry`.
    """
    data = _read_json(json_fp, ben_constants.BEN_S1_V1_0_JSON_KEYS)
    # Silently fix key error in S1
    if "lly" in data["coordinates"]:
        data["coordinates"]["lry"] = data["coordinates"].pop("lly")
    return data


def read_S2_json(json_fp: FilePath) -> Dict[str, str]:
    """
    A helper function that *safely* reads a BigEarthNet-S1 json file.
    It will ensure that all expected entries are present and only read those
    entries.
    This helps to avoid issues where the JSON files were accidentally modified
    or partially deleted.
    """
    return _read_json(json_fp, ben_constants.BEN_S2_V1_0_JSON_KEYS)


# Cell
@validate_arguments
def get_s2_patch_directories(dir_path: DirectoryPath) -> List[Path]:
    """
    Will find all S2 patch directories in the provided `dir_path`.
    Only directories that strictly cohere to the naming convention will be returned.
    """
    return [
        p
        for p in dir_path.iterdir()
        if ben_constants.BEN_S2_RE.fullmatch(p.name) is not None
    ]


@validate_arguments
def get_s1_patch_directories(dir_path: DirectoryPath) -> List[Path]:
    """
    Will find all S1 patch directories in the provided `dir_path`.
    Only directories that strictly cohere to the naming convention will be returned.
    """
    return [
        p
        for p in dir_path.iterdir()
        if ben_constants.BEN_S1_RE.fullmatch(p.name) is not None
    ]


# Cell
PATCHES_WITH_SNOW_URL = (
    "http://bigearth.net/static/documents/patches_with_seasonal_snow.csv"
)
PATCHES_WITH_CLOUD_AND_SHADOW_URL = (
    "http://bigearth.net/static/documents/get_patches_with_cloud_and_shadow.csv"
)


# Cell
@functools.lru_cache()
def _load_s1_s2_patch_name_mapping(from_s1_to_s2: bool = True) -> Dict[str, str]:
    """
    Load a dictionary which maps the S1 patch name to the S2 patch name (if `from_s1_to_s2`) or
    the S2 patch name to the S1 patch name.

    Stored as bz2 compressed csv file as `s1_s2_mapping.csv.bz2`.
    The first column is the s1 patch name and the second column the s2 name.

    The data could be regenerated with (requires the output of `bigearthnet_gdf_builder`):

    >>> import geopandas
    >>> raw_gdf = geopandas.read_parquet("raw_ben_s1_gdf.parquet")
    >>> s1_s2_map = raw_gdf[["name", "corresponding_s2_patch"]]
    >>> s1_s2_map.rename({"name": "s1_name", "corresponding_s2_patch": "s2_name"}, axis=1).to_csv("s1_s2_mapping.csv.gzip", index=False)
    """
    resource = "s1_s2_mapping.csv.bz2"
    if not resources.is_resource(bigearthnet_common, resource):
        raise ValueError(
            f"{resource} resource is not available! This means that it was forgotten to be packaged."
        )

    with resources.path(bigearthnet_common, resource) as resource_path:
        with bz2.open(resource_path, mode="rt") as csv_file:
            reader = csv.DictReader(
                csv_file
            )  # field-names are encoded as first csv row
            key, value = (
                ("s1_name", "s2_name") if from_s1_to_s2 else ("s2_name", "s1_name")
            )
            return {row[key]: row[value] for row in reader}


def get_complete_s1_to_s2_patch_name_mapping() -> Dict[str, str]:
    """
    Load entire Sentinel-1 to Sentinel-2 BigEarthNet patch name mapping.

    Returns:
        Dict[str, str]: Sentinel-1 patch name keys with corresponding Sentinel-2 patch name as value
    """
    return _load_s1_s2_patch_name_mapping(from_s1_to_s2=True)


def get_complete_s2_to_s1_patch_name_mapping() -> Dict[str, str]:
    """
    Load entire Sentinel-2 to Sentinel-1 BigEarthNet patch name mapping.

    Returns:
        Dict[str, str]: Sentinel-2 patch name keys with corresponding Sentinel-1 patch name as value
    """
    return _load_s1_s2_patch_name_mapping(from_s1_to_s2=False)


def s1_to_s2_patch_name(s1_patch_name: str) -> str:
    """
    Convert BigEarthNet Sentinel-1 patch name to Sentinel-2 patch name.
    The function caches intermediate results.
    The function should be highly performant.

    Args:
        s1_patch_name (str): complete BigEarthNet Sentinel-1 patch name

    Returns:
        str: Corresponding Sentinel-2 patch name
    """
    return get_complete_s1_to_s2_patch_name_mapping()[s1_patch_name]


def s2_to_s1_patch_name(s2_patch_name: str) -> str:
    """
    Convert BigEarthNet Sentinel-2 patch name to Sentinel-1 patch name.
    The function caches intermediate results.
    The function should be highly performant.

    Args:
        s2_patch_name (str): complete BigEarthNet Sentinel-2 patch name

    Returns:
        str: Corresponding Sentinel-1 patch name
    """
    return get_complete_s2_to_s1_patch_name_mapping()[s2_patch_name]


# Cell
@validate_arguments
def _conv_single_col_csv_resource_to_set(
    resource: str,
) -> Set:
    """
    Given a `resource` name of an encoded CSV file *without* a header
    line and only a single column, return the set of
    all values.
    """
    if not resources.is_resource(bigearthnet_common, resource):
        raise ValueError(f"{resource} is an unknown resource!")

    with resources.path(bigearthnet_common, resource) as resource_path:
        with bz2.open(resource_path, mode="rt") as csv_file:
            col_name = "Column"
            reader = csv.DictReader(csv_file, fieldnames=[col_name])
            return {row[col_name] for row in reader}


@functools.lru_cache()
def get_s2_patches_with_seasonal_snow() -> Set:
    """List all patches with seasonal snow from **original** BigEarthNet-S2 dataset."""
    return _conv_single_col_csv_resource_to_set(_patches_with_seasonal_snow_resource)


@functools.lru_cache()
def get_s2_patches_with_cloud_and_shadow() -> Set:
    """List all patches with cloud and shadow from **original** BigEarthNet-S2 dataset."""
    return _conv_single_col_csv_resource_to_set(_patches_with_cloud_and_snow_resource)


@functools.lru_cache()
def get_s1_patches_with_seasonal_snow() -> Set:
    """List all patches with seasonal snow from **original** BigEarthNet-S1 dataset."""
    _s2_patches_with_snow = get_s2_patches_with_seasonal_snow()
    return {s2_to_s1_patch_name(s2_patch) for s2_patch in _s2_patches_with_snow}


@functools.lru_cache()
def get_s1_patches_with_cloud_and_shadow() -> Set:
    """List all patches with cloud and shadow from **original** BigEarthNet-S1 dataset."""
    _s2_patches_with_clouds = get_s2_patches_with_cloud_and_shadow()
    return {s2_to_s1_patch_name(s2_patch) for s2_patch in _s2_patches_with_clouds}


# Cell
@validate_arguments
def is_snowy_patch(patch_name: str):
    """
    Fast function that checks whether `patch_name` is a patch
    that contains a lot of seasonal snow.

    This works for S1 and S2 patch names!
    """
    return (
        patch_name in get_s2_patches_with_seasonal_snow()
        or patch_name in get_s1_patches_with_seasonal_snow()
    )


@validate_arguments
def is_cloudy_shadowy_patch(patch_name: str):
    """
    Fast function that checks whether `patch_name` is a patch
    that contains a lot of shadow or is obstructed by clouds.

    This works for S1 and S2 patch names!
    """
    return (
        patch_name in get_s2_patches_with_cloud_and_shadow()
        or patch_name in get_s1_patches_with_cloud_and_shadow()
    )


# Cell

# "https://git.tu-berlin.de/rsim/BigEarthNet-S2_19-classes_models/-/raw/master/splits/train.csv",
@functools.lru_cache()
def get_s2_patches_from_original_train_split() -> Set:
    """
    List all Sentinel-2 train patches from the original train/validation/test split.
    """
    return _conv_single_col_csv_resource_to_set(_train_csv_resource)


@functools.lru_cache()
def get_s1_patches_from_original_train_split() -> Set:
    """
    List all Sentinel-1 train patches from the original train/validation/test split.
    """
    s2_train_patches = get_s2_patches_from_original_train_split()
    return {s2_to_s1_patch_name(s2_patch) for s2_patch in s2_train_patches}


@functools.lru_cache()
def get_s2_patches_from_original_validation_split() -> Set:
    """
    List all Sentinel-2 validation patches from the original train/validation/test split.
    """
    return _conv_single_col_csv_resource_to_set(_val_csv_resource)


@functools.lru_cache()
def get_s1_patches_from_original_validation_split() -> Set:
    """
    List all Sentinel-1 validation patches from the original train/validation/test split.
    """
    s2_validation_patches = get_s2_patches_from_original_validation_split()
    return {s2_to_s1_patch_name(s2_patch) for s2_patch in s2_validation_patches}


@functools.lru_cache()
def get_s2_patches_from_original_test_split() -> Set:
    """
    List all Sentinel-2 test patches from the original train/validation/test split.
    """
    return _conv_single_col_csv_resource_to_set(_test_resource)


@functools.lru_cache()
def get_s1_patches_from_original_test_split() -> Set:
    """
    List all Sentinel-1 test patches from the original train/validation/test split.
    """
    s2_test_patches = get_s2_patches_from_original_test_split()
    return {s2_to_s1_patch_name(s2_patch) for s2_patch in s2_test_patches}


# Cell
@validate_arguments
def get_original_split_from_patch_name(patch: str) -> Optional[str]:
    """
    Returns "train"/"validation"/"test" or `None`.
    The value is retrieved from the original BigEarthNet-S1/S2
    train/validation/test split. If the input is not present
    in any split, it will return `None` and raise a UserWarning.
    This happens for patches that are either in the
    cloud/shadow or seasonal snow set or if there exists no 19-label target.

    Note: This works for Sentinel-2 and Sentinel-1 patch names!
    """
    s1_train = get_s1_patches_from_original_train_split()
    s2_train = get_s2_patches_from_original_train_split()
    s1_validation = get_s1_patches_from_original_validation_split()
    s2_validation = get_s2_patches_from_original_validation_split()
    s1_test = get_s1_patches_from_original_test_split()
    s2_test = get_s2_patches_from_original_test_split()

    if patch in s1_train or patch in s2_train:
        return "train"
    elif patch in s1_validation or patch in s2_validation:
        return "validation"
    elif patch in s1_test or patch in s2_test:
        return "test"
    warnings.warn(
        "Provided an input patch name which was not part of the original split.",
        UserWarning,
    )
    return None


# Cell
@validate_arguments
def _old2new_label(old_label: str) -> Optional[str]:
    """
    Converts old-style BigEearthNet label to the
    new labels.

    > Note: Some labels were removed! This function
    will return `None` if the label was removed and
    raise a `KeyError` if the input label is unknown.
    """
    return OLD2NEW_LABELS_DICT[old_label]


def old2new_labels(old_labels: Sequence[str]) -> Optional[List[str]]:
    """
    Converts a list of old-style BigEarthNet labels
    to a list of labels.

    If there are no corresponding new labels (which can happen with original BEN patches!)
    then the function will return `None` and raise a user warning.

    If an illegal/unknown input label is provided, a `KeyError` is raised.
    """
    new_labels = [
        _old2new_label(l) for l in old_labels if _old2new_label(l) is not None
    ]
    if len(old_labels) > 0 and len(new_labels) == 0:
        warnings.warn(
            "Provided a list of old labels that only contains `removed` labels!",
            UserWarning,
        )
        new_labels = None
    return new_labels


# Cell
@validate_arguments
def ben_19_labels_to_multi_hot(labels: Sequence[str]) -> List[float]:
    """
    Convenience function that converts an input sequence of labels into
    a multi-hot encoded vector.
    The naturally ordered label list is used as an encoder reference
    - `bigearthnet_common.NEW_LABELS`

    If an unknown label is given, a `KeyError` is raised.

    Be aware that this approach assumes that **all** labels are actually used in the dataset!
    This is not necessarily the case if you are using a subset!
    For example, the "Agro-forestry areas" class is only present in Portugal and in no other country!
    """
    idxs = [ben_constants.NEW_LABELS_TO_IDX[label] for label in labels]
    multi_hot = fc.L([0] * len(ben_constants.NEW_LABELS))
    multi_hot[idxs] = 1.0
    return list(multi_hot)


@validate_arguments
def ben_43_labels_to_multi_hot(labels: Sequence[str]) -> List[float]:
    """
    Convenience function that converts an input sequence of labels into
    a multi-hot encoded vector.
    The naturally ordered label list is used as an encoder reference
    - `bigearthnet_common.OLD_LABELS`

    If an unknown label is given, a `KeyError` is raised.

    Be aware that this approach assumes that **all** labels are actually used in the dataset!
    This is not necessarily the case if you are using a subset!
    For example, the "Agro-forestry areas" class is only present in Portugal and in no other country!
    """
    idxs = [ben_constants.OLD_LABELS_TO_IDX[label] for label in labels]
    multi_hot = fc.L([0] * len(ben_constants.OLD_LABELS))
    multi_hot[idxs] = 1.0
    return list(multi_hot)
