#!/usr/bin/env python

__author__ = "Alessandro Coppe"

'''
Filters a VCF file generated by vs_format_converter.py from the VarScan2 output

Parameters:
    -v the path to the VCF
    -f maximum frequency for filtering (default: 0)
'''

import argparse
import os.path
import sys


class varscan_vcf_format:
    '''The values from the normal or tumoral FORMAT '''
    def __init__(self, line, type):
        ''' 
        The constructor

        Parameters:
            line: the line from the VCF
            type: normal (n) or tumor (t) format
        ''' 
        line = line[:-1]
        splitted_line = line.split("\t")
        if type == "n":
            format_values = splitted_line[9].split(":")
        if type == "t":
            format_values = splitted_line[10].split(":")
        # GT: consensus genotype call (like 0/1)
        self.gt = format_values[0]
        # GQ: Quality of the assigned genotype
        self.gq = format_values[1]
        # DP: filtered depth
        self.dp = int(format_values[2])
        # RD: Reads supporting reference
        self.rd = int(format_values[3])
        # AD: Reads supporting variant
        self.ad = int(format_values[4])
        # FREQ: Variant allele frequency
        self.freq = float(format_values[5][:-1])


def is_a_varscan_vcf(vcf_name):
    '''
    Does a VCF file derives from VarScan2 after using vs_format_converter.py from varscan_accessories

    Parameters:

    vcf_name (string): the path of the VCF to check

    Returns:

    True or False
    '''
    varscan = False
    with open(vcf_name) as vcf:
        for line in vcf:
            if line.startswith("##source=VarScan2"):
                return True
    return False


def main():
    parser = argparse.ArgumentParser(description="Removes variants present in normal sample")
    parser.add_argument('-v', '--vcf', action='store', type=str, help="The VCF to be filtered by SNPs from COSMIC", required=True)
    parser.add_argument('-f', '--max_control_frequency', type=float, help="The max frequency in control sample", default=0)
    args = parser.parse_args()

    is_varscan_vcf = is_a_varscan_vcf(args.vcf)

    if is_varscan_vcf == False:
        return -1

    with open(args.vcf) as vcf:
        for line in vcf:
            if line.startswith("#"):
                print(line[:-1])
            else:
                normal_variant = varscan_vcf_format(line, "n")
                tumor_variant = varscan_vcf_format(line, "t")
                if args.max_control_frequency == 0:
                    if (normal_variant.freq == 0):
                        print(line[:-1])
                else:
                    if normal_variant.freq < args.max_control_frequency:
                        print(line[:-1])

    return 0
 

if __name__ == "__main__":
    main()
