# AUTOGENERATED! DO NOT EDIT! File to edit: 00_utils.ipynb (unless otherwise specified).

__all__ = ['settings_dump', 'settings_load', 'CircArrayBuffer', 'DataCube']

# Cell
import json
import datetime
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import xarray as xr
from fastcore.foundation import patch

# Cell
from typing import Union, Callable

# Cell

def settings_dump(fname:str,settings:dict):
    with open(fname, 'w') as outfile:
        json.dump(settings, outfile,indent=4,)

def settings_load(fname:str) -> dict:
    with open(fname) as json_file:
        openhsi_settings = json.load(json_file)
    return openhsi_settings

# Cell

class CircArrayBuffer():
    """Circular Buffer implementation on ndarrays. Each push is a (n-1)darray.
    The insert axis can be specified and modified after initialisation using `update_axis`."""

    def __init__(self,size:tuple=(100,100),axis:int=0,dtype=np.uint16,show_func=None):
        """initialise array and write pointer"""
        self.data = np.zeros(size,dtype=dtype)
        self.size = size
        self.axis = axis
        self.write_pos = [slice(None,None,None) if i != axis else 0 for i in range(len(size)) ]
        self.show_func = show_func

    def __getitem__(self,key):
        return self.data[key]

@patch
def update_axis(self:CircArrayBuffer, axis, write_pos=0):
    """changes the write axis in array"""
    self.axis = axis
    self.write_pos = [slice(None,None,None) if i != axis else write_pos for i in range(len(self.size)) ]

@patch
def push(self:CircArrayBuffer,line):
    """writes a (n-1)darray into the ndarray"""
    self.data[tuple(self.write_pos)] = line

    self.write_pos[self.axis] += 1
    if self.write_pos[self.axis] == self.size[self.axis]:
        self.write_pos[self.axis] = 0
        # write data to file?

@patch
def show(self:CircArrayBuffer):
    """Display the data"""
    if self.show_func is None:
        if len(self.size) == 2:
            plt.imshow(self.data,cmap='gray')
            plt.show()
        elif len(self.size) == 3:
            plt.imshow(np.sum(self.data,axis=-1),cmap='gray')
            plt.show()
        elif len(self.size) == 1:
            print(f"#({self.size[0]}) {self.data}")
    elif self.show_func is not None:
        self.show_func(self.data)
    else:
        print("Unsupported array shape. Please use 2D or 3D shapes or use your own custom show function")

@patch
def dump(self:CircArrayBuffer,fname:str):
    if ".nc" in fname:
        pass
    else:
        with open(fname,'wb') as handle:
            pickle.dump(self.data,handle)

@patch
def load(self:CircArrayBuffer,fname:str):
    if ".nc" in fname:
        pass
    else:
        with open(fname,'rb') as handle:
            self.data = pickle.load(handle)
    self.size = np.shape(self.data)
    self.write_pos = [slice(None,None,None) if i != axis else 0 for i in range(len(size)) ]



# Cell

class DataCube(CircArrayBuffer):
    """Provides smile corrected views of a hyperspectral datacube. Also collapses spectral pixels into bands."""

    def __init__(self,line_sz:tuple=(20,100),row_slice:tuple=(148,598),n_lines:int=10,axis:int=1,dtype=np.uint16,
                 shifts:np.ndarray=None,fwhm_nm:float=50,wavelength_range:tuple=(380,820)):

        self.row_slice = slice(row_slice[0],row_slice[-1],None)
        self.fwhm_nm = fwhm_nm
        self.wavelength_range = wavelength_range
        self.timestamps = pd.date_range(datetime.datetime.now(), periods=n_lines, freq="1s").to_numpy() # preallocate array

        # for smile corrected functionality
        self.shifts = shifts if shifts is not None else np.zeros((np.ptp(row_slice),),dtype=np.uint16)
        size = [np.ptp(row_slice), n_lines, line_sz[1] - np.max(self.shifts)]

        # for collapsing spectral pixels into bands
        self.width = np.uint16(fwhm_nm*line_sz[1]/np.ptp(self.wavelength_range))
        self.rows, self.cols = size[0], size[2]
        self.byte_sz    = dtype(0).nbytes
        self.reduced_shape = (self.rows,n_lines,self.cols//self.width)

        self.wavelengths = np.linspace(*wavelength_range,num=line_sz[1],dtype=np.float32)[np.max(self.shifts):]
        self.wavelengths = np.lib.stride_tricks.as_strided(self.wavelengths,
                                strides=(self.width*4,4),
                                shape=(len(self.wavelengths)//self.width,self.width))
        self.wavelengths = np.around(self.wavelengths.mean(axis=1),decimals=1)
        super().__init__(self.reduced_shape,axis,dtype)

        self.line_buff = CircArrayBuffer((size[0],size[2]),axis=0,dtype=dtype)


    def __repr__(self):
        return f"DataCube:\nshifts=\n{self.shifts}\n" + super().__repr__()

    def push(self,line:np.ndarray) -> None:
        """writes a (n-1)darray into the ndarray"""
        self.timestamps[ self.write_pos[self.axis] ] = np.datetime64(datetime.datetime.now())

        for i in range(self.size[0]):
            self.line_buff.push(line[self.row_slice.start+i,self.shifts[i]:self.shifts[i]+self.line_buff.size[1]])

        buff = np.lib.stride_tricks.as_strided(self.line_buff.data,
                        shape=(self.rows,self.cols//self.width,self.width),
                        strides=(self.cols*self.byte_sz,self.width*self.byte_sz,self.byte_sz))
        super().push(buff.sum(axis=-1))

    def show(self,key):
        if isinstance(key[1],int):
            plt.imshow(self.data[key],extent=[*self.wavelength_range[:2],self.rows,0],cmap="gray")
        else:
            plt.imshow(self.data[key],cmap="gray")
        plt.colorbar()
        plt.show()