# AUTOGENERATED! DO NOT EDIT! File to edit: 02_capture.ipynb (unless otherwise specified).

__all__ = ['SimulatedBasicCamera', 'HgAr_lines_approx_sorted', 'OpenHSI']

# Cell
from fastcore.foundation import patch
from PIL import Image
import numpy as np
import matplotlib.pyplot as plt
from ximea import xiapi
from tqdm import tqdm
import json
from typing import Iterable, Union

# Cell

from scipy.signal.windows import gaussian

class SimulatedBasicCamera():
    """Simulated Camera for testing. Can return image of spectral lines."""
    def __init__(self,shape:tuple=(772,2064),dtype=np.uint16,row_slice:tuple=(210,680),peak:int=255):
        self.shape = shape
        self.dtype = dtype
        self.row_slice = slice(row_slice[0],row_slice[1],None)
        self.peak  = peak

    def get_raw_img(self,spectral_lines:Union[bool,Iterable]=False,λ_range_nm:tuple=(350,850)):
        img = np.zeros(self.shape,dtype=self.dtype)
        if spectral_lines:
            spectra = self.spectra1d(spectral_lines,λ_range_nm)
            for i in range(self.row_slice.start,self.row_slice.stop):
                img[i,:] = spectra
        else:
            img[self.row_slice,:] = self.peak
        return img

    def spectra1d(self,spectral_lines:list,λ_range_nm:tuple=(350,850),fwhm_nm:float=1.5,window:bool=False) -> np.array:
        # spectral_lines approx sorted by emission strength

        spectra = np.zeros((self.shape[1],),dtype=self.dtype)
        wavelengths = np.linspace(λ_range_nm[0],λ_range_nm[1],len(spectra))

        width   = np.uint16(fwhm_nm*len(spectra)/np.ptp(wavelengths))

        for i in range(len(spectral_lines)):
            indx = np.sum(wavelengths<spectral_lines[i])
            if indx > 0 and indx < len(spectra):
                strength = self.peak if window else self.peak*np.exp(-0.15*i)
                spectra[indx-width//2:indx+width-width//2] += np.uint16(strength*gaussian(width,0.34*width))

        return spectra



# Cell

# approx sorted by emission strength
HgAr_lines_approx_sorted = [254,436,546,764,405,365,578,750,738,697,812,772,912,801,842,795,706,826,852,727]


# Cell
class OpenHSI(object):
    """XIMEA camera class to change camera settings and take images"""

    def __init__(self, exposure_ms:int = 100, gain:int = 0,
                 xbinwidth:int = 896, xbinoffset:int = 528, serialnumber:str = None):
        """Init the camera"""

        self.xicam = xiapi.Camera()
        self.xicam.open_device_by_SN(serialnumber) if serialnumber else self.xicam.open_device()

        print(f'Connected to device {self.xicam.get_device_sn()}')

        self.xbinwidth  = xbinwidth
        self.xbinoffset = xbinoffset
        self.exposure   = exposure_ms
        self.gain       = 0

        self.xicam.set_width(self.xbinwidth)
        self.xicam.set_offsetX(self.xbinoffset)
        self.xicam.set_exposure_direct(1000*self.exposure)
        self.xicam.set_gain_direct(self.gain)

        self.xicam.set_imgdataformat("XI_RAW16")
        self.xicam.set_output_bit_depth("XI_BPP_12")
        self.xicam.enable_output_bit_packing()
        self.xicam.disable_aeag()

        self.xicam.set_binning_vertical(2)
        self.xicam.set_binning_vertical_mode("XI_BIN_MODE_SUM")

        self.rows, self.cols = self.xicam.get_height(), self.xicam.get_width()

        self.img = xiapi.Image()

        self.load_cam_settings()

    def __str__(self):
        return f"Connected to device {self.xicam.get_device_sn()}"

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.xicam.stop_acquisition()
        self.xicam.close_device()

    @property
    def exposure(self) -> float: # or int?
        """Current exposure in ms."""
        return self.xicam.get_exposure() / 1000

    @exposure.setter
    def exposure(self, val):
        """Update exposure in ms."""
        self.xicam.set_exposure_direct(val * 1000)

    @property
    def gain(self) -> float: # or int?
        """Current exposure property in dB from 0 to 24 dB."""
        return self.xicam.get_gain()

    @gain.setter
    def gain(self, val):
        """Update gain in dB."""
        self.xicam.set_gain_direct(val)

# Cell
@patch
def get_raw_img(self:OpenHSI,save_file_name:str=None):
    self.xicam.start_acquisition()
    self.xicam.get_image(self.img)
    img = self.img.get_image_data_numpy()
    self.xicam.stop_acquisition()


    if save_file_name is not None:
        im = Image.fromarray(img)
        im.save(save_file_name)
    return img

# Cell
@patch
def load_cam_settings(self:OpenHSI,fname:str="calibration_files/settings.txt"):
    with open(fname) as json_file:
        self.settings = json.load(json_file)
    self.row_slice    = self.settings["row_slice"]
    self.smile_shifts = np.array(self.settings["smile_shifts"])
    self.resolution   = self.settings["resolution"]
    self.wavelength_range = self.settings["index2wavelength_range"][:2]
    self.fwhm         = self.settings["fwhm_nm"]



# Cell

from typing import Callable

@patch
def get_datacube(self:OpenHSI, n:int = 1, show:bool = True) -> np.ndarray:
    """Take `n` images with option to `show` first image taken."""

    self.xicam.start_acquisition()
    self.dc = DataCube(self.resolution,self.row_slice,n_lines=n,axis=1,shifts=self.smile_shifts,fwhm_nm=4.,wavelength_range=self.wavelength_range)

    for i in tqdm(range(n)):

        self.xicam.get_image(self.img)
        self.dc.push(self.img.get_image_data_numpy())

    self.xicam.stop_acquisition()

    if show and n > 1:
        plt.imshow(np.sum(self.dc.data,axis=2),cmap="gray")
        plt.ylabel('Cross-track pixels')
        plt.xlabel('Along-track pixels')
        plt.show()
    elif show and n==1:
        extent=(self.settings["index2wavelength_range"][0],self.settings["index2wavelength_range"][1],0,self.resolution[0])
        plt.imshow(self.dc.data[:,0,:],cmap="gray",extent=extent)
        plt.xlabel('Wavelength (nm)')
        plt.ylabel('Cross-track pixels')
        plt.show()

    return self.dc.data