# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2015-2019 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Created on Mon August 24, 2021
Authors: elena.manea@infp.ro, laurentiu.danciu@sed.ethz.ch

Module exports: `ManeaEtAl2021`
"""

import numpy as np
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, SA
from scipy.constants import g


def _compute_magnitude_term(C, mag):
    """
    Compute magnitude term
    """
    term01 = mag - 5.7
    mag_term = C['phi0'] + C['phi1'] * term01 + C['phi2'] * term01**2
    return mag_term


def _compute_distance_term(C, rhypo, backarc):
    """
    Compute distance term and regional term
    """
    term = C['phi3'] * np.log(rhypo) + _compute_arc_regional_term(
        C, backarc) * rhypo
    return term


def _compute_arc_regional_term(C, backarc):
    """
    Compute regional term - location with respect to the arc:
    fore-arc sites (2 used in the paper / 0 in OQ),
    back-arc sites (0 used in the paper/ 1 in OQ),
    along-arc sites (1 used in the paper / 2 in OQ)
    """
    term = np.zeros(len(backarc))
    # fore-arc sites
    term[backarc == 0] = C['phi7']
    # back-arc sites
    term[backarc == 1] = C['phi6']
    # along-arc sites
    term[backarc == 2] = C['phi5']
    return term


def _compute_depth_term(C, hypo_depth):
    return C['phi4']*hypo_depth


def _get_site_amplification(C, sites):
    """
    Compute site terms - dummy terms + amplif
    """
    ssa, ssb, ssc = _get_site_type_dummy_variables(sites)

    return ((ssc*C['phi11'] + ssb*C['phi10'] +
            + ssa*C['phi9']) * np.ones(len(sites.vs30)) +
            + _compute_site_amplif(C, sites))


def _get_site_type_dummy_variables(sites):
    """
    Get site type dummy variables, three different EC8 site classes
    The recording sites are classified into 3 classes,
    based on the shear wave velocity intervals in the uppermost 30 m, Vs30,
    according to the EC8 (CEN 2003):
    class A: Vs30 > 800 m/s
    class B: Vs30 = 360 − 800 m/s
    class C: Vs30 = 180 - 360 m/s
    """
    ssa = np.zeros(len(sites.vs30))
    ssb = np.zeros(len(sites.vs30))
    ssc = np.zeros(len(sites.vs30))
    # SClass C; 180 m/s <= Vs30 <= 360 m/s.
    idx = (sites.vs30 >= 180.0) & (sites.vs30 < 360.0)
    ssc[idx] = 1.0
    # Class B; 360 m/s <= Vs30 <= 800 m/s.
    idx = (sites.vs30 >= 360.0) & (sites.vs30 < 800)
    ssb[idx] = 1.0
    # Class A; Vs30 > 800 m/s.
    idx = (sites.vs30 >= 800.0)
    ssa[idx] = 1.0
    return ssa, ssb, ssc


def _compute_site_amplif(C, sites):
    """
    Compute site amplification using fundamental frequency of resonance(f0)
    If f0 is not defined, please set it equal with the reference frequency
    (fref=15, attributed to the reference site condition-rock sites)
    """
    return C['phi8'] * np.log(sites.f0/15)


def get_mean_values(C, ctx):
    """
    Returns the mean values for a specific IMT
    """
    mean = (_compute_magnitude_term(C, ctx.mag) +
            _compute_distance_term(C, ctx.rhypo, ctx.backarc) +
            _compute_depth_term(C, ctx.hypo_depth) +
            _get_site_amplification(C, ctx))
    return mean


class ManeaEtAl2021(GMPE):
    """
    Implements the Subduction GMPE developed by Elena Florinela Manea,
    Carmen Ortanza Cioflan and Laurentiu Danciu, otherwise known as the
    "Ground-motion models for Vrancea intermediate-depth earthquakes
    (Earthquake Spectra,2021,87552930211032985), for subduction inslab events.
    """
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.SUBDUCTION_INTRASLAB
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, SA}
    #: Supported intensity measure component is the geometric mean component
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.GEOMETRIC_MEAN
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL,
        const.StdDev.INTER_EVENT,
        const.StdDev.INTRA_EVENT}
    #: Required site measure as Vs30, fundamental frequency of resonance
    #: (if unknown set it as 15 - rock sites),
    REQUIRES_SITES_PARAMETERS = {'vs30', 'f0', 'backarc'}
    #: Required rupture parameters are magnitude and depth for the inslab model
    REQUIRES_RUPTURE_PARAMETERS = {'mag', 'hypo_depth'}
    #: Required distance measure is hypocentral distance
    REQUIRES_DISTANCES = {'rhypo'}

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            # compute mean and convert from cm/s**2 to g
            mean[m] = np.log(np.exp(get_mean_values(C, ctx)) * 1e-2 / g)
            # Get standard deviations
            tau[m] = C["tau"]
            phi[m] = C["phi"]
            sig[m] = C['sigma']

    COEFFS = CoeffsTable(sa_damping=5, table="""\
   imt     phi0        phi1        phi2        phi3        phi4        phi5        phi6        phi7        phi8        phi9        phi10       phi11       phi      tau        sigma
   pga     4.025595    1.621585    -0.292480   0.214168    -0.007613   -0.012832   -0.016391   -0.007789   -0.079350   -0.312903   -0.220612   -0.216223   0.692827    0.241538    0.733723
   0.0500  3.952591    1.504418    -0.313232   0.333729    -0.007394   -0.013859   -0.017524   -0.008054   -0.057029   -0.598011   -0.444958   -0.428217   0.719027    0.218638    0.751533
   0.1000  2.614581    1.406083    -0.345507   0.702137    -0.007345   -0.015811   -0.020344   -0.009906   -0.078821   -0.519976   -0.238897   -0.258565   0.749747    0.186531    0.772603
   0.1500  5.432465    1.466375    -0.374716   0.075349    -0.007155   -0.012035   -0.017251   -0.007230   -0.103554   -0.453729   -0.226770   -0.261769   0.743654    0.248487    0.784071
   0.2000  5.595978    1.543539    -0.396745   -0.035388   -0.006680   -0.011705   -0.016533   -0.007998   -0.086191   -0.054641   0.236662    0.354709    0.721183    0.259187    0.766344
   0.2500  7.200716    1.654710    -0.392268   -0.401341   -0.006585   -0.009857   -0.014155   -0.006217   -0.091235   0.048698    0.131088    0.240840    0.745346    0.305120    0.805382
   0.3000  7.961656    1.763811    -0.383947   -0.597177   -0.006754   -0.008640   -0.012733   -0.005281   -0.099966   0.053849    0.117093    0.200180    0.745312    0.344535    0.821094
   0.3500  7.302146    1.825933    -0.400855   -0.489150   -0.006894   -0.009029   -0.012891   -0.005852   -0.100287   0.166838    0.252812    0.295754    0.746893    0.335421    0.818753
   0.4000  7.496009    1.912918    -0.390131   -0.542885   -0.006664   -0.009455   -0.012652   -0.005911   -0.094347   0.192523    0.231624    0.257723    0.764606    0.327001    0.831596
   0.4500  7.250573    1.991307    -0.385370   -0.453492   -0.007012   -0.010327   -0.013020   -0.006531   -0.083213   0.118375    0.111060    0.121214    0.776354    0.326521    0.842224
   0.5000  7.052669    2.037875    -0.387456   -0.405306   -0.007599   -0.010856   -0.013115   -0.006926   -0.077841   0.166895    0.106499    0.094953    0.772706    0.336974    0.842987
   0.5500  6.967267    2.077844    -0.411739   -0.382824   -0.007991   -0.010992   -0.013050   -0.006983   -0.072419   0.145601    0.074511    0.053116    0.772536    0.329396    0.839830
   0.6000  7.004557    2.108458    -0.416254   -0.412492   -0.008011   -0.010684   -0.012434   -0.006634   -0.071739   0.113151    0.010839    -0.020847   0.766393    0.335123    0.836460
   0.6500  6.956747    2.149386    -0.418802   -0.407586   -0.008154   -0.010426   -0.011887   -0.006405   -0.074530   0.013880    -0.102924   -0.152134   0.757849    0.329303    0.826303
   0.7000  6.732008    2.181437    -0.398318   -0.375138   -0.008368   -0.010341   -0.011572   -0.006401   -0.075376   -0.069936   -0.171714   -0.229259   0.753294    0.355305    0.832883
   0.7500  6.585760    2.187655    -0.439662   -0.354944   -0.008359   -0.010191   -0.011258   -0.006328   -0.083573   -0.160015   -0.245907   -0.320181   0.754468    0.346585    0.830267
   0.8000  6.093086    2.206244    -0.449031   -0.257233   -0.008489   -0.010544   -0.011435   -0.006749   -0.085265   -0.202901   -0.262145   -0.355927   0.740951    0.350736    0.819771
   0.8500  5.948997    2.229070    -0.445908   -0.238133   -0.008727   -0.010581   -0.011197   -0.006770   -0.089310   -0.204653   -0.294351   -0.409076   0.735361    0.353616    0.815966
   0.9000  5.337920    2.238156    -0.456065   -0.127986   -0.008709   -0.011154   -0.011390   -0.007350   -0.091805   -0.173925   -0.254632   -0.378706   0.726668    0.346358    0.804991
   0.9500  4.998766    2.258973    -0.458789   -0.076782   -0.008583   -0.011468   -0.011461   -0.007643   -0.090701   -0.126019   -0.231895   -0.353432   0.716099    0.339620    0.792552
   1.0000  4.897982    2.282278    -0.450046   -0.089679   -0.008454   -0.011468   -0.011081   -0.007639   -0.090177   -0.027730   -0.186378   -0.302624   0.705812    0.336331    0.781849
   1.1000  4.615709    2.333505    -0.426365   -0.104075   -0.007995   -0.011435   -0.010927   -0.007699   -0.089359   0.127145    -0.071883   -0.166692   0.705873    0.329173    0.778853
   1.2000  5.009129    2.383702    -0.408751   -0.249553   -0.007674   -0.010710   -0.009937   -0.007028   -0.088494   0.174421    -0.042242   -0.112955   0.691329    0.333641    0.767628
   1.3000  4.839285    2.433264    -0.400524   -0.271119   -0.007346   -0.010318   -0.009494   -0.006859   -0.089278   0.245378    0.019830    -0.029044   0.680833    0.326774    0.755192
   1.4000  4.434322    2.471533    -0.387664   -0.236810   -0.006950   -0.010354   -0.009473   -0.007066   -0.087648   0.293191    0.095394    0.055729    0.669157    0.317031    0.740459
   1.5000  4.169711    2.496787    -0.378644   -0.215603   -0.006742   -0.010223   -0.009220   -0.007100   -0.087498   0.276668    0.093114    0.065743    0.656745    0.312779    0.727423
   1.6000  4.211424    2.518935    -0.370811   -0.255826   -0.006395   -0.009916   -0.008852   -0.006838   -0.086538   0.225123    0.047277    0.026968    0.651815    0.305979    0.720060
   1.7000  4.097084    2.541586    -0.355083   -0.278278   -0.006068   -0.009539   -0.008515   -0.006585   -0.087183   0.223726    0.060206    0.043074    0.652182    0.296756    0.716524
   1.8000  4.307410    2.559459    -0.344041   -0.369365   -0.005661   -0.008915   -0.007809   -0.006008   -0.087453   0.191436    0.051648    0.033602    0.649543    0.285686    0.709593
   1.9000  3.797734    2.571833    -0.332671   -0.292231   -0.005335   -0.009113   -0.007997   -0.006293   -0.087638   0.171411    0.051321    0.038333    0.639571    0.266418    0.692842
   2.0000  3.370490    2.589052    -0.319873   -0.233897   -0.004889   -0.009358   -0.008212   -0.006594   -0.088846   0.170065    0.061527    0.055814    0.647108    0.254443    0.695334
   2.1000  2.956452    2.607204    -0.302444   -0.175029   -0.004635   -0.009644   -0.008408   -0.006870   -0.085729   0.162500    0.091491    0.093258    0.656329    0.240549    0.699021
   2.2000  2.901685    2.624680    -0.284117   -0.193166   -0.004464   -0.009569   -0.008262   -0.006733   -0.081777   0.151522    0.104528    0.110614    0.660777    0.230285    0.699755
   2.3000  2.813618    2.627550    -0.268304   -0.204068   -0.004394   -0.009455   -0.008090   -0.006665   -0.081409   0.144908    0.109549    0.129229    0.661932    0.226109    0.699485
   2.4000  3.074816    2.640652    -0.252432   -0.296135   -0.004003   -0.008974   -0.007624   -0.006186   -0.080854   0.119768    0.095382    0.117497    0.663796    0.214714    0.697658
   2.5000  2.807016    2.640416    -0.238659   -0.267752   -0.003614   -0.009039   -0.007689   -0.006335   -0.079027   0.082256    0.093179    0.124254    0.668324    0.208346    0.700046
   2.6000  2.904319    2.643930    -0.224966   -0.316500   -0.003303   -0.008770   -0.007338   -0.006019   -0.077719   0.047052    0.065673    0.096888    0.676112    0.201788    0.705581
   2.7000  2.875456    2.645508    -0.214820   -0.332043   -0.002971   -0.008667   -0.007238   -0.005922   -0.075891   0.001728    0.038158    0.072240    0.682245    0.198367    0.710498
   2.8000  2.794017    2.646570    -0.203554   -0.336384   -0.002743   -0.008603   -0.007175   -0.005889   -0.075500   -0.022875   0.025633    0.064037    0.689871    0.196452    0.717297
   2.9000  2.522889    2.643149    -0.190522   -0.303117   -0.002634   -0.008660   -0.007322   -0.005997   -0.075814   -0.020022   0.040952    0.081952    0.695800    0.194110    0.722369
   3.0000  2.312946    2.648337    -0.175360   -0.278351   -0.002380   -0.008692   -0.007397   -0.006100   -0.076898   -0.049648   0.024080    0.070438    0.701986    0.192013    0.727773
   3.1000  2.151450    2.650207    -0.160228   -0.263009   -0.002268   -0.008633   -0.007413   -0.006112   -0.077856   -0.115408   0.008491    0.057257    0.704771    0.185422    0.728755
   3.2000  2.056625    2.648915    -0.149747   -0.262495   -0.002000   -0.008587   -0.007414   -0.006122   -0.077166   -0.144040   -0.006955   0.047906    0.707167    0.183832    0.730670
   3.3000  5.269873    2.650354    -0.135319   -1.031899   -0.001053   -0.004761   -0.004093   -0.002263   -0.076361   -0.301840   -0.138659   -0.072602   0.717227    0.184126    0.740484
   3.4000  5.762147    2.645678    -0.124965   -1.167761   -0.000678   -0.004030   -0.003429   -0.001564   -0.078144   -0.362487   -0.180988   -0.112920   0.727447    0.184205    0.750407
   3.5000  5.131832    2.638281    -0.116809   -1.040917   -0.000575   -0.004605   -0.004028   -0.002170   -0.077762   -0.369328   -0.177531   -0.107450   0.725887    0.181784    0.748302
   3.6000  5.021665    2.636266    -0.103422   -1.035700   -0.000491   -0.004530   -0.004006   -0.002101   -0.076775   -0.396029   -0.182224   -0.109471   0.733950    0.179405    0.755558
   3.7000  4.835488    2.636399    -0.089750   -1.018651   -0.000303   -0.004526   -0.004038   -0.002126   -0.077561   -0.392749   -0.172060   -0.095269   0.741616    0.177857    0.762645
   3.8000  4.627948    2.636029    -0.076824   -0.994568   -0.000150   -0.004543   -0.004103   -0.002203   -0.078754   -0.399887   -0.161466   -0.082044   0.744687    0.177263    0.765494
   3.9000  1.121060    2.633780    -0.066457   -0.196526   -0.000691   -0.008375   -0.007568   -0.006192   -0.079827   -0.269073   -0.050528   0.024293    0.765880    0.177242    0.786121
   4.0000  0.952929    2.636896    -0.049726   -0.180697   -0.000491   -0.008384   -0.007596   -0.006219   -0.080341   -0.273917   -0.048731   0.029321    0.774264    0.177968    0.794454
   4.1000  0.961349    2.648192    -0.023919   -0.203557   -0.000302   -0.008256   -0.007453   -0.006119   -0.080701   -0.293124   -0.047317   0.033262    0.778548    0.182263    0.799598
   4.2000  0.828220    2.661316    0.000315    -0.191451   -0.000151   -0.008255   -0.007463   -0.006138   -0.080785   -0.302693   -0.051699   0.032576    0.786332    0.182392    0.807208
   4.3000  0.839092    2.656932    0.009474    -0.214055   0.000061    -0.008125   -0.007333   -0.006007   -0.079667   -0.304442   -0.049018   0.041007    0.796289    0.179591    0.816289
   4.4000  0.804939    2.651500    0.016900    -0.223459   0.000293    -0.008056   -0.007263   -0.005954   -0.079516   -0.328959   -0.056073   0.037063    0.797705    0.176905    0.817085
   4.5000  0.600590    2.643826    0.023140    -0.191230   0.000471    -0.008172   -0.007384   -0.006121   -0.080837   -0.346529   -0.065766   0.030399    0.805124    0.174718    0.823863
   4.6000  0.609642    2.634749    0.031878    -0.207898   0.000557    -0.008015   -0.007214   -0.005997   -0.081246   -0.378450   -0.082195   0.017602    0.807041    0.176093    0.826029
   4.7000  0.555030    2.624061    0.036834    -0.208140   0.000644    -0.008006   -0.007193   -0.005984   -0.081187   -0.392058   -0.087627   0.012655    0.811518    0.176248    0.830437
   4.8000  0.405500    2.614079    0.042089    -0.187945   0.000750    -0.008050   -0.007258   -0.006046   -0.081087   -0.404887   -0.092405   0.008914    0.816722    0.176637    0.835605
   4.9000  0.242533    2.604766    0.048724    -0.165201   0.000853    -0.008104   -0.007351   -0.006131   -0.081669   -0.412782   -0.095936   0.006237    0.822804    0.176873    0.841600
   5.0000  0.034662    2.595820    0.055424    -0.133164   0.000955    -0.008179   -0.007481   -0.006264   -0.082231   -0.415797   -0.092658   0.012042    0.827872    0.176553    0.846488
   5.1000  -0.081979   2.587119    0.061763    -0.123442   0.001090    -0.008151   -0.007499   -0.006276   -0.082921   -0.429938   -0.089073   0.016195    0.834572    0.176364    0.853003
   5.2000  -0.249066   2.577884    0.068770    -0.101407   0.001212    -0.008234   -0.007583   -0.006383   -0.082476   -0.424205   -0.079496   0.028762    0.841251    0.174192    0.859096
   5.3000  -0.338950   2.568046    0.076142    -0.095754   0.001287    -0.008226   -0.007585   -0.006387   -0.082091   -0.423397   -0.075478   0.035186    0.847640    0.171850    0.864884
   5.4000  -0.623212   2.560750    0.079686    -0.049798   0.001409    -0.008385   -0.007780   -0.006543   -0.081040   -0.388027   -0.052914   0.056856    0.853866    0.171347    0.870888
   5.5000  -0.604771   2.550260    0.090096    -0.072397   0.001594    -0.008266   -0.007651   -0.006458   -0.080097   -0.402135   -0.043697   0.071949    0.859402    0.172879    0.876618
   5.6000  -0.639384   2.538733    0.101040    -0.081970   0.001653    -0.008155   -0.007577   -0.006371   -0.080495   -0.418025   -0.037315   0.081465    0.865244    0.164453    0.880734
   5.7000  -0.739179   2.530967    0.108785    -0.077048   0.001833    -0.008144   -0.007577   -0.006394   -0.080578   -0.411862   -0.024578   0.094688    0.866586    0.163912    0.881951
   5.8000  -0.892723   2.523105    0.116266    -0.057205   0.001919    -0.008164   -0.007637   -0.006448   -0.080507   -0.431490   -0.015450   0.103633    0.869301    0.161580    0.884190
   5.9000  -1.315659   2.513765    0.124501    0.023737    0.001937    -0.008465   -0.007971   -0.006799   -0.080693   -0.415834   0.004600    0.125270    0.870787    0.155898    0.884633
   6.0000  -1.410412   2.506959    0.131751    0.029681    0.002057    -0.008441   -0.007970   -0.006791   -0.080021   -0.411275   0.014911    0.137719    0.872233    0.153245    0.885593
   6.1000  -1.779048   2.499572    0.137823    0.114078    0.002217    -0.008957   -0.008405   -0.007260   -0.079338   -0.417246   -0.018151   0.101433    0.865988    0.149447    0.878789
   6.2000  -1.949350   2.492869    0.144747    0.135361    0.002337    -0.008991   -0.008466   -0.007318   -0.079277   -0.418337   0.001653    0.119978    0.868436    0.147816    0.880926
   6.3000  -2.040512   2.485843    0.151142    0.139510    0.002439    -0.008988   -0.008468   -0.007316   -0.079847   -0.417087   0.021930    0.138143    0.871654    0.145691    0.883745
   6.4000  -2.061646   2.478410    0.158019    0.131269    0.002506    -0.008911   -0.008406   -0.007245   -0.079136   -0.417063   0.030345    0.146480    0.875096    0.143556    0.886792
   6.5000  -2.146219   2.470837    0.165034    0.139223    0.002571    -0.008901   -0.008427   -0.007265   -0.078674   -0.423313   0.034515    0.148782    0.874003    0.142450    0.885536
   6.6000  -2.152551   2.463297    0.171719    0.128148    0.002679    -0.008801   -0.008348   -0.007163   -0.077909   -0.446100   0.032629    0.148052    0.876552    0.140831    0.887793
   6.7000  -2.138325   2.456315    0.177916    0.112966    0.002837    -0.008744   -0.008285   -0.007098   -0.076566   -0.438987   0.035094    0.152092    0.877505    0.140113    0.888621
   6.8000  -2.160192   2.448234    0.184721    0.105756    0.002990    -0.008751   -0.008276   -0.007083   -0.074384   -0.423937   0.044650    0.164666    0.881340    0.140697    0.892500
   6.9000  -2.290402   2.441229    0.190884    0.124898    0.003075    -0.008846   -0.008364   -0.007175   -0.072423   -0.418936   0.054664    0.174472    0.883473    0.139439    0.894409
   7.0000  -2.397502   2.434074    0.196971    0.135832    0.003188    -0.008829   -0.008378   -0.007214   -0.072835   -0.419921   0.061951    0.182703    0.885949    0.139733    0.896901
   7.1000  -2.436700   2.427000    0.203163    0.134903    0.003279    -0.008789   -0.008356   -0.007189   -0.071864   -0.430437   0.060083    0.180965    0.886775    0.139301    0.897650
   7.2000  -2.549058   2.419390    0.208920    0.149705    0.003352    -0.008836   -0.008392   -0.007226   -0.072061   -0.434239   0.062306    0.181259    0.890337    0.139268    0.901163
   7.3000  -2.357014   2.413523    0.215577    0.093224    0.003566    -0.008613   -0.008130   -0.006958   -0.070514   -0.436228   0.058741    0.181447    0.891032    0.138999    0.901809
   7.4000  -2.423501   2.405897    0.220664    0.100303    0.003603    -0.008628   -0.008152   -0.006979   -0.069940   -0.441199   0.060592    0.181813    0.894874    0.139167    0.905630
   7.5000  -2.453612   2.398102    0.226071    0.099083    0.003640    -0.008601   -0.008122   -0.006968   -0.069555   -0.446268   0.060839    0.183217    0.896089    0.139740    0.906920
   7.6000  -2.603462   2.391606    0.230925    0.120985    0.003740    -0.008659   -0.008191   -0.007049   -0.070554   -0.456967   0.066954    0.189882    0.890583    0.140775    0.901641
   7.7000  -2.679675   2.386746    0.234973    0.123845    0.003833    -0.008630   -0.008183   -0.007040   -0.070766   -0.472204   0.086189    0.209543    0.892594    0.146015    0.904458
   7.8000  -2.691767   2.379787    0.240702    0.117456    0.003919    -0.008607   -0.008172   -0.007031   -0.069837   -0.458075   0.092924    0.217087    0.894976    0.146680    0.906917
   7.9000  -2.781496   2.371842    0.246096    0.131388    0.003890    -0.008613   -0.008213   -0.007030   -0.069085   -0.479308   0.089821    0.211131    0.896757    0.147023    0.908729
   8.0000  -2.901346   2.365853    0.250960    0.149531    0.004011    -0.008698   -0.008305   -0.007111   -0.068199   -0.467973   0.091263    0.213052    0.901146    0.146867    0.913035
   8.1000  -3.045008   2.359509    0.256370    0.172975    0.004162    -0.008885   -0.008466   -0.007291   -0.066343   -0.456402   0.106187    0.230562    0.895943    0.147787    0.908050
   8.2000  -3.092243   2.352116    0.260626    0.179196    0.004140    -0.008868   -0.008461   -0.007274   -0.065301   -0.469387   0.098897    0.221791    0.896335    0.147584    0.908404
   8.3000  -3.097478   2.344849    0.264811    0.174304    0.004165    -0.008817   -0.008424   -0.007231   -0.064763   -0.469697   0.093763    0.218110    0.896688    0.147747    0.908778
   8.4000  -3.185070   2.338723    0.269213    0.186137    0.004242    -0.008827   -0.008464   -0.007267   -0.064400   -0.478334   0.093348    0.216641    0.897821    0.148492    0.910018
   8.5000  -3.350357   2.331819    0.273119    0.218186    0.004231    -0.008941   -0.008593   -0.007402   -0.063994   -0.483229   0.094545    0.217275    0.898920    0.149444    0.911258
   8.6000  -3.330392   2.328035    0.280075    0.175555    0.004659    -0.008753   -0.008392   -0.007198   -0.064588   -0.420103   0.178036    0.302854    0.889291    0.152424    0.902259
   8.7000  -3.392340   2.321025    0.284445    0.180877    0.004717    -0.008749   -0.008407   -0.007202   -0.063835   -0.421852   0.183748    0.309511    0.888169    0.153201    0.901285
   8.8000  -3.456748   2.313342    0.288613    0.189188    0.004737    -0.008763   -0.008437   -0.007206   -0.061785   -0.428406   0.189497    0.312399    0.889024    0.151440    0.901830
   8.9000  -3.434744   2.306339    0.292976    0.174187    0.004809    -0.008661   -0.008357   -0.007127   -0.062125   -0.438823   0.195710    0.319989    0.887006    0.152840    0.900078
   9.0000  -3.433326   2.299317    0.297524    0.165413    0.004876    -0.008620   -0.008324   -0.007081   -0.061014   -0.439115   0.204649    0.327306    0.884734    0.152638    0.897804
   9.1000  -3.491904   2.291504    0.302632    0.168939    0.004913    -0.008595   -0.008323   -0.007077   -0.060686   -0.435906   0.216078    0.338931    0.878472    0.151907    0.891510
   9.2000  -3.511270   2.284402    0.307237    0.165634    0.004941    -0.008562   -0.008291   -0.007048   -0.060177   -0.429080   0.220544    0.343661    0.878234    0.152354    0.891352
   9.3000  -3.453968   2.277751    0.311614    0.143382    0.005018    -0.008466   -0.008162   -0.006922   -0.060223   -0.427530   0.222052    0.345689    0.877773    0.153055    0.891017
   9.4000  -3.540874   2.270059    0.315538    0.157709    0.004975    -0.008506   -0.008205   -0.006987   -0.059648   -0.419388   0.230555    0.352981    0.874911    0.153508    0.888276
   9.5000  -3.566015   2.262472    0.319397    0.157093    0.004961    -0.008492   -0.008186   -0.006992   -0.059754   -0.403957   0.237209    0.360211    0.874452    0.153758    0.887867
   9.6000  -3.703056   2.253803    0.322444    0.186570    0.004812    -0.008614   -0.008316   -0.007140   -0.058755   -0.403818   0.250112    0.369341    0.877059    0.154630    0.890586
   9.7000  -3.702683   2.247372    0.326258    0.179027    0.004860    -0.008600   -0.008268   -0.007091   -0.057804   -0.402150   0.257712    0.374427    0.879543    0.155256    0.893141
   9.8000  -3.670173   2.240554    0.329974    0.164013    0.004901    -0.008512   -0.008185   -0.007010   -0.057201   -0.394873   0.263624    0.380283    0.881711    0.156561    0.895503
   9.9000  -3.665753   2.233205    0.334535    0.154174    0.004926    -0.008418   -0.008095   -0.006933   -0.058529   -0.412660   0.265804    0.382110    0.881617    0.157471    0.895570
   10.000  -3.589180   2.227312    0.337769    0.127973    0.005004    -0.008280   -0.007953   -0.006790   -0.057341   -0.404070   0.270101    0.390929    0.884076    0.157986    0.898081
             """)
