>>> import jsonlines
>>> import pandas as pd
>>> with jsonlines.open('experiments.ljson') as fin:
...     lines = list(fin)
...
>>> df = pd.DataFrame(lines)
>>> df.nlayers
>>> df.num_layers
0     1
1     2
2     3
3     5
4     1
5     2
6     3
7     5
8     1
9     2
10    3
11    5
12    1
13    2
14    3
15    5
16    1
17    2
18    3
19    5
20    1
21    2
22    3
23    5
24    1
25    2
26    3
27    5
28    1
29    2
30    3
31    5
32    1
33    2
34    3
35    5
36    1
37    2
38    3
39    5
40    1
41    2
42    3
43    5
44    1
45    2
46    3
47    5
48    1
49    2
50    3
51    5
52    1
Name: num_layers, dtype: int64
>>> df.epochs
0      1
1      1
2      1
3      1
4      1
5      1
6      1
7      1
8      1
9      1
10     1
11     1
12     1
13     1
14     1
15     1
16     1
17     1
18     1
19     1
20     1
21     1
22     1
23     1
24    12
25    12
26    12
27    12
28    12
29    12
30    12
31    12
32    12
33    12
34    12
35    12
36    12
37    12
38    12
39    12
40    12
41    12
42    12
43    12
44    12
45    12
46    12
47    12
48    32
49    32
50    32
51    32
52    32
Name: epochs, dtype: int64
>>> df.sort_values('test_loss')
    annealing_loss_improvement_pct  batch_size  bptt  ...  val_perplexity  test_loss test_perplexity
52                             1.0          20    35  ...      177.802131   5.101517      164.270880
50                             1.0          20    35  ...      194.838210   5.196208      180.586133
38                             1.0          20    35  ...      198.210331   5.207420      182.622272
49                             1.0          20    35  ...      199.311340   5.215048      184.020707
37                             1.0          20    35  ...      203.976231   5.228101      186.438438
28                             1.0          20    35  ...      204.690330   5.228490      186.510935
30                             1.0          20    35  ...      207.963855   5.256755      191.857883
29                             1.0          20    35  ...      209.857287   5.267297      193.891097
31                             1.0          20    35  ...      212.702379   5.276428      195.669731
48                             1.0          20    35  ...      217.853971   5.294482      199.234413
51                             1.0          20    35  ...      216.413480   5.295230      199.383573
39                             1.0          20    35  ...      218.442620   5.306296      201.602045
36                             1.0          20    35  ...      224.973002   5.321423      204.674880
44                             1.0          20    35  ...      242.052660   5.398864      221.154958
45                             1.0          20    35  ...      251.828183   5.430191      228.192723
46                             1.0          20    35  ...      259.405820   5.471271      237.762291
33                             1.0          20    35  ...      261.535547   5.484255      240.869530
34                             1.0          20    35  ...      261.819516   5.486270      241.355315
26                             1.0          20    35  ...      265.634997   5.504034      245.681083
47                             1.0          20    35  ...      273.255506   5.525502      251.012319
25                             1.0          20    35  ...      274.597171   5.530284      252.215519
35                             1.0          20    35  ...      279.079548   5.553082      258.031624
41                             1.0          20    35  ...      294.237635   5.603401      271.347757
32                             1.0          20    35  ...      299.449529   5.621965      276.431995
27                             1.0          20    35  ...      301.153885   5.628237      278.171189
42                             1.0          20    35  ...      303.189960   5.635567      280.217711
40                             1.0          20    35  ...      310.416470   5.655370      285.822098
24                             1.0          20    35  ...      317.040789   5.680450      293.081403
43                             1.0          20    35  ...      320.910744   5.693568      296.951166
13                             1.0          20    35  ...      439.133338   5.999726      403.318467
5                              1.0          20    35  ...      449.150789   6.021574      412.226988
14                             1.0          20    35  ...      463.501280   6.055093      426.278573
6                              1.0          20    35  ...      472.775357   6.074285      434.538736
21                             1.0          20    35  ...      476.872072   6.080334      437.175260
22                             1.0          20    35  ...      485.805748   6.103339      447.348791
4                              1.0          20    35  ...      485.917391   6.109470      450.099917
12                             1.0          20    35  ...      498.185914   6.132175      460.436378
7                              1.0          20    35  ...      502.913230   6.134720      461.609970
20                             1.0          20    35  ...      500.455889   6.138457      463.337950
15                             1.0          20    35  ...      513.819828   6.158565      472.749048
23                             1.0          20    35  ...      548.536685   6.229937      507.723572
10                             1.0          20    35  ...      630.762516   6.373701      586.223308
17                             1.0          20    35  ...      645.871846   6.393924      598.199474
18                             1.0          20    35  ...      653.707452   6.408444      606.948531
11                             1.0          20    35  ...      678.318165   6.452393      634.218112
8                              1.0          20    35  ...     1021.839217   6.454904      635.812391
16                             1.0          20    35  ...      698.041867   6.475868      649.282838
19                             1.0          20    35  ...      698.347867   6.478120      650.746515
0                              1.0          20    35  ...      727.693280   6.519769      678.421797
2                              1.0          20    35  ...      838.446364   6.661679      781.862929
9                              1.0          20    35  ...     1803.304547   6.713112      823.128027
1                              1.0          20    35  ...     1451.106151   6.837989      932.611851
3                              1.0          20    35  ...     1054.820222   6.897221      989.521082

[53 rows x 29 columns]
>>> df[df.columns[3:]].sort_values('test_loss')
    clip  cuda           datapath device  ...  val_loss  val_perplexity  test_loss  test_perplexity
52  0.25  True  ./data/wikitext-2         ...  5.180671      177.802131   5.101517       164.270880
50  0.25  True  ./data/wikitext-2         ...  5.272170      194.838210   5.196208       180.586133
38  0.25  True  ./data/wikitext-2         ...  5.289329      198.210331   5.207420       182.622272
49  0.25  True  ./data/wikitext-2         ...  5.294868      199.311340   5.215048       184.020707
37  0.25  True  ./data/wikitext-2         ...  5.318003      203.976231   5.228101       186.438438
28  0.25  True  ./data/wikitext-2         ...  5.321498      204.690330   5.228490       186.510935
30  0.25  True  ./data/wikitext-2         ...  5.337364      207.963855   5.256755       191.857883
29  0.25  True  ./data/wikitext-2         ...  5.346428      209.857287   5.267297       193.891097
31  0.25  True  ./data/wikitext-2         ...  5.359894      212.702379   5.276428       195.669731
48  0.25  True  ./data/wikitext-2         ...  5.383825      217.853971   5.294482       199.234413
51  0.25  True  ./data/wikitext-2         ...  5.377191      216.413480   5.295230       199.383573
39  0.25  True  ./data/wikitext-2         ...  5.386523      218.442620   5.306296       201.602045
36  0.25  True  ./data/wikitext-2         ...  5.415980      224.973002   5.321423       204.674880
44  0.25  True  ./data/wikitext-2         ...  5.489155      242.052660   5.398864       221.154958
45  0.25  True  ./data/wikitext-2         ...  5.528747      251.828183   5.430191       228.192723
46  0.25  True  ./data/wikitext-2         ...  5.558394      259.405820   5.471271       237.762291
33  0.25  True  ./data/wikitext-2         ...  5.566570      261.535547   5.484255       240.869530
34  0.25  True  ./data/wikitext-2         ...  5.567655      261.819516   5.486270       241.355315
26  0.25  True  ./data/wikitext-2         ...  5.582123      265.634997   5.504034       245.681083
47  0.25  True  ./data/wikitext-2         ...  5.610407      273.255506   5.525502       251.012319
25  0.25  True  ./data/wikitext-2         ...  5.615305      274.597171   5.530284       252.215519
35  0.25  True  ./data/wikitext-2         ...  5.631497      279.079548   5.553082       258.031624
41  0.25  True  ./data/wikitext-2         ...  5.684388      294.237635   5.603401       271.347757
32  0.25  True  ./data/wikitext-2         ...  5.701946      299.449529   5.621965       276.431995
27  0.25  True  ./data/wikitext-2         ...  5.707621      301.153885   5.628237       278.171189
42  0.25  True  ./data/wikitext-2         ...  5.714360      303.189960   5.635567       280.217711
40  0.25  True  ./data/wikitext-2         ...  5.737915      310.416470   5.655370       285.822098
24  0.25  True  ./data/wikitext-2         ...  5.759030      317.040789   5.680450       293.081403
43  0.25  True  ./data/wikitext-2         ...  5.771163      320.910744   5.693568       296.951166
13  0.25  True  ./data/wikitext-2         ...  6.084803      439.133338   5.999726       403.318467
5   0.25  True  ./data/wikitext-2         ...  6.107359      449.150789   6.021574       412.226988
14  0.25  True  ./data/wikitext-2         ...  6.138809      463.501280   6.055093       426.278573
6   0.25  True  ./data/wikitext-2         ...  6.158620      472.775357   6.074285       434.538736
21  0.25  True  ./data/wikitext-2         ...  6.167248      476.872072   6.080334       437.175260
22  0.25  True  ./data/wikitext-2         ...  6.185809      485.805748   6.103339       447.348791
4   0.25  True  ./data/wikitext-2         ...  6.186039      485.917391   6.109470       450.099917
12  0.25  True  ./data/wikitext-2         ...  6.210973      498.185914   6.132175       460.436378
7   0.25  True  ./data/wikitext-2         ...  6.220418      502.913230   6.134720       461.609970
20  0.25  True  ./data/wikitext-2         ...  6.215519      500.455889   6.138457       463.337950
15  0.25  True  ./data/wikitext-2         ...  6.241873      513.819828   6.158565       472.749048
23  0.25  True  ./data/wikitext-2         ...  6.307254      548.536685   6.229937       507.723572
10  0.25  True  ./data/wikitext-2         ...  6.446929      630.762516   6.373701       586.223308
17  0.25  True  ./data/wikitext-2         ...  6.470601      645.871846   6.393924       598.199474
18  0.25  True  ./data/wikitext-2         ...  6.482660      653.707452   6.408444       606.948531
11  0.25  True  ./data/wikitext-2         ...  6.519616      678.318165   6.452393       634.218112
8   0.25  True  ./data/wikitext-2         ...  6.929359     1021.839217   6.454904       635.812391
16  0.25  True  ./data/wikitext-2         ...  6.548279      698.041867   6.475868       649.282838
19  0.25  True  ./data/wikitext-2         ...  6.548717      698.347867   6.478120       650.746515
0   0.25  True  ./data/wikitext-2         ...  6.589880      727.693280   6.519769       678.421797
2   0.25  True  ./data/wikitext-2         ...  6.731551      838.446364   6.661679       781.862929
9   0.25  True  ./data/wikitext-2         ...  7.497376     1803.304547   6.713112       823.128027
1   0.25  True  ./data/wikitext-2         ...  7.280081     1451.106151   6.837989       932.611851
3   0.25  True  ./data/wikitext-2         ...  6.961126     1054.820222   6.897221       989.521082

[53 rows x 26 columns]
>>> interesting_columns = [c for c in df.columns if df[c].nunique > 1]
>>> interesting_columns = [c for c in df.columns if df[c].nunique() > 1]
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  ...  val_loss  val_perplexity  test_loss  test_perplexity
3       0.0       1  0.5           5  ...  6.961126     1054.820222   6.897221       989.521082
1       0.0       1  0.5           2  ...  7.280081     1451.106151   6.837989       932.611851
9       0.2       1  0.5           2  ...  7.497376     1803.304547   6.713112       823.128027
2       0.0       1  0.5           3  ...  6.731551      838.446364   6.661679       781.862929
0       0.0       1  0.5           1  ...  6.589880      727.693280   6.519769       678.421797
19      0.5       1  0.5           5  ...  6.548717      698.347867   6.478120       650.746515
16      0.5       1  0.5           1  ...  6.548279      698.041867   6.475868       649.282838
8       0.2       1  0.5           1  ...  6.929359     1021.839217   6.454904       635.812391
11      0.2       1  0.5           5  ...  6.519616      678.318165   6.452393       634.218112
18      0.5       1  0.5           3  ...  6.482660      653.707452   6.408444       606.948531
17      0.5       1  0.5           2  ...  6.470601      645.871846   6.393924       598.199474
10      0.2       1  0.5           3  ...  6.446929      630.762516   6.373701       586.223308
23      0.5       1  2.0           5  ...  6.307254      548.536685   6.229937       507.723572
15      0.2       1  2.0           5  ...  6.241873      513.819828   6.158565       472.749048
20      0.5       1  2.0           1  ...  6.215519      500.455889   6.138457       463.337950
7       0.0       1  2.0           5  ...  6.220418      502.913230   6.134720       461.609970
12      0.2       1  2.0           1  ...  6.210973      498.185914   6.132175       460.436378
4       0.0       1  2.0           1  ...  6.186039      485.917391   6.109470       450.099917
22      0.5       1  2.0           3  ...  6.185809      485.805748   6.103339       447.348791
21      0.5       1  2.0           2  ...  6.167248      476.872072   6.080334       437.175260
6       0.0       1  2.0           3  ...  6.158620      472.775357   6.074285       434.538736
14      0.2       1  2.0           3  ...  6.138809      463.501280   6.055093       426.278573
5       0.0       1  2.0           2  ...  6.107359      449.150789   6.021574       412.226988
13      0.2       1  2.0           2  ...  6.084803      439.133338   5.999726       403.318467
43      0.5      12  0.5           5  ...  5.771163      320.910744   5.693568       296.951166
24      0.0      12  0.5           1  ...  5.759030      317.040789   5.680450       293.081403
40      0.5      12  0.5           1  ...  5.737915      310.416470   5.655370       285.822098
42      0.5      12  0.5           3  ...  5.714360      303.189960   5.635567       280.217711
27      0.0      12  0.5           5  ...  5.707621      301.153885   5.628237       278.171189
32      0.2      12  0.5           1  ...  5.701946      299.449529   5.621965       276.431995
41      0.5      12  0.5           2  ...  5.684388      294.237635   5.603401       271.347757
35      0.2      12  0.5           5  ...  5.631497      279.079548   5.553082       258.031624
25      0.0      12  0.5           2  ...  5.615305      274.597171   5.530284       252.215519
47      0.5      12  2.0           5  ...  5.610407      273.255506   5.525502       251.012319
26      0.0      12  0.5           3  ...  5.582123      265.634997   5.504034       245.681083
34      0.2      12  0.5           3  ...  5.567655      261.819516   5.486270       241.355315
33      0.2      12  0.5           2  ...  5.566570      261.535547   5.484255       240.869530
46      0.5      12  2.0           3  ...  5.558394      259.405820   5.471271       237.762291
45      0.5      12  2.0           2  ...  5.528747      251.828183   5.430191       228.192723
44      0.5      12  2.0           1  ...  5.489155      242.052660   5.398864       221.154958
36      0.2      12  2.0           1  ...  5.415980      224.973002   5.321423       204.674880
39      0.2      12  2.0           5  ...  5.386523      218.442620   5.306296       201.602045
51      0.0      32  0.5           5  ...  5.377191      216.413480   5.295230       199.383573
48      0.0      32  0.5           1  ...  5.383825      217.853971   5.294482       199.234413
31      0.0      12  2.0           5  ...  5.359894      212.702379   5.276428       195.669731
29      0.0      12  2.0           2  ...  5.346428      209.857287   5.267297       193.891097
30      0.0      12  2.0           3  ...  5.337364      207.963855   5.256755       191.857883
28      0.0      12  2.0           1  ...  5.321498      204.690330   5.228490       186.510935
37      0.2      12  2.0           2  ...  5.318003      203.976231   5.228101       186.438438
49      0.0      32  0.5           2  ...  5.294868      199.311340   5.215048       184.020707
38      0.2      12  2.0           3  ...  5.289329      198.210331   5.207420       182.622272
50      0.0      32  0.5           3  ...  5.272170      194.838210   5.196208       180.586133
52      0.0      32  2.0           1  ...  5.180671      177.802131   5.101517       164.270880

[53 rows x 12 columns]
>>> pd.options.display.max_columns = 14
>>> pd.options.display.max_columns = 14
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  \
3       0.0       1  0.5           5   
1       0.0       1  0.5           2   
9       0.2       1  0.5           2   
2       0.0       1  0.5           3   
0       0.0       1  0.5           1   
19      0.5       1  0.5           5   
16      0.5       1  0.5           1   
8       0.2       1  0.5           1   
11      0.2       1  0.5           5   
18      0.5       1  0.5           3   
17      0.5       1  0.5           2   
10      0.2       1  0.5           3   
23      0.5       1  2.0           5   
15      0.2       1  2.0           5   
20      0.5       1  2.0           1   
7       0.0       1  2.0           5   
12      0.2       1  2.0           1   
4       0.0       1  2.0           1   
22      0.5       1  2.0           3   
21      0.5       1  2.0           2   
6       0.0       1  2.0           3   
14      0.2       1  2.0           3   
5       0.0       1  2.0           2   
13      0.2       1  2.0           2   
43      0.5      12  0.5           5   
24      0.0      12  0.5           1   
40      0.5      12  0.5           1   
42      0.5      12  0.5           3   
27      0.0      12  0.5           5   
32      0.2      12  0.5           1   
41      0.5      12  0.5           2   
35      0.2      12  0.5           5   
25      0.0      12  0.5           2   
47      0.5      12  2.0           5   
26      0.0      12  0.5           3   
34      0.2      12  0.5           3   
33      0.2      12  0.5           2   
46      0.5      12  2.0           3   
45      0.5      12  2.0           2   
44      0.5      12  2.0           1   
36      0.2      12  2.0           1   
39      0.2      12  2.0           5   
51      0.0      32  0.5           5   
48      0.0      32  0.5           1   
31      0.0      12  2.0           5   
29      0.0      12  2.0           2   
30      0.0      12  2.0           3   
28      0.0      12  2.0           1   
37      0.2      12  2.0           2   
49      0.0      32  0.5           2   
38      0.2      12  2.0           3   
50      0.0      32  0.5           3   
52      0.0      32  2.0           1   

                                             filename  best_val_loss  \
3   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
1   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
9   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
2   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
0   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
19  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
16  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
8   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
11  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
18  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
17  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
10  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
23  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
15  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
20  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
7   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
12  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
4   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
22  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
21  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
6   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
14  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
5   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
13  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
43  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.803356   
24  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.764420   
40  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.769231   
42  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.732135   
27  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.746945   
32  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.721411   
41  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.714733   
35  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.657560   
25  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.647062   
47  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.650389   
26  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.616970   
34  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.595128   
33  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.598579   
46  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.551295   
45  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.517190   
44  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.493749   
36  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.427752   
39  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.386652   
51  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.386441   
48  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.380273   
31  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.365011   
29  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.352579   
30  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.352212   
28  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.440840   
37  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.333794   
49  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.321274   
38  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.401936   
50  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.274836   
52  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.181029   

    epoch_num  epoch_time  val_loss  val_perplexity  test_loss  \
3           1   55.457326  6.961126     1054.820222   6.897221   
1           1   32.106639  7.280081     1451.106151   6.837989   
9           1   35.513603  7.497376     1803.304547   6.713112   
2           1   38.042581  6.731551      838.446364   6.661679   
0           1   22.678797  6.589880      727.693280   6.519769   
19          1   67.410955  6.548717      698.347867   6.478120   
16          1   22.873118  6.548279      698.041867   6.475868   
8           1   22.738191  6.929359     1021.839217   6.454904   
11          1   67.846113  6.519616      678.318165   6.452393   
18          1   46.151242  6.482660      653.707452   6.408444   
17          1   35.783531  6.470601      645.871846   6.393924   
10          1   46.116142  6.446929      630.762516   6.373701   
23          1   67.591126  6.307254      548.536685   6.229937   
15          1   68.007911  6.241873      513.819828   6.158565   
20          1   22.870234  6.215519      500.455889   6.138457   
7           1   55.487360  6.220418      502.913230   6.134720   
12          1   22.753431  6.210973      498.185914   6.132175   
4           1   22.612858  6.186039      485.917391   6.109470   
22          1   46.234442  6.185809      485.805748   6.103339   
21          1   35.493594  6.167248      476.872072   6.080334   
6           1   38.298247  6.158620      472.775357   6.074285   
14          1   46.534271  6.138809      463.501280   6.055093   
5           1   32.427224  6.107359      449.150789   6.021574   
13          1   35.634995  6.084803      439.133338   5.999726   
43         12   67.546237  5.771163      320.910744   5.693568   
24         12   22.663883  5.759030      317.040789   5.680450   
40         12   22.722072  5.737915      310.416470   5.655370   
42         12   46.440262  5.714360      303.189960   5.635567   
27         12   56.313533  5.707621      301.153885   5.628237   
32         12   22.746994  5.701946      299.449529   5.621965   
41         12   35.532670  5.684388      294.237635   5.603401   
35         12   67.252183  5.631497      279.079548   5.553082   
25         12   32.404777  5.615305      274.597171   5.530284   
47         12   67.707756  5.610407      273.255506   5.525502   
26         12   38.795682  5.582123      265.634997   5.504034   
34         12   46.276322  5.567655      261.819516   5.486270   
33         12   35.548976  5.566570      261.535547   5.484255   
46         12   46.421073  5.558394      259.405820   5.471271   
45         12   35.522535  5.528747      251.828183   5.430191   
44         12   23.129305  5.489155      242.052660   5.398864   
36         12   22.752869  5.415980      224.973002   5.321423   
39         12   67.399188  5.386523      218.442620   5.306296   
51         32   56.175895  5.377191      216.413480   5.295230   
48         32   22.848217  5.383825      217.853971   5.294482   
31         12   54.730397  5.359894      212.702379   5.276428   
29         12   32.457682  5.346428      209.857287   5.267297   
30         12   38.802311  5.337364      207.963855   5.256755   
28         12   22.664476  5.321498      204.690330   5.228490   
37         12   35.434054  5.318003      203.976231   5.228101   
49         32   32.353001  5.294868      199.311340   5.215048   
38         12   46.135516  5.289329      198.210331   5.207420   
50         32   37.355055  5.272170      194.838210   5.196208   
52         32   22.903723  5.180671      177.802131   5.101517   

    test_perplexity  
3        989.521082  
1        932.611851  
9        823.128027  
2        781.862929  
0        678.421797  
19       650.746515  
16       649.282838  
8        635.812391  
11       634.218112  
18       606.948531  
17       598.199474  
10       586.223308  
23       507.723572  
15       472.749048  
20       463.337950  
7        461.609970  
12       460.436378  
4        450.099917  
22       447.348791  
21       437.175260  
6        434.538736  
14       426.278573  
5        412.226988  
13       403.318467  
43       296.951166  
24       293.081403  
40       285.822098  
42       280.217711  
27       278.171189  
32       276.431995  
41       271.347757  
35       258.031624  
25       252.215519  
47       251.012319  
26       245.681083  
34       241.355315  
33       240.869530  
46       237.762291  
45       228.192723  
44       221.154958  
36       204.674880  
39       201.602045  
51       199.383573  
48       199.234413  
31       195.669731  
29       193.891097  
30       191.857883  
28       186.510935  
37       186.438438  
49       184.020707  
38       182.622272  
50       180.586133  
52       164.270880  
>>> pd.options.display.max_columns = 13
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  \
3       0.0       1  0.5           5   
1       0.0       1  0.5           2   
9       0.2       1  0.5           2   
2       0.0       1  0.5           3   
0       0.0       1  0.5           1   
19      0.5       1  0.5           5   
16      0.5       1  0.5           1   
8       0.2       1  0.5           1   
11      0.2       1  0.5           5   
18      0.5       1  0.5           3   
17      0.5       1  0.5           2   
10      0.2       1  0.5           3   
23      0.5       1  2.0           5   
15      0.2       1  2.0           5   
20      0.5       1  2.0           1   
7       0.0       1  2.0           5   
12      0.2       1  2.0           1   
4       0.0       1  2.0           1   
22      0.5       1  2.0           3   
21      0.5       1  2.0           2   
6       0.0       1  2.0           3   
14      0.2       1  2.0           3   
5       0.0       1  2.0           2   
13      0.2       1  2.0           2   
43      0.5      12  0.5           5   
24      0.0      12  0.5           1   
40      0.5      12  0.5           1   
42      0.5      12  0.5           3   
27      0.0      12  0.5           5   
32      0.2      12  0.5           1   
41      0.5      12  0.5           2   
35      0.2      12  0.5           5   
25      0.0      12  0.5           2   
47      0.5      12  2.0           5   
26      0.0      12  0.5           3   
34      0.2      12  0.5           3   
33      0.2      12  0.5           2   
46      0.5      12  2.0           3   
45      0.5      12  2.0           2   
44      0.5      12  2.0           1   
36      0.2      12  2.0           1   
39      0.2      12  2.0           5   
51      0.0      32  0.5           5   
48      0.0      32  0.5           1   
31      0.0      12  2.0           5   
29      0.0      12  2.0           2   
30      0.0      12  2.0           3   
28      0.0      12  2.0           1   
37      0.2      12  2.0           2   
49      0.0      32  0.5           2   
38      0.2      12  2.0           3   
50      0.0      32  0.5           3   
52      0.0      32  2.0           1   

                                             filename  best_val_loss  \
3   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
1   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
9   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
2   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
0   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
19  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
16  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
8   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
11  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
18  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
17  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
10  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
23  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
15  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
20  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
7   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
12  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
4   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
22  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
21  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
6   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
14  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
5   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
13  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
43  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.803356   
24  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.764420   
40  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.769231   
42  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.732135   
27  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.746945   
32  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.721411   
41  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.714733   
35  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.657560   
25  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.647062   
47  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.650389   
26  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.616970   
34  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.595128   
33  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.598579   
46  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.551295   
45  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.517190   
44  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.493749   
36  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.427752   
39  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.386652   
51  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.386441   
48  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.380273   
31  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.365011   
29  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.352579   
30  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.352212   
28  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.440840   
37  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.333794   
49  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.321274   
38  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.401936   
50  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.274836   
52  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.181029   

    epoch_num  epoch_time  val_loss  val_perplexity  test_loss  \
3           1   55.457326  6.961126     1054.820222   6.897221   
1           1   32.106639  7.280081     1451.106151   6.837989   
9           1   35.513603  7.497376     1803.304547   6.713112   
2           1   38.042581  6.731551      838.446364   6.661679   
0           1   22.678797  6.589880      727.693280   6.519769   
19          1   67.410955  6.548717      698.347867   6.478120   
16          1   22.873118  6.548279      698.041867   6.475868   
8           1   22.738191  6.929359     1021.839217   6.454904   
11          1   67.846113  6.519616      678.318165   6.452393   
18          1   46.151242  6.482660      653.707452   6.408444   
17          1   35.783531  6.470601      645.871846   6.393924   
10          1   46.116142  6.446929      630.762516   6.373701   
23          1   67.591126  6.307254      548.536685   6.229937   
15          1   68.007911  6.241873      513.819828   6.158565   
20          1   22.870234  6.215519      500.455889   6.138457   
7           1   55.487360  6.220418      502.913230   6.134720   
12          1   22.753431  6.210973      498.185914   6.132175   
4           1   22.612858  6.186039      485.917391   6.109470   
22          1   46.234442  6.185809      485.805748   6.103339   
21          1   35.493594  6.167248      476.872072   6.080334   
6           1   38.298247  6.158620      472.775357   6.074285   
14          1   46.534271  6.138809      463.501280   6.055093   
5           1   32.427224  6.107359      449.150789   6.021574   
13          1   35.634995  6.084803      439.133338   5.999726   
43         12   67.546237  5.771163      320.910744   5.693568   
24         12   22.663883  5.759030      317.040789   5.680450   
40         12   22.722072  5.737915      310.416470   5.655370   
42         12   46.440262  5.714360      303.189960   5.635567   
27         12   56.313533  5.707621      301.153885   5.628237   
32         12   22.746994  5.701946      299.449529   5.621965   
41         12   35.532670  5.684388      294.237635   5.603401   
35         12   67.252183  5.631497      279.079548   5.553082   
25         12   32.404777  5.615305      274.597171   5.530284   
47         12   67.707756  5.610407      273.255506   5.525502   
26         12   38.795682  5.582123      265.634997   5.504034   
34         12   46.276322  5.567655      261.819516   5.486270   
33         12   35.548976  5.566570      261.535547   5.484255   
46         12   46.421073  5.558394      259.405820   5.471271   
45         12   35.522535  5.528747      251.828183   5.430191   
44         12   23.129305  5.489155      242.052660   5.398864   
36         12   22.752869  5.415980      224.973002   5.321423   
39         12   67.399188  5.386523      218.442620   5.306296   
51         32   56.175895  5.377191      216.413480   5.295230   
48         32   22.848217  5.383825      217.853971   5.294482   
31         12   54.730397  5.359894      212.702379   5.276428   
29         12   32.457682  5.346428      209.857287   5.267297   
30         12   38.802311  5.337364      207.963855   5.256755   
28         12   22.664476  5.321498      204.690330   5.228490   
37         12   35.434054  5.318003      203.976231   5.228101   
49         32   32.353001  5.294868      199.311340   5.215048   
38         12   46.135516  5.289329      198.210331   5.207420   
50         32   37.355055  5.272170      194.838210   5.196208   
52         32   22.903723  5.180671      177.802131   5.101517   

    test_perplexity  
3        989.521082  
1        932.611851  
9        823.128027  
2        781.862929  
0        678.421797  
19       650.746515  
16       649.282838  
8        635.812391  
11       634.218112  
18       606.948531  
17       598.199474  
10       586.223308  
23       507.723572  
15       472.749048  
20       463.337950  
7        461.609970  
12       460.436378  
4        450.099917  
22       447.348791  
21       437.175260  
6        434.538736  
14       426.278573  
5        412.226988  
13       403.318467  
43       296.951166  
24       293.081403  
40       285.822098  
42       280.217711  
27       278.171189  
32       276.431995  
41       271.347757  
35       258.031624  
25       252.215519  
47       251.012319  
26       245.681083  
34       241.355315  
33       240.869530  
46       237.762291  
45       228.192723  
44       221.154958  
36       204.674880  
39       201.602045  
51       199.383573  
48       199.234413  
31       195.669731  
29       193.891097  
30       191.857883  
28       186.510935  
37       186.438438  
49       184.020707  
38       182.622272  
50       180.586133  
52       164.270880  
>>> pd.options.display.max_columns = 12
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  \
3       0.0       1  0.5           5   
1       0.0       1  0.5           2   
9       0.2       1  0.5           2   
2       0.0       1  0.5           3   
0       0.0       1  0.5           1   
19      0.5       1  0.5           5   
16      0.5       1  0.5           1   
8       0.2       1  0.5           1   
11      0.2       1  0.5           5   
18      0.5       1  0.5           3   
17      0.5       1  0.5           2   
10      0.2       1  0.5           3   
23      0.5       1  2.0           5   
15      0.2       1  2.0           5   
20      0.5       1  2.0           1   
7       0.0       1  2.0           5   
12      0.2       1  2.0           1   
4       0.0       1  2.0           1   
22      0.5       1  2.0           3   
21      0.5       1  2.0           2   
6       0.0       1  2.0           3   
14      0.2       1  2.0           3   
5       0.0       1  2.0           2   
13      0.2       1  2.0           2   
43      0.5      12  0.5           5   
24      0.0      12  0.5           1   
40      0.5      12  0.5           1   
42      0.5      12  0.5           3   
27      0.0      12  0.5           5   
32      0.2      12  0.5           1   
41      0.5      12  0.5           2   
35      0.2      12  0.5           5   
25      0.0      12  0.5           2   
47      0.5      12  2.0           5   
26      0.0      12  0.5           3   
34      0.2      12  0.5           3   
33      0.2      12  0.5           2   
46      0.5      12  2.0           3   
45      0.5      12  2.0           2   
44      0.5      12  2.0           1   
36      0.2      12  2.0           1   
39      0.2      12  2.0           5   
51      0.0      32  0.5           5   
48      0.0      32  0.5           1   
31      0.0      12  2.0           5   
29      0.0      12  2.0           2   
30      0.0      12  2.0           3   
28      0.0      12  2.0           1   
37      0.2      12  2.0           2   
49      0.0      32  0.5           2   
38      0.2      12  2.0           3   
50      0.0      32  0.5           3   
52      0.0      32  2.0           1   

                                             filename  best_val_loss  \
3   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
1   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
9   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
2   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
0   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
19  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
16  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
8   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
11  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
18  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
17  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
10  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
23  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
15  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
20  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
7   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
12  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
4   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
22  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
21  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
6   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
14  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
5   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
13  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...            NaN   
43  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.803356   
24  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.764420   
40  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.769231   
42  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.732135   
27  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.746945   
32  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.721411   
41  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.714733   
35  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.657560   
25  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.647062   
47  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.650389   
26  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.616970   
34  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.595128   
33  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.598579   
46  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.551295   
45  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.517190   
44  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.493749   
36  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.427752   
39  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.386652   
51  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.386441   
48  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.380273   
31  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.365011   
29  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.352579   
30  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.352212   
28  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.440840   
37  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.333794   
49  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.321274   
38  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...       5.401936   
50  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.274836   
52  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...       5.181029   

    epoch_num  epoch_time  val_loss  val_perplexity  test_loss  \
3           1   55.457326  6.961126     1054.820222   6.897221   
1           1   32.106639  7.280081     1451.106151   6.837989   
9           1   35.513603  7.497376     1803.304547   6.713112   
2           1   38.042581  6.731551      838.446364   6.661679   
0           1   22.678797  6.589880      727.693280   6.519769   
19          1   67.410955  6.548717      698.347867   6.478120   
16          1   22.873118  6.548279      698.041867   6.475868   
8           1   22.738191  6.929359     1021.839217   6.454904   
11          1   67.846113  6.519616      678.318165   6.452393   
18          1   46.151242  6.482660      653.707452   6.408444   
17          1   35.783531  6.470601      645.871846   6.393924   
10          1   46.116142  6.446929      630.762516   6.373701   
23          1   67.591126  6.307254      548.536685   6.229937   
15          1   68.007911  6.241873      513.819828   6.158565   
20          1   22.870234  6.215519      500.455889   6.138457   
7           1   55.487360  6.220418      502.913230   6.134720   
12          1   22.753431  6.210973      498.185914   6.132175   
4           1   22.612858  6.186039      485.917391   6.109470   
22          1   46.234442  6.185809      485.805748   6.103339   
21          1   35.493594  6.167248      476.872072   6.080334   
6           1   38.298247  6.158620      472.775357   6.074285   
14          1   46.534271  6.138809      463.501280   6.055093   
5           1   32.427224  6.107359      449.150789   6.021574   
13          1   35.634995  6.084803      439.133338   5.999726   
43         12   67.546237  5.771163      320.910744   5.693568   
24         12   22.663883  5.759030      317.040789   5.680450   
40         12   22.722072  5.737915      310.416470   5.655370   
42         12   46.440262  5.714360      303.189960   5.635567   
27         12   56.313533  5.707621      301.153885   5.628237   
32         12   22.746994  5.701946      299.449529   5.621965   
41         12   35.532670  5.684388      294.237635   5.603401   
35         12   67.252183  5.631497      279.079548   5.553082   
25         12   32.404777  5.615305      274.597171   5.530284   
47         12   67.707756  5.610407      273.255506   5.525502   
26         12   38.795682  5.582123      265.634997   5.504034   
34         12   46.276322  5.567655      261.819516   5.486270   
33         12   35.548976  5.566570      261.535547   5.484255   
46         12   46.421073  5.558394      259.405820   5.471271   
45         12   35.522535  5.528747      251.828183   5.430191   
44         12   23.129305  5.489155      242.052660   5.398864   
36         12   22.752869  5.415980      224.973002   5.321423   
39         12   67.399188  5.386523      218.442620   5.306296   
51         32   56.175895  5.377191      216.413480   5.295230   
48         32   22.848217  5.383825      217.853971   5.294482   
31         12   54.730397  5.359894      212.702379   5.276428   
29         12   32.457682  5.346428      209.857287   5.267297   
30         12   38.802311  5.337364      207.963855   5.256755   
28         12   22.664476  5.321498      204.690330   5.228490   
37         12   35.434054  5.318003      203.976231   5.228101   
49         32   32.353001  5.294868      199.311340   5.215048   
38         12   46.135516  5.289329      198.210331   5.207420   
50         32   37.355055  5.272170      194.838210   5.196208   
52         32   22.903723  5.180671      177.802131   5.101517   

    test_perplexity  
3        989.521082  
1        932.611851  
9        823.128027  
2        781.862929  
0        678.421797  
19       650.746515  
16       649.282838  
8        635.812391  
11       634.218112  
18       606.948531  
17       598.199474  
10       586.223308  
23       507.723572  
15       472.749048  
20       463.337950  
7        461.609970  
12       460.436378  
4        450.099917  
22       447.348791  
21       437.175260  
6        434.538736  
14       426.278573  
5        412.226988  
13       403.318467  
43       296.951166  
24       293.081403  
40       285.822098  
42       280.217711  
27       278.171189  
32       276.431995  
41       271.347757  
35       258.031624  
25       252.215519  
47       251.012319  
26       245.681083  
34       241.355315  
33       240.869530  
46       237.762291  
45       228.192723  
44       221.154958  
36       204.674880  
39       201.602045  
51       199.383573  
48       199.234413  
31       195.669731  
29       193.891097  
30       191.857883  
28       186.510935  
37       186.438438  
49       184.020707  
38       182.622272  
50       180.586133  
52       164.270880  
>>> pd.options.display.max_columns = 11
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  \
3       0.0       1  0.5           5   
1       0.0       1  0.5           2   
9       0.2       1  0.5           2   
2       0.0       1  0.5           3   
0       0.0       1  0.5           1   
19      0.5       1  0.5           5   
16      0.5       1  0.5           1   
8       0.2       1  0.5           1   
11      0.2       1  0.5           5   
18      0.5       1  0.5           3   
17      0.5       1  0.5           2   
10      0.2       1  0.5           3   
23      0.5       1  2.0           5   
15      0.2       1  2.0           5   
20      0.5       1  2.0           1   
7       0.0       1  2.0           5   
12      0.2       1  2.0           1   
4       0.0       1  2.0           1   
22      0.5       1  2.0           3   
21      0.5       1  2.0           2   
6       0.0       1  2.0           3   
14      0.2       1  2.0           3   
5       0.0       1  2.0           2   
13      0.2       1  2.0           2   
43      0.5      12  0.5           5   
24      0.0      12  0.5           1   
40      0.5      12  0.5           1   
42      0.5      12  0.5           3   
27      0.0      12  0.5           5   
32      0.2      12  0.5           1   
41      0.5      12  0.5           2   
35      0.2      12  0.5           5   
25      0.0      12  0.5           2   
47      0.5      12  2.0           5   
26      0.0      12  0.5           3   
34      0.2      12  0.5           3   
33      0.2      12  0.5           2   
46      0.5      12  2.0           3   
45      0.5      12  2.0           2   
44      0.5      12  2.0           1   
36      0.2      12  2.0           1   
39      0.2      12  2.0           5   
51      0.0      32  0.5           5   
48      0.0      32  0.5           1   
31      0.0      12  2.0           5   
29      0.0      12  2.0           2   
30      0.0      12  2.0           3   
28      0.0      12  2.0           1   
37      0.2      12  2.0           2   
49      0.0      32  0.5           2   
38      0.2      12  2.0           3   
50      0.0      32  0.5           3   
52      0.0      32  2.0           1   

                                             filename  ...  epoch_time  \
3   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   55.457326   
1   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   32.106639   
9   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.513603   
2   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   38.042581   
0   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.678797   
19  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   67.410955   
16  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.873118   
8   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.738191   
11  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   67.846113   
18  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.151242   
17  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.783531   
10  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.116142   
23  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   67.591126   
15  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   68.007911   
20  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.870234   
7   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   55.487360   
12  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.753431   
4   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.612858   
22  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.234442   
21  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.493594   
6   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   38.298247   
14  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.534271   
5   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   32.427224   
13  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.634995   
43  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.546237   
24  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.663883   
40  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.722072   
42  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.440262   
27  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   56.313533   
32  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.746994   
41  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.532670   
35  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.252183   
25  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   32.404777   
47  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.707756   
26  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   38.795682   
34  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.276322   
33  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.548976   
46  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.421073   
45  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.522535   
44  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   23.129305   
36  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.752869   
39  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.399188   
51  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   56.175895   
48  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   22.848217   
31  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   54.730397   
29  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   32.457682   
30  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   38.802311   
28  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.664476   
37  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.434054   
49  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   32.353001   
38  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.135516   
50  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   37.355055   
52  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   22.903723   

    val_loss  val_perplexity  test_loss  test_perplexity  
3   6.961126     1054.820222   6.897221       989.521082  
1   7.280081     1451.106151   6.837989       932.611851  
9   7.497376     1803.304547   6.713112       823.128027  
2   6.731551      838.446364   6.661679       781.862929  
0   6.589880      727.693280   6.519769       678.421797  
19  6.548717      698.347867   6.478120       650.746515  
16  6.548279      698.041867   6.475868       649.282838  
8   6.929359     1021.839217   6.454904       635.812391  
11  6.519616      678.318165   6.452393       634.218112  
18  6.482660      653.707452   6.408444       606.948531  
17  6.470601      645.871846   6.393924       598.199474  
10  6.446929      630.762516   6.373701       586.223308  
23  6.307254      548.536685   6.229937       507.723572  
15  6.241873      513.819828   6.158565       472.749048  
20  6.215519      500.455889   6.138457       463.337950  
7   6.220418      502.913230   6.134720       461.609970  
12  6.210973      498.185914   6.132175       460.436378  
4   6.186039      485.917391   6.109470       450.099917  
22  6.185809      485.805748   6.103339       447.348791  
21  6.167248      476.872072   6.080334       437.175260  
6   6.158620      472.775357   6.074285       434.538736  
14  6.138809      463.501280   6.055093       426.278573  
5   6.107359      449.150789   6.021574       412.226988  
13  6.084803      439.133338   5.999726       403.318467  
43  5.771163      320.910744   5.693568       296.951166  
24  5.759030      317.040789   5.680450       293.081403  
40  5.737915      310.416470   5.655370       285.822098  
42  5.714360      303.189960   5.635567       280.217711  
27  5.707621      301.153885   5.628237       278.171189  
32  5.701946      299.449529   5.621965       276.431995  
41  5.684388      294.237635   5.603401       271.347757  
35  5.631497      279.079548   5.553082       258.031624  
25  5.615305      274.597171   5.530284       252.215519  
47  5.610407      273.255506   5.525502       251.012319  
26  5.582123      265.634997   5.504034       245.681083  
34  5.567655      261.819516   5.486270       241.355315  
33  5.566570      261.535547   5.484255       240.869530  
46  5.558394      259.405820   5.471271       237.762291  
45  5.528747      251.828183   5.430191       228.192723  
44  5.489155      242.052660   5.398864       221.154958  
36  5.415980      224.973002   5.321423       204.674880  
39  5.386523      218.442620   5.306296       201.602045  
51  5.377191      216.413480   5.295230       199.383573  
48  5.383825      217.853971   5.294482       199.234413  
31  5.359894      212.702379   5.276428       195.669731  
29  5.346428      209.857287   5.267297       193.891097  
30  5.337364      207.963855   5.256755       191.857883  
28  5.321498      204.690330   5.228490       186.510935  
37  5.318003      203.976231   5.228101       186.438438  
49  5.294868      199.311340   5.215048       184.020707  
38  5.289329      198.210331   5.207420       182.622272  
50  5.272170      194.838210   5.196208       180.586133  
52  5.180671      177.802131   5.101517       164.270880  

[53 rows x 12 columns]
>>> pd.options.display.max_columns = 10
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  \
3       0.0       1  0.5           5   
1       0.0       1  0.5           2   
9       0.2       1  0.5           2   
2       0.0       1  0.5           3   
0       0.0       1  0.5           1   
19      0.5       1  0.5           5   
16      0.5       1  0.5           1   
8       0.2       1  0.5           1   
11      0.2       1  0.5           5   
18      0.5       1  0.5           3   
17      0.5       1  0.5           2   
10      0.2       1  0.5           3   
23      0.5       1  2.0           5   
15      0.2       1  2.0           5   
20      0.5       1  2.0           1   
7       0.0       1  2.0           5   
12      0.2       1  2.0           1   
4       0.0       1  2.0           1   
22      0.5       1  2.0           3   
21      0.5       1  2.0           2   
6       0.0       1  2.0           3   
14      0.2       1  2.0           3   
5       0.0       1  2.0           2   
13      0.2       1  2.0           2   
43      0.5      12  0.5           5   
24      0.0      12  0.5           1   
40      0.5      12  0.5           1   
42      0.5      12  0.5           3   
27      0.0      12  0.5           5   
32      0.2      12  0.5           1   
41      0.5      12  0.5           2   
35      0.2      12  0.5           5   
25      0.0      12  0.5           2   
47      0.5      12  2.0           5   
26      0.0      12  0.5           3   
34      0.2      12  0.5           3   
33      0.2      12  0.5           2   
46      0.5      12  2.0           3   
45      0.5      12  2.0           2   
44      0.5      12  2.0           1   
36      0.2      12  2.0           1   
39      0.2      12  2.0           5   
51      0.0      32  0.5           5   
48      0.0      32  0.5           1   
31      0.0      12  2.0           5   
29      0.0      12  2.0           2   
30      0.0      12  2.0           3   
28      0.0      12  2.0           1   
37      0.2      12  2.0           2   
49      0.0      32  0.5           2   
38      0.2      12  2.0           3   
50      0.0      32  0.5           3   
52      0.0      32  2.0           1   

                                             filename  ...  epoch_time  \
3   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   55.457326   
1   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   32.106639   
9   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.513603   
2   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   38.042581   
0   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.678797   
19  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   67.410955   
16  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.873118   
8   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.738191   
11  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   67.846113   
18  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.151242   
17  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.783531   
10  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.116142   
23  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   67.591126   
15  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   68.007911   
20  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.870234   
7   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   55.487360   
12  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.753431   
4   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   22.612858   
22  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.234442   
21  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.493594   
6   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   38.298247   
14  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   46.534271   
5   model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   32.427224   
13  model_epochs_1_rnn_type_RNN_TANH_hidden_size_2...  ...   35.634995   
43  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.546237   
24  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.663883   
40  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.722072   
42  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.440262   
27  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   56.313533   
32  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.746994   
41  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.532670   
35  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.252183   
25  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   32.404777   
47  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.707756   
26  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   38.795682   
34  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.276322   
33  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.548976   
46  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.421073   
45  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.522535   
44  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   23.129305   
36  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.752869   
39  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   67.399188   
51  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   56.175895   
48  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   22.848217   
31  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   54.730397   
29  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   32.457682   
30  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   38.802311   
28  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   22.664476   
37  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   35.434054   
49  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   32.353001   
38  model_epochs_12_rnn_type_RNN_TANH_hidden_size_...  ...   46.135516   
50  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   37.355055   
52  model_epochs_32_rnn_type_RNN_TANH_hidden_size_...  ...   22.903723   

    val_loss  val_perplexity  test_loss  test_perplexity  
3   6.961126     1054.820222   6.897221       989.521082  
1   7.280081     1451.106151   6.837989       932.611851  
9   7.497376     1803.304547   6.713112       823.128027  
2   6.731551      838.446364   6.661679       781.862929  
0   6.589880      727.693280   6.519769       678.421797  
19  6.548717      698.347867   6.478120       650.746515  
16  6.548279      698.041867   6.475868       649.282838  
8   6.929359     1021.839217   6.454904       635.812391  
11  6.519616      678.318165   6.452393       634.218112  
18  6.482660      653.707452   6.408444       606.948531  
17  6.470601      645.871846   6.393924       598.199474  
10  6.446929      630.762516   6.373701       586.223308  
23  6.307254      548.536685   6.229937       507.723572  
15  6.241873      513.819828   6.158565       472.749048  
20  6.215519      500.455889   6.138457       463.337950  
7   6.220418      502.913230   6.134720       461.609970  
12  6.210973      498.185914   6.132175       460.436378  
4   6.186039      485.917391   6.109470       450.099917  
22  6.185809      485.805748   6.103339       447.348791  
21  6.167248      476.872072   6.080334       437.175260  
6   6.158620      472.775357   6.074285       434.538736  
14  6.138809      463.501280   6.055093       426.278573  
5   6.107359      449.150789   6.021574       412.226988  
13  6.084803      439.133338   5.999726       403.318467  
43  5.771163      320.910744   5.693568       296.951166  
24  5.759030      317.040789   5.680450       293.081403  
40  5.737915      310.416470   5.655370       285.822098  
42  5.714360      303.189960   5.635567       280.217711  
27  5.707621      301.153885   5.628237       278.171189  
32  5.701946      299.449529   5.621965       276.431995  
41  5.684388      294.237635   5.603401       271.347757  
35  5.631497      279.079548   5.553082       258.031624  
25  5.615305      274.597171   5.530284       252.215519  
47  5.610407      273.255506   5.525502       251.012319  
26  5.582123      265.634997   5.504034       245.681083  
34  5.567655      261.819516   5.486270       241.355315  
33  5.566570      261.535547   5.484255       240.869530  
46  5.558394      259.405820   5.471271       237.762291  
45  5.528747      251.828183   5.430191       228.192723  
44  5.489155      242.052660   5.398864       221.154958  
36  5.415980      224.973002   5.321423       204.674880  
39  5.386523      218.442620   5.306296       201.602045  
51  5.377191      216.413480   5.295230       199.383573  
48  5.383825      217.853971   5.294482       199.234413  
31  5.359894      212.702379   5.276428       195.669731  
29  5.346428      209.857287   5.267297       193.891097  
30  5.337364      207.963855   5.256755       191.857883  
28  5.321498      204.690330   5.228490       186.510935  
37  5.318003      203.976231   5.228101       186.438438  
49  5.294868      199.311340   5.215048       184.020707  
38  5.289329      198.210331   5.207420       182.622272  
50  5.272170      194.838210   5.196208       180.586133  
52  5.180671      177.802131   5.101517       164.270880  

[53 rows x 12 columns]
>>> interesting_columns = [c for c in df.columns if df[c].nunique() > 1 and df[c].apply(str).apply(len) < 13]
>>> interesting_columns = [c for c in df.columns if df[c].nunique() > 1 and df[c].apply(str).apply(len).max() < 13]
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
>>> df = df.round(3)
>>> interesting_columns = [c for c in df.columns if df[c].nunique() > 1 and df[c].apply(str).apply(len).max() < 15]
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  best_val_loss  ...  epoch_time  \
3       0.0       1  0.5           5            NaN  ...      55.457   
1       0.0       1  0.5           2            NaN  ...      32.107   
9       0.2       1  0.5           2            NaN  ...      35.514   
2       0.0       1  0.5           3            NaN  ...      38.043   
0       0.0       1  0.5           1            NaN  ...      22.679   
19      0.5       1  0.5           5            NaN  ...      67.411   
16      0.5       1  0.5           1            NaN  ...      22.873   
8       0.2       1  0.5           1            NaN  ...      22.738   
11      0.2       1  0.5           5            NaN  ...      67.846   
18      0.5       1  0.5           3            NaN  ...      46.151   
17      0.5       1  0.5           2            NaN  ...      35.784   
10      0.2       1  0.5           3            NaN  ...      46.116   
23      0.5       1  2.0           5            NaN  ...      67.591   
15      0.2       1  2.0           5            NaN  ...      68.008   
20      0.5       1  2.0           1            NaN  ...      22.870   
7       0.0       1  2.0           5            NaN  ...      55.487   
12      0.2       1  2.0           1            NaN  ...      22.753   
4       0.0       1  2.0           1            NaN  ...      22.613   
22      0.5       1  2.0           3            NaN  ...      46.234   
21      0.5       1  2.0           2            NaN  ...      35.494   
6       0.0       1  2.0           3            NaN  ...      38.298   
14      0.2       1  2.0           3            NaN  ...      46.534   
5       0.0       1  2.0           2            NaN  ...      32.427   
13      0.2       1  2.0           2            NaN  ...      35.635   
43      0.5      12  0.5           5          5.803  ...      67.546   
24      0.0      12  0.5           1          5.764  ...      22.664   
40      0.5      12  0.5           1          5.769  ...      22.722   
42      0.5      12  0.5           3          5.732  ...      46.440   
27      0.0      12  0.5           5          5.747  ...      56.314   
32      0.2      12  0.5           1          5.721  ...      22.747   
41      0.5      12  0.5           2          5.715  ...      35.533   
35      0.2      12  0.5           5          5.658  ...      67.252   
25      0.0      12  0.5           2          5.647  ...      32.405   
47      0.5      12  2.0           5          5.650  ...      67.708   
26      0.0      12  0.5           3          5.617  ...      38.796   
34      0.2      12  0.5           3          5.595  ...      46.276   
33      0.2      12  0.5           2          5.599  ...      35.549   
46      0.5      12  2.0           3          5.551  ...      46.421   
45      0.5      12  2.0           2          5.517  ...      35.523   
44      0.5      12  2.0           1          5.494  ...      23.129   
36      0.2      12  2.0           1          5.428  ...      22.753   
39      0.2      12  2.0           5          5.387  ...      67.399   
51      0.0      32  0.5           5          5.386  ...      56.176   
48      0.0      32  0.5           1          5.380  ...      22.848   
31      0.0      12  2.0           5          5.365  ...      54.730   
29      0.0      12  2.0           2          5.353  ...      32.458   
30      0.0      12  2.0           3          5.352  ...      38.802   
37      0.2      12  2.0           2          5.334  ...      35.434   
28      0.0      12  2.0           1          5.441  ...      22.664   
49      0.0      32  0.5           2          5.321  ...      32.353   
38      0.2      12  2.0           3          5.402  ...      46.136   
50      0.0      32  0.5           3          5.275  ...      37.355   
52      0.0      32  2.0           1          5.181  ...      22.904   

    val_loss  val_perplexity  test_loss  test_perplexity  
3      6.961        1054.820      6.897          989.521  
1      7.280        1451.106      6.838          932.612  
9      7.497        1803.305      6.713          823.128  
2      6.732         838.446      6.662          781.863  
0      6.590         727.693      6.520          678.422  
19     6.549         698.348      6.478          650.747  
16     6.548         698.042      6.476          649.283  
8      6.929        1021.839      6.455          635.812  
11     6.520         678.318      6.452          634.218  
18     6.483         653.707      6.408          606.949  
17     6.471         645.872      6.394          598.199  
10     6.447         630.763      6.374          586.223  
23     6.307         548.537      6.230          507.724  
15     6.242         513.820      6.159          472.749  
20     6.216         500.456      6.138          463.338  
7      6.220         502.913      6.135          461.610  
12     6.211         498.186      6.132          460.436  
4      6.186         485.917      6.109          450.100  
22     6.186         485.806      6.103          447.349  
21     6.167         476.872      6.080          437.175  
6      6.159         472.775      6.074          434.539  
14     6.139         463.501      6.055          426.279  
5      6.107         449.151      6.022          412.227  
13     6.085         439.133      6.000          403.318  
43     5.771         320.911      5.694          296.951  
24     5.759         317.041      5.680          293.081  
40     5.738         310.416      5.655          285.822  
42     5.714         303.190      5.636          280.218  
27     5.708         301.154      5.628          278.171  
32     5.702         299.450      5.622          276.432  
41     5.684         294.238      5.603          271.348  
35     5.631         279.080      5.553          258.032  
25     5.615         274.597      5.530          252.216  
47     5.610         273.256      5.526          251.012  
26     5.582         265.635      5.504          245.681  
34     5.568         261.820      5.486          241.355  
33     5.567         261.536      5.484          240.870  
46     5.558         259.406      5.471          237.762  
45     5.529         251.828      5.430          228.193  
44     5.489         242.053      5.399          221.155  
36     5.416         224.973      5.321          204.675  
39     5.387         218.443      5.306          201.602  
51     5.377         216.413      5.295          199.384  
48     5.384         217.854      5.294          199.234  
31     5.360         212.702      5.276          195.670  
29     5.346         209.857      5.267          193.891  
30     5.337         207.964      5.257          191.858  
37     5.318         203.976      5.228          186.438  
28     5.321         204.690      5.228          186.511  
49     5.295         199.311      5.215          184.021  
38     5.289         198.210      5.207          182.622  
50     5.272         194.838      5.196          180.586  
52     5.181         177.802      5.102          164.271  

[53 rows x 11 columns]
>>> df = df.round(2)
>>> interesting_columns = [c for c in df.columns if df[c].nunique() > 1 and df[c].apply(str).apply(len).max() < 10]
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  best_val_loss  ...  epoch_time  \
3       0.0       1  0.5           5            NaN  ...       55.46   
1       0.0       1  0.5           2            NaN  ...       32.11   
9       0.2       1  0.5           2            NaN  ...       35.51   
2       0.0       1  0.5           3            NaN  ...       38.04   
0       0.0       1  0.5           1            NaN  ...       22.68   
19      0.5       1  0.5           5            NaN  ...       67.41   
16      0.5       1  0.5           1            NaN  ...       22.87   
8       0.2       1  0.5           1            NaN  ...       22.74   
11      0.2       1  0.5           5            NaN  ...       67.85   
18      0.5       1  0.5           3            NaN  ...       46.15   
17      0.5       1  0.5           2            NaN  ...       35.78   
10      0.2       1  0.5           3            NaN  ...       46.12   
23      0.5       1  2.0           5            NaN  ...       67.59   
15      0.2       1  2.0           5            NaN  ...       68.01   
20      0.5       1  2.0           1            NaN  ...       22.87   
7       0.0       1  2.0           5            NaN  ...       55.49   
12      0.2       1  2.0           1            NaN  ...       22.75   
4       0.0       1  2.0           1            NaN  ...       22.61   
22      0.5       1  2.0           3            NaN  ...       46.23   
21      0.5       1  2.0           2            NaN  ...       35.49   
6       0.0       1  2.0           3            NaN  ...       38.30   
14      0.2       1  2.0           3            NaN  ...       46.53   
5       0.0       1  2.0           2            NaN  ...       32.43   
13      0.2       1  2.0           2            NaN  ...       35.64   
43      0.5      12  0.5           5           5.80  ...       67.55   
24      0.0      12  0.5           1           5.76  ...       22.66   
40      0.5      12  0.5           1           5.77  ...       22.72   
42      0.5      12  0.5           3           5.73  ...       46.44   
27      0.0      12  0.5           5           5.75  ...       56.31   
32      0.2      12  0.5           1           5.72  ...       22.75   
41      0.5      12  0.5           2           5.72  ...       35.53   
35      0.2      12  0.5           5           5.66  ...       67.25   
25      0.0      12  0.5           2           5.65  ...       32.40   
47      0.5      12  2.0           5           5.65  ...       67.71   
26      0.0      12  0.5           3           5.62  ...       38.80   
34      0.2      12  0.5           3           5.60  ...       46.28   
33      0.2      12  0.5           2           5.60  ...       35.55   
46      0.5      12  2.0           3           5.55  ...       46.42   
45      0.5      12  2.0           2           5.52  ...       35.52   
44      0.5      12  2.0           1           5.49  ...       23.13   
36      0.2      12  2.0           1           5.43  ...       22.75   
39      0.2      12  2.0           5           5.39  ...       67.40   
51      0.0      32  0.5           5           5.39  ...       56.18   
48      0.0      32  0.5           1           5.38  ...       22.85   
31      0.0      12  2.0           5           5.36  ...       54.73   
29      0.0      12  2.0           2           5.35  ...       32.46   
30      0.0      12  2.0           3           5.35  ...       38.80   
37      0.2      12  2.0           2           5.33  ...       35.43   
28      0.0      12  2.0           1           5.44  ...       22.66   
49      0.0      32  0.5           2           5.32  ...       32.35   
38      0.2      12  2.0           3           5.40  ...       46.14   
50      0.0      32  0.5           3           5.28  ...       37.35   
52      0.0      32  2.0           1           5.18  ...       22.90   

    val_loss  val_perplexity  test_loss  test_perplexity  
3       6.96         1054.82       6.90           989.52  
1       7.28         1451.11       6.84           932.61  
9       7.50         1803.30       6.71           823.13  
2       6.73          838.45       6.66           781.86  
0       6.59          727.69       6.52           678.42  
19      6.55          698.35       6.48           650.75  
16      6.55          698.04       6.48           649.28  
8       6.93         1021.84       6.46           635.81  
11      6.52          678.32       6.45           634.22  
18      6.48          653.71       6.41           606.95  
17      6.47          645.87       6.39           598.20  
10      6.45          630.76       6.37           586.22  
23      6.31          548.54       6.23           507.72  
15      6.24          513.82       6.16           472.75  
20      6.22          500.46       6.14           463.34  
7       6.22          502.91       6.14           461.61  
12      6.21          498.19       6.13           460.44  
4       6.19          485.92       6.11           450.10  
22      6.19          485.81       6.10           447.35  
21      6.17          476.87       6.08           437.18  
6       6.16          472.78       6.07           434.54  
14      6.14          463.50       6.06           426.28  
5       6.11          449.15       6.02           412.23  
13      6.08          439.13       6.00           403.32  
43      5.77          320.91       5.69           296.95  
24      5.76          317.04       5.68           293.08  
40      5.74          310.42       5.66           285.82  
42      5.71          303.19       5.64           280.22  
27      5.71          301.15       5.63           278.17  
32      5.70          299.45       5.62           276.43  
41      5.68          294.24       5.60           271.35  
35      5.63          279.08       5.55           258.03  
25      5.62          274.60       5.53           252.22  
47      5.61          273.26       5.53           251.01  
26      5.58          265.64       5.50           245.68  
34      5.57          261.82       5.49           241.36  
33      5.57          261.54       5.48           240.87  
46      5.56          259.41       5.47           237.76  
45      5.53          251.83       5.43           228.19  
44      5.49          242.05       5.40           221.16  
36      5.42          224.97       5.32           204.68  
39      5.39          218.44       5.31           201.60  
51      5.38          216.41       5.30           199.38  
48      5.38          217.85       5.29           199.23  
31      5.36          212.70       5.28           195.67  
29      5.35          209.86       5.27           193.89  
30      5.34          207.96       5.26           191.86  
37      5.32          203.98       5.23           186.44  
28      5.32          204.69       5.23           186.51  
49      5.30          199.31       5.22           184.02  
38      5.29          198.21       5.21           182.62  
50      5.27          194.84       5.20           180.59  
52      5.18          177.80       5.10           164.27  

[53 rows x 11 columns]
>>> interesting_columns = [c for c in df.columns if df[c].nunique() > 1 and df[c].apply(str).apply(len).max() < 10 and len(c) <= 10]
>>> df[interesting_columns].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  epoch_num  epoch_time  val_loss  \
3       0.0       1  0.5           5          1       55.46      6.96   
1       0.0       1  0.5           2          1       32.11      7.28   
9       0.2       1  0.5           2          1       35.51      7.50   
2       0.0       1  0.5           3          1       38.04      6.73   
0       0.0       1  0.5           1          1       22.68      6.59   
19      0.5       1  0.5           5          1       67.41      6.55   
16      0.5       1  0.5           1          1       22.87      6.55   
8       0.2       1  0.5           1          1       22.74      6.93   
11      0.2       1  0.5           5          1       67.85      6.52   
18      0.5       1  0.5           3          1       46.15      6.48   
17      0.5       1  0.5           2          1       35.78      6.47   
10      0.2       1  0.5           3          1       46.12      6.45   
23      0.5       1  2.0           5          1       67.59      6.31   
15      0.2       1  2.0           5          1       68.01      6.24   
20      0.5       1  2.0           1          1       22.87      6.22   
7       0.0       1  2.0           5          1       55.49      6.22   
12      0.2       1  2.0           1          1       22.75      6.21   
4       0.0       1  2.0           1          1       22.61      6.19   
22      0.5       1  2.0           3          1       46.23      6.19   
21      0.5       1  2.0           2          1       35.49      6.17   
6       0.0       1  2.0           3          1       38.30      6.16   
14      0.2       1  2.0           3          1       46.53      6.14   
5       0.0       1  2.0           2          1       32.43      6.11   
13      0.2       1  2.0           2          1       35.64      6.08   
43      0.5      12  0.5           5         12       67.55      5.77   
24      0.0      12  0.5           1         12       22.66      5.76   
40      0.5      12  0.5           1         12       22.72      5.74   
42      0.5      12  0.5           3         12       46.44      5.71   
27      0.0      12  0.5           5         12       56.31      5.71   
32      0.2      12  0.5           1         12       22.75      5.70   
41      0.5      12  0.5           2         12       35.53      5.68   
35      0.2      12  0.5           5         12       67.25      5.63   
25      0.0      12  0.5           2         12       32.40      5.62   
47      0.5      12  2.0           5         12       67.71      5.61   
26      0.0      12  0.5           3         12       38.80      5.58   
34      0.2      12  0.5           3         12       46.28      5.57   
33      0.2      12  0.5           2         12       35.55      5.57   
46      0.5      12  2.0           3         12       46.42      5.56   
45      0.5      12  2.0           2         12       35.52      5.53   
44      0.5      12  2.0           1         12       23.13      5.49   
36      0.2      12  2.0           1         12       22.75      5.42   
39      0.2      12  2.0           5         12       67.40      5.39   
51      0.0      32  0.5           5         32       56.18      5.38   
48      0.0      32  0.5           1         32       22.85      5.38   
31      0.0      12  2.0           5         12       54.73      5.36   
29      0.0      12  2.0           2         12       32.46      5.35   
30      0.0      12  2.0           3         12       38.80      5.34   
37      0.2      12  2.0           2         12       35.43      5.32   
28      0.0      12  2.0           1         12       22.66      5.32   
49      0.0      32  0.5           2         32       32.35      5.30   
38      0.2      12  2.0           3         12       46.14      5.29   
50      0.0      32  0.5           3         32       37.35      5.27   
52      0.0      32  2.0           1         32       22.90      5.18   

    test_loss  
3        6.90  
1        6.84  
9        6.71  
2        6.66  
0        6.52  
19       6.48  
16       6.48  
8        6.46  
11       6.45  
18       6.41  
17       6.39  
10       6.37  
23       6.23  
15       6.16  
20       6.14  
7        6.14  
12       6.13  
4        6.11  
22       6.10  
21       6.08  
6        6.07  
14       6.06  
5        6.02  
13       6.00  
43       5.69  
24       5.68  
40       5.66  
42       5.64  
27       5.63  
32       5.62  
41       5.60  
35       5.55  
25       5.53  
47       5.53  
26       5.50  
34       5.49  
33       5.48  
46       5.47  
45       5.43  
44       5.40  
36       5.32  
39       5.31  
51       5.30  
48       5.29  
31       5.28  
29       5.27  
30       5.26  
37       5.23  
28       5.23  
49       5.22  
38       5.21  
50       5.20  
52       5.10  
>>> cols = 'dropout  epochs   lr  num_layers  epoch_time  val_loss test_loss'.split()
>>> df[cols].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  epoch_time  val_loss  test_loss
3       0.0       1  0.5           5       55.46      6.96       6.90
1       0.0       1  0.5           2       32.11      7.28       6.84
9       0.2       1  0.5           2       35.51      7.50       6.71
2       0.0       1  0.5           3       38.04      6.73       6.66
0       0.0       1  0.5           1       22.68      6.59       6.52
19      0.5       1  0.5           5       67.41      6.55       6.48
16      0.5       1  0.5           1       22.87      6.55       6.48
8       0.2       1  0.5           1       22.74      6.93       6.46
11      0.2       1  0.5           5       67.85      6.52       6.45
18      0.5       1  0.5           3       46.15      6.48       6.41
17      0.5       1  0.5           2       35.78      6.47       6.39
10      0.2       1  0.5           3       46.12      6.45       6.37
23      0.5       1  2.0           5       67.59      6.31       6.23
15      0.2       1  2.0           5       68.01      6.24       6.16
20      0.5       1  2.0           1       22.87      6.22       6.14
7       0.0       1  2.0           5       55.49      6.22       6.14
12      0.2       1  2.0           1       22.75      6.21       6.13
4       0.0       1  2.0           1       22.61      6.19       6.11
22      0.5       1  2.0           3       46.23      6.19       6.10
21      0.5       1  2.0           2       35.49      6.17       6.08
6       0.0       1  2.0           3       38.30      6.16       6.07
14      0.2       1  2.0           3       46.53      6.14       6.06
5       0.0       1  2.0           2       32.43      6.11       6.02
13      0.2       1  2.0           2       35.64      6.08       6.00
43      0.5      12  0.5           5       67.55      5.77       5.69
24      0.0      12  0.5           1       22.66      5.76       5.68
40      0.5      12  0.5           1       22.72      5.74       5.66
42      0.5      12  0.5           3       46.44      5.71       5.64
27      0.0      12  0.5           5       56.31      5.71       5.63
32      0.2      12  0.5           1       22.75      5.70       5.62
41      0.5      12  0.5           2       35.53      5.68       5.60
35      0.2      12  0.5           5       67.25      5.63       5.55
25      0.0      12  0.5           2       32.40      5.62       5.53
47      0.5      12  2.0           5       67.71      5.61       5.53
26      0.0      12  0.5           3       38.80      5.58       5.50
34      0.2      12  0.5           3       46.28      5.57       5.49
33      0.2      12  0.5           2       35.55      5.57       5.48
46      0.5      12  2.0           3       46.42      5.56       5.47
45      0.5      12  2.0           2       35.52      5.53       5.43
44      0.5      12  2.0           1       23.13      5.49       5.40
36      0.2      12  2.0           1       22.75      5.42       5.32
39      0.2      12  2.0           5       67.40      5.39       5.31
51      0.0      32  0.5           5       56.18      5.38       5.30
48      0.0      32  0.5           1       22.85      5.38       5.29
31      0.0      12  2.0           5       54.73      5.36       5.28
29      0.0      12  2.0           2       32.46      5.35       5.27
30      0.0      12  2.0           3       38.80      5.34       5.26
37      0.2      12  2.0           2       35.43      5.32       5.23
28      0.0      12  2.0           1       22.66      5.32       5.23
49      0.0      32  0.5           2       32.35      5.30       5.22
38      0.2      12  2.0           3       46.14      5.29       5.21
50      0.0      32  0.5           3       37.35      5.27       5.20
52      0.0      32  2.0           1       22.90      5.18       5.10
>>> df = pd.DataFrame(lines)
>>> df = pd.DataFrame(lines).round(2)
>>> with jsonlines.open('experiments.ljson') as fin:
...     lines = list(fin)
...
>>> df = pd.DataFrame(lines).round(2)
>>> df[cols].sort_values('test_loss', ascending=False)
    dropout  epochs   lr  num_layers  epoch_time  val_loss  test_loss
3       0.0       1  0.5           5       55.46      6.96       6.90
1       0.0       1  0.5           2       32.11      7.28       6.84
9       0.2       1  0.5           2       35.51      7.50       6.71
2       0.0       1  0.5           3       38.04      6.73       6.66
0       0.0       1  0.5           1       22.68      6.59       6.52
16      0.5       1  0.5           1       22.87      6.55       6.48
19      0.5       1  0.5           5       67.41      6.55       6.48
8       0.2       1  0.5           1       22.74      6.93       6.45
11      0.2       1  0.5           5       67.85      6.52       6.45
18      0.5       1  0.5           3       46.15      6.48       6.41
17      0.5       1  0.5           2       35.78      6.47       6.39
10      0.2       1  0.5           3       46.12      6.45       6.37
23      0.5       1  2.0           5       67.59      6.31       6.23
15      0.2       1  2.0           5       68.01      6.24       6.16
20      0.5       1  2.0           1       22.87      6.22       6.14
7       0.0       1  2.0           5       55.49      6.22       6.13
12      0.2       1  2.0           1       22.75      6.21       6.13
4       0.0       1  2.0           1       22.61      6.19       6.11
22      0.5       1  2.0           3       46.23      6.19       6.10
21      0.5       1  2.0           2       35.49      6.17       6.08
6       0.0       1  2.0           3       38.30      6.16       6.07
14      0.2       1  2.0           3       46.53      6.14       6.06
5       0.0       1  2.0           2       32.43      6.11       6.02
13      0.2       1  2.0           2       35.63      6.08       6.00
43      0.5      12  0.5           5       67.55      5.77       5.69
24      0.0      12  0.5           1       22.66      5.76       5.68
40      0.5      12  0.5           1       22.72      5.74       5.66
42      0.5      12  0.5           3       46.44      5.71       5.64
27      0.0      12  0.5           5       56.31      5.71       5.63
32      0.2      12  0.5           1       22.75      5.70       5.62
41      0.5      12  0.5           2       35.53      5.68       5.60
35      0.2      12  0.5           5       67.25      5.63       5.55
47      0.5      12  2.0           5       67.71      5.61       5.53
25      0.0      12  0.5           2       32.40      5.62       5.53
26      0.0      12  0.5           3       38.80      5.58       5.50
34      0.2      12  0.5           3       46.28      5.57       5.49
33      0.2      12  0.5           2       35.55      5.57       5.48
46      0.5      12  2.0           3       46.42      5.56       5.47
45      0.5      12  2.0           2       35.52      5.53       5.43
44      0.5      12  2.0           1       23.13      5.49       5.40
36      0.2      12  2.0           1       22.75      5.42       5.32
39      0.2      12  2.0           5       67.40      5.39       5.31
51      0.0      32  0.5           5       56.18      5.38       5.30
48      0.0      32  0.5           1       22.85      5.38       5.29
31      0.0      12  2.0           5       54.73      5.36       5.28
29      0.0      12  2.0           2       32.46      5.35       5.27
30      0.0      12  2.0           3       38.80      5.34       5.26
28      0.0      12  2.0           1       22.66      5.32       5.23
37      0.2      12  2.0           2       35.43      5.32       5.23
49      0.0      32  0.5           2       32.35      5.29       5.22
38      0.2      12  2.0           3       46.14      5.29       5.21
50      0.0      32  0.5           3       37.36      5.27       5.20
52      0.0      32  2.0           1       22.90      5.18       5.10
53      0.0      32  2.0           2       32.49      5.14       5.06
>>> hist -o -p -f hypertune_experiments.hist.md
