"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
                `${props.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.adminDns,
                            props.endpoints.clusterDns,
                            props.endpoints.managerDns,
                            props.endpoints.telemetryDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${props.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n    - path: port\n      objectAlias: dbport\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                    {
                                        objectName: 'dbport',
                                        key: 'dbport',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...props.HelmOptions,
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                // ingressController: {
                //   enabled: true,
                //   installCRDs: false,
                //   image: {
                //     repository: 'kong/kubernetes-ingress-controller',
                //     tag: '1.3.1-alpine',
                //   },
                // },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-cp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-cp-cluster-issuer-secret/ca.crt',
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                    pg_port: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbport',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: 'true',
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 443,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: 'true',
                        servicePort: 443,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-cp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,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