"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongAdminTaskdefinition = void 0;
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
// import { Certificates, CustomImage } from '../../../../kong-core';
const kong_core_1 = require("kong-core");
class KongAdminTaskdefinition extends aws_ecs_1.FargateTaskDefinition {
    constructor(scope, id, props) {
        super(scope, id, props.kongTaskProps);
        // Initialize the port variables or user defined input
        const adminPort = props.kongFeaturesProps.adminProps.httpPort || 8001;
        const adminHttpsPort = props.kongFeaturesProps.adminProps.httpsPort || 8444;
        const managerPort = props.kongFeaturesProps.kongManagerProps.httpPort || 8002;
        const managerHttpsPort = props.kongFeaturesProps.kongManagerProps.httpsPort || 8445;
        const devPortalPort = props.kongFeaturesProps.devPortalProps.port || 8003;
        const clusterPort = props.kongFeaturesProps.clusterProps.port || 8005;
        const telemetryPort = props.kongFeaturesProps.clusterTelemetryProps.port || 8006;
        // Define the ADMIN Listen value based on user input
        const adminHttpListen = '0.0.0.0:' + adminPort;
        const adminHttpsListen = '0.0.0.0:' + adminHttpsPort;
        let adminListen = adminHttpsListen + ' ssl';
        if (props.kongFeaturesProps.adminProps.enableHttp) {
            adminListen = adminHttpListen + ',' + adminHttpsListen + ' ssl';
        }
        // Define the Manager Listen value based on user input
        const managerHttpListen = '0.0.0.0:' + managerPort;
        const managerHttpsListen = '0.0.0.0:' + managerHttpsPort;
        let managerListen = managerHttpsListen + ' ssl';
        if (props.kongFeaturesProps.kongManagerProps.enableHttp) {
            managerListen = managerHttpListen + ',' + managerHttpsListen + ' ssl';
        }
        //Define the Kong Environment
        let env = {};
        env.KONG_LOG_LEVEL = 'debug';
        env.KONG_DATABASE = 'postgres';
        env.KONG_PG_HOST = props.dbInstanceEndpointAddress;
        env.KONG_PG_PORT = props.dbInstanceEndpointPort;
        env.KONG_ROLE = 'control_plane';
        env.KONG_CLUSTER_MTLS = 'pki';
        env.KONG_CLUSTER_CA_CERT = '/certs/ca.crt';
        env.KONG_CLUSTER_CERT = '/certs/cluster.crt';
        env.KONG_CLUSTER_CERT_KEY = '/certs/cluster.key';
        env.KONG_ADMIN_ACCESS_LOG = '/dev/stdout';
        env.KONG_ADMIN_ERROR_LOG = '/dev/stderr';
        env.KONG_PROXY_ACCESS_LOG = '/dev/stdout';
        env.KONG_PROXY_ERROR_LOG = '/dev/stderr';
        env.KONG_ADMIN_LISTEN = adminListen;
        env.KONG_ADMIN_API_URI = 'http://admin.' + props.hostedZoneName;
        env.KONG_ADMIN_SSL = 'on';
        env.KONG_ADMIN_CERT = '/certs/cluster.crt';
        env.KONG_ADMIN_CERT_KEY = '/certs/cluster.key';
        env.KONG_ADMIN_API_URI = 'https://admin.' + props.hostedZoneName;
        env.KONG_ADMIN_GUI_LISTEN = managerListen;
        env.KONG_ADMIN_GUI_SSL_CERT = '/certs/cluster.crt';
        env.KONG_ADMIN_GUI_SSL_CERT_KEY = '/certs/cluster.key';
        if (props.kongFeaturesProps.devPortalProps.enabled) {
            env.KONG_PORTAL = 'on';
            env.KONG_PORTAL_GUI_HOST = 'devportal.' + props.hostedZoneName;
        }
        if (props.kongFeaturesProps.clusterTelemetryProps.enabled) {
            env.KONG_CLUSTER_TELEMETRY_LISTEN = '0.0.0.0:' + telemetryPort;
        }
        //ALlow kong task to pull image from ECR
        this.addToExecutionRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
        }));
        //Allow kong task to create Log group and pull certificates from ACM.
        this.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'acm-pca:Get*',
                'acm:ExportCertificate',
            ],
        }));
        //Define Port mapping for Kong Container
        let portMapping = [];
        portMapping = portMapping.concat({ containerPort: adminHttpsPort, hostPort: adminHttpsPort, protocol: aws_ecs_1.Protocol.TCP });
        if (props.kongFeaturesProps.adminProps.enableHttp) {
            portMapping = portMapping.concat({ containerPort: adminPort, hostPort: adminPort, protocol: aws_ecs_1.Protocol.TCP });
        }
        if (props.kongFeaturesProps.kongManagerProps.enabled) {
            portMapping = portMapping.concat({ containerPort: managerHttpsPort, hostPort: managerHttpsPort, protocol: aws_ecs_1.Protocol.TCP });
            if (props.kongFeaturesProps.kongManagerProps.enableHttp) {
                portMapping = portMapping.concat({ containerPort: managerPort, hostPort: managerPort, protocol: aws_ecs_1.Protocol.TCP });
            }
        }
        ;
        if (props.kongFeaturesProps.clusterProps.enabled) {
            portMapping = portMapping.concat({ containerPort: clusterPort, hostPort: clusterPort, protocol: aws_ecs_1.Protocol.TCP });
        }
        if (props.kongFeaturesProps.clusterTelemetryProps.enabled) {
            portMapping = portMapping.concat({ containerPort: telemetryPort, hostPort: telemetryPort, protocol: aws_ecs_1.Protocol.TCP });
        }
        if (props.kongFeaturesProps.devPortalProps.enabled) {
            portMapping = portMapping.concat({ containerPort: devPortalPort, hostPort: devPortalPort, protocol: aws_ecs_1.Protocol.TCP });
        }
        ;
        //Add the main Container
        const kongCertImage = new kong_core_1.CustomImage(this, 'KongCustomImage', {
            imageName: 'docker.io/kong/kong-gateway:2.8.1.0-alpine',
        });
        const kongContainer = this.addContainer('KongControlPlaneContainer', {
            image: aws_ecs_1.ContainerImage.fromRegistry(props.image),
            environment: env,
            essential: true,
            secrets: props.secrets,
            logging: aws_ecs_1.LogDrivers.firelens({
                options: {
                    'Name': 'cloudwatch',
                    'region': props.cluster.stack.region,
                    'log_group_name': props.cluster.clusterName + '-KongControlPlaneContainer',
                    'auto_create_group': 'true',
                    'log_stream_prefix': 'from-control-plane-',
                    'log-driver-buffer-limit': '2097152',
                },
            }),
            portMappings: portMapping,
        });
        //Add the Certificates Container to pull certs
        const kongContainerCerts = this.addContainer('KongControlPlaneContainerCerts', {
            image: kongCertImage.kongCustomImage,
            essential: false,
            cpu: 256,
            environment: {
                certArn: props.certificates.certificate.ref,
                rootCAArn: props.certificates.certificate.certificateAuthorityArn || '',
            },
            logging: aws_ecs_1.LogDrivers.firelens({
                options: {
                    'Name': 'cloudwatch',
                    'region': props.cluster.stack.region,
                    'log_group_name': props.cluster.clusterName + '-KongControlPlaneContainerCerts',
                    'auto_create_group': 'true',
                    'log_stream_prefix': 'from-control-plane-',
                    'log-driver-buffer-limit': '2097152',
                },
            }),
            command: ['sh', '/scripts/getCertificates.sh'],
        });
        let kongMigrationsContainer;
        if (props.kongFeaturesProps.kongBootstrapMigration) {
            //Add the migration  Container if enabled
            kongMigrationsContainer = this.addContainer('KongControlPlaneContainerMigrations', {
                image: aws_ecs_1.ContainerImage.fromRegistry(props.image),
                environment: env,
                essential: false,
                cpu: 256,
                logging: aws_ecs_1.LogDrivers.firelens({
                    options: {
                        'Name': 'cloudwatch',
                        'region': props.cluster.stack.region,
                        'log_group_name': props.cluster.clusterName + '-KongControlPlaneContainerMigrations',
                        'auto_create_group': 'true',
                        'log_stream_prefix': 'from-control-plane-',
                        'log-driver-buffer-limit': '2097152',
                    },
                }),
                command: ['kong', 'migrations', 'bootstrap'],
                secrets: props.secrets,
            });
            kongMigrationsContainer.addContainerDependencies({
                container: kongContainerCerts,
                condition: aws_ecs_1.ContainerDependencyCondition.SUCCESS,
            });
            kongContainer.addContainerDependencies({
                container: kongMigrationsContainer,
                condition: aws_ecs_1.ContainerDependencyCondition.SUCCESS,
            });
        }
        else {
            kongContainer.addContainerDependencies({
                container: kongContainerCerts,
                condition: aws_ecs_1.ContainerDependencyCondition.SUCCESS,
            });
        }
        //Add the firelens log router.
        this.addFirelensLogRouter('KongFirelens', {
            image: aws_ecs_1.ContainerImage.fromRegistry('amazon/aws-for-fluent-bit'),
            firelensConfig: {
                type: aws_ecs_1.FirelensLogRouterType.FLUENTBIT,
            },
        });
        //Define a mount point to share between containers
        const mountPoint = {
            containerPath: '/certs',
            readOnly: false,
            sourceVolume: 'certs',
        };
        kongContainer.addMountPoints(mountPoint);
        kongContainerCerts.addMountPoints(mountPoint);
        if (kongMigrationsContainer) {
            kongMigrationsContainer.addMountPoints(mountPoint);
        }
        this.addVolume({
            name: mountPoint.sourceVolume,
        });
    }
}
exports.KongAdminTaskdefinition = KongAdminTaskdefinition;
//# sourceMappingURL=data:application/json;base64,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