"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const aws_redshift_1 = require("aws-cdk-lib/aws-redshift");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Node Types to use in the cluster
 * used for defining `ClusterProps.nodeType`.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.xlplus
     */
    NodeType["RA3_XLPLUS"] = "ra3.xlplus";
    /**
     * ra3.4xlarge
     */
    NodeType["RA3_4XLARGE"] = "ra3.4xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * What cluster type to use.
 * Used by `ClusterProps.clusterType`
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the `ClusterProps.numberOfNodes` parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using `ClusterProps.numberOfNodes` parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType = exports.ClusterType || (exports.ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends aws_cdk_lib_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.parameterGroup = props.parameterGroup;
        this.roles = props?.roles ? [...props.roles] : [];
        const removalPolicy = props.removalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN;
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.ClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: removalPolicy,
        });
        const securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = this.validateNodeCount(clusterType, props.numberOfNodes);
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingProperties) {
            loggingProperties = {
                bucketName: props.loggingProperties.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingProperties.loggingKeyPrefix,
            };
            props.loggingProperties.loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                actions: [
                    's3:GetBucketAcl',
                    's3:PutObject',
                ],
                resources: [
                    props.loggingProperties.loggingBucket.arnForObjects('*'),
                    props.loggingProperties.loggingBucket.bucketArn,
                ],
                principals: [
                    new iam.ServicePrincipal('redshift.amazonaws.com'),
                ],
            }));
        }
        this.cluster = new aws_redshift_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.clusterSubnetGroupName,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin (unsafeUnwrap here is safe)
            masterUsername: secret?.secretValueFromJson('username').unsafeUnwrap() ?? props.masterUser.masterUsername,
            masterUserPassword: secret?.secretValueFromJson('password').unsafeUnwrap()
                ?? props.masterUser.masterPassword?.unsafeUnwrap()
                ?? 'default',
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: aws_cdk_lib_1.Lazy.list({ produce: () => this.roles.map(role => role.roleArn) }, { omitEmpty: true }),
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: props.publiclyAccessible || false,
            // Encryption
            kmsKeyId: props.encryptionKey?.keyId,
            encrypted: props.encrypted ?? true,
            classic: props.classicResizing,
            elasticIp: props.elasticIp,
            enhancedVpcRouting: props.enhancedVpcRouting,
        });
        this.cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = this.cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = aws_cdk_lib_1.Token.asNumber(this.cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
        if (props.rebootForParameterChanges) {
            this.enableRebootForParameterChanges();
        }
        // Add default role if specified and also available in the roles list
        if (props.defaultRole) {
            if (props.roles?.some(x => x === props.defaultRole)) {
                this.addDefaultIamRole(props.defaultRole);
            }
            else {
                throw new Error('Default role must be included in role list.');
            }
        }
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_RotationMultiUserOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationMultiUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    validateNodeCount(clusterType, numberOfNodes) {
        if (clusterType === ClusterType.SINGLE_NODE) {
            // This property must not be set for single-node clusters; be generous and treat a value of 1 node as undefined.
            if (numberOfNodes !== undefined && numberOfNodes !== 1) {
                throw new Error('Number of nodes must be not be supplied or be 1 for cluster type single-node');
            }
            return undefined;
        }
        else {
            if (aws_cdk_lib_1.Token.isUnresolved(numberOfNodes)) {
                return numberOfNodes;
            }
            const nodeCount = numberOfNodes ?? 2;
            if (nodeCount < 2 || nodeCount > 100) {
                throw new Error('Number of nodes for cluster type multi-node must be at least 2 and no more than 100');
            }
            return nodeCount;
        }
    }
    /**
     * Adds a parameter to the Clusters' parameter group
     *
     * @param name the parameter name
     * @param value the parameter name
     */
    addToParameterGroup(name, value) {
        if (!this.parameterGroup) {
            const param = {};
            param[name] = value;
            this.parameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: this.cluster.clusterIdentifier ? `Parameter Group for the ${this.cluster.clusterIdentifier} Redshift cluster` : 'Cluster parameter group for family redshift-1.0',
                parameters: param,
            });
            this.cluster.clusterParameterGroupName = this.parameterGroup.clusterParameterGroupName;
        }
        else if (this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup) {
            this.parameterGroup.addParameter(name, value);
        }
        else {
            throw new Error('Cannot add a parameter to an imported parameter group.');
        }
    }
    /**
     * Enables automatic cluster rebooting when changes to the cluster's parameter group require a restart to apply.
     */
    enableRebootForParameterChanges() {
        if (this.node.tryFindChild('RedshiftClusterRebooterCustomResource')) {
            return;
        }
        const rebootFunction = new lambda.SingletonFunction(this, 'RedshiftClusterRebooterFunction', {
            uuid: '511e207f-13df-4b8b-b632-c32b30b65ac2',
            runtime: lambda.Runtime.NODEJS_16_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'cluster-parameter-change-reboot-handler')),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(900),
        });
        rebootFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift:DescribeClusters'],
            resources: ['*'],
        }));
        rebootFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift:RebootCluster'],
            resources: [
                aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'redshift',
                    resource: 'cluster',
                    resourceName: this.clusterName,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
        }));
        const provider = new custom_resources_1.Provider(this, 'ResourceProvider', {
            onEventHandler: rebootFunction,
        });
        const customResource = new aws_cdk_lib_1.CustomResource(this, 'RedshiftClusterRebooterCustomResource', {
            resourceType: 'Custom::RedshiftClusterRebooter',
            serviceToken: provider.serviceToken,
            properties: {
                ClusterId: this.clusterName,
                ParameterGroupName: aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (!this.parameterGroup) {
                            throw new Error('Cannot enable reboot for parameter changes when there is no associated ClusterParameterGroup.');
                        }
                        return this.parameterGroup.clusterParameterGroupName;
                    },
                }),
                ParametersString: aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (!(this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup)) {
                            throw new Error('Cannot enable reboot for parameter changes when using an imported parameter group.');
                        }
                        return JSON.stringify(this.parameterGroup.parameters);
                    },
                }),
            },
        });
        aws_cdk_lib_1.Lazy.any({
            produce: () => {
                if (!this.parameterGroup) {
                    throw new Error('Cannot enable reboot for parameter changes when there is no associated ClusterParameterGroup.');
                }
                customResource.node.addDependency(this, this.parameterGroup);
            },
        });
    }
    /**
     * Adds default IAM role to cluster. The default IAM role must be already associated to the cluster to be added as the default role.
     *
     * @param defaultIamRole the IAM role to be set as the default role
     */
    addDefaultIamRole(defaultIamRole) {
        // Get list of IAM roles attached to cluster
        const clusterRoleList = this.roles ?? [];
        // Check to see if default role is included in list of cluster IAM roles
        var roleAlreadyOnCluster = false;
        for (var i = 0; i < clusterRoleList.length; i++) {
            if (clusterRoleList[i] === defaultIamRole) {
                roleAlreadyOnCluster = true;
                break;
            }
        }
        if (!roleAlreadyOnCluster) {
            throw new Error('Default role must be associated to the Redshift cluster to be set as the default role.');
        }
        // On UPDATE or CREATE define the default IAM role. On DELETE, remove the default IAM role
        const defaultRoleCustomResource = new custom_resources_1.AwsCustomResource(this, 'default-role', {
            onUpdate: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: this.cluster.ref,
                    DefaultIamRoleArn: defaultIamRole.roleArn,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${defaultIamRole.roleArn}-${this.cluster.ref}`),
            },
            onDelete: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: this.cluster.ref,
                    DefaultIamRoleArn: '',
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${defaultIamRole.roleArn}-${this.cluster.ref}`),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
            installLatestAwsSdk: false,
        });
        defaultIamRole.grantPassRole(defaultRoleCustomResource.grantPrincipal);
    }
    /**
     * Adds a role to the cluster
     *
     * @param role the role to add
     */
    addIamRole(role) {
        const clusterRoleList = this.roles;
        if (clusterRoleList.includes(role)) {
            throw new Error(`Role '${role.roleArn}' is already attached to the cluster`);
        }
        clusterRoleList.push(role);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Cluster", version: "2.75.1-alpha.0" };
exports.Cluster = Cluster;
//# sourceMappingURL=data:application/json;base64,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