"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack.
     */
    static resourceServer(server, scope) {
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client.
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient.
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 *
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 *
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 *
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool.
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client.
 */
class UserPoolClient extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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